package org.bidib.wizard.migration.labels;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.transform.TransformerException;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.assertj.core.api.Assertions;
import org.bidib.jbidibc.core.schema.bidibbase.PortType;
import org.bidib.jbidibc.core.schema.bidiblabels.LabelFactory;
import org.bidib.jbidibc.core.schema.bidiblabels.NodeLabel;
import org.bidib.jbidibc.core.schema.bidiblabels.NodeLabels;
import org.bidib.jbidibc.core.schema.bidiblabels.PortLabel;
import org.bidib.jbidibc.core.schema.bidiblabels.PortLabels;
import org.bidib.wizard.api.context.ApplicationContext;
import org.bidib.wizard.migration.migrator.MigrationContext;
import org.bidib.wizard.migration.schema.nodes.Nodes;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.SAXException;
import org.xmlunit.builder.DiffBuilder;
import org.xmlunit.diff.Diff;

public class OldWizardLabelMigratorTest {

    private static final Logger LOGGER = LoggerFactory.getLogger(OldWizardLabelMigratorTest.class);

    private static final String DATA1_XML = "/xml-test/labels/wizard-old/LightPortLabels";

    private static final String DATA2_XML = "/xml-test/labels/wizard-old/NodeLabels";

    @Test
    @DisplayName("Test XSL transformation and compare to xml from file.")
    public void transformationLightPortLabelsTest()
        throws TransformerException, URISyntaxException, SAXException, IOException {

        InputStream inputXSL = WizardLabelMigratorTest.class.getResourceAsStream(OldWizardLabelMigrator.INPUT4_XSL);
        Assertions.assertThat(inputXSL).isNotNull();

        InputStream dataXML = WizardLabelMigratorTest.class.getResourceAsStream(DATA1_XML);
        Assertions.assertThat(dataXML).isNotNull();

        StringWriter outputXML = new StringWriter();

        try {
            OldWizardLabelMigrator migrator = new OldWizardLabelMigrator();
            Map<String, String> params = new HashMap<>();

            params.put("port_type", "light");
            params.put("search_uniqueId", Long.toString(1464607285711414L));

            migrator.doTransform(params, inputXSL, dataXML, outputXML);
        }
        finally {
            if (outputXML != null) {
                try {
                    outputXML.close();
                }
                catch (IOException ex) {
                    LOGGER.warn("Close outputXML writer failed.", ex);
                }
            }

            if (inputXSL != null) {
                try {
                    inputXSL.close();
                }
                catch (IOException ex) {
                    LOGGER.warn("Close inputXSL stream failed.", ex);
                }
            }
        }

        LOGGER.info("The generated XML document is:\r\n{}", outputXML);

        Assertions.assertThat(outputXML.getBuffer().length()).isGreaterThan(0);

        InputStream is =
            WizardLabelMigratorTest.class.getResourceAsStream("/xml-test/result/wizard-old/LightPortLabels.xml");
        final String xmlContent = IOUtils.toString(is, "UTF-8");

        Diff myDiffSimilar = DiffBuilder.compare(xmlContent).withTest(outputXML.toString()).checkForSimilar().build();
        Assertions
            .assertThat(myDiffSimilar.hasDifferences()).withFailMessage("XML similar " + myDiffSimilar.toString())
            .isFalse();
    }

    @Test
    @DisplayName("Test XSL transformation and compare to xml from file.")
    public void transformationNodeLabelsTest()
        throws TransformerException, URISyntaxException, SAXException, IOException {

        InputStream inputXSL = OldWizardLabelMigratorTest.class.getResourceAsStream(OldWizardLabelMigrator.INPUT5_XSL);
        Assertions.assertThat(inputXSL).isNotNull();

        InputStream dataXML = OldWizardLabelMigratorTest.class.getResourceAsStream(DATA2_XML);
        Assertions.assertThat(dataXML).isNotNull();

        StringWriter outputXML = new StringWriter();

        try {
            OldWizardLabelMigrator migrator = new OldWizardLabelMigrator();

            Map<String, String> params = new HashMap<>();
            params.put("port_type", "node");
            params.put("search_uniqueId", Long.toString(1464607285711414L));

            migrator.doTransform(params, inputXSL, dataXML, outputXML);
        }
        finally {
            if (outputXML != null) {
                try {
                    outputXML.close();
                }
                catch (IOException ex) {
                    LOGGER.warn("Close outputXML writer failed.", ex);
                }
            }

            if (inputXSL != null) {
                try {
                    inputXSL.close();
                }
                catch (IOException ex) {
                    LOGGER.warn("Close inputXSL stream failed.", ex);
                }
            }
        }

        LOGGER.info("The generated XML document is:\r\n{}", outputXML);

        Assertions.assertThat(outputXML.getBuffer().length()).isGreaterThan(0);

        InputStream is =
            OldWizardLabelMigratorTest.class.getResourceAsStream("/xml-test/result/wizard-old/NodeLabels.xml");
        final String xmlContent = IOUtils.toString(is, "UTF-8");

        Diff myDiffSimilar = DiffBuilder.compare(xmlContent).withTest(outputXML.toString()).checkForSimilar().build();
        Assertions
            .assertThat(myDiffSimilar.hasDifferences()).withFailMessage("XML similar " + myDiffSimilar.toString())
            .isFalse();
    }

    @Test
    public void findAllNodesInWizardLightPortLabelsTest() {
        URL url = OldWizardLabelMigratorTest.class.getResource(DATA1_XML);
        Assertions.assertThat(url).isNotNull();

        String searchPath = url.getPath();
        // get the path only
        searchPath = FilenameUtils.getFullPath(searchPath);

        LOGGER.info("Prepared search path: {}", searchPath);

        OldWizardLabelMigrator migrator = new OldWizardLabelMigrator();
        Nodes nodes =
            migrator
                .performFindAllNodesInWizardLabels("LightPortLabels", OldWizardLabelMigrator.INPUT5_XSL, searchPath);

        Assertions.assertThat(nodes).isNotNull();
        Assertions.assertThat(nodes.getNodeLabel()).isNotNull();

        List<org.bidib.wizard.migration.schema.nodes.NodeLabel> nodeLabels = nodes.getNodeLabel();

        LOGGER.info("Current nodeLabels: {}", nodeLabels);
        Assertions.assertThat(nodeLabels.size()).isEqualTo(2);
    }

    @Test
    public void findAllNodesInWizardLabelsTest() {
        URL url = OldWizardLabelMigratorTest.class.getResource(DATA1_XML);
        Assertions.assertThat(url).isNotNull();

        String searchPath = url.getPath();
        // get the path only
        searchPath = FilenameUtils.getFullPath(searchPath);

        LOGGER.info("Prepared search path: {}", searchPath);

        OldWizardLabelMigrator migrator = new OldWizardLabelMigrator();
        Nodes nodes = migrator.findAllNodesInWizardLabels(searchPath);

        Assertions.assertThat(nodes).isNotNull();
        Assertions.assertThat(nodes.getNodeLabel()).isNotNull();

        List<org.bidib.wizard.migration.schema.nodes.NodeLabel> nodeLabels = nodes.getNodeLabel();

        LOGGER.info("Current nodeLabels: {}", nodeLabels);
        Assertions.assertThat(nodeLabels.size()).isEqualTo(3);
    }

    @Test
    public void migrationOldAccessoryLabelsToNodeLabelsTest() {
        URL url = OldWizardLabelMigratorTest.class.getResource(DATA2_XML);
        Assertions.assertThat(url).isNotNull();

        String searchPath = url.getPath();
        // get the path only
        searchPath = FilenameUtils.getFullPath(searchPath);

        LOGGER.info("Prepared search path: {}", searchPath);

        final ApplicationContext context = new MigrationContext();

        OldWizardLabelMigrator migrator = new OldWizardLabelMigrator();
        NodeLabels nodeLabels = migrator.performWizardLabelsMigration(context, 0x5340D75001236L, searchPath);

        Assertions.assertThat(nodeLabels).isNotNull();
        Assertions.assertThat(nodeLabels.getNodeLabel()).isNotNull();

        NodeLabel nodeLabel = nodeLabels.getNodeLabel();
        Assertions.assertThat(nodeLabel.getUniqueId()).isEqualTo(0x5340D75001236L);
        Assertions.assertThat(nodeLabel.getUserName()).isEqualTo("Node_0_Test");

        Assertions.assertThat(nodeLabels.getPortLabels()).isNotNull();
        PortLabels portLabels = nodeLabels.getPortLabels();
        Assertions.assertThat(portLabels.getPortLabel()).isNotNull();
        List<PortLabel> portLabelList = portLabels.getPortLabel();

        // check total number of ports
        Assertions.assertThat(portLabelList.size()).isEqualTo(3);

        // check the ports
        Assertions.assertThat(LabelFactory.getPortsOfType(portLabelList, PortType.SERVO).size()).isEqualTo(0);
        Assertions.assertThat(LabelFactory.getPortsOfType(portLabelList, PortType.SWITCH).size()).isEqualTo(0);
        Assertions.assertThat(LabelFactory.getPortsOfType(portLabelList, PortType.INPUT).size()).isEqualTo(0);

        Assertions.assertThat(LabelFactory.getPortsOfType(portLabelList, PortType.LIGHT).size()).isEqualTo(3);

    }
}
