package org.bidib.wizard.migration.labels;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.XMLConstants;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.assertj.core.api.Assertions;
import org.bidib.jbidibc.core.schema.bidibbase.PortType;
import org.bidib.jbidibc.core.schema.bidiblabels.AccessoryLabels;
import org.bidib.jbidibc.core.schema.bidiblabels.FeedbackPortLabels;
import org.bidib.jbidibc.core.schema.bidiblabels.LabelFactory;
import org.bidib.jbidibc.core.schema.bidiblabels.MacroLabels;
import org.bidib.jbidibc.core.schema.bidiblabels.NodeLabel;
import org.bidib.jbidibc.core.schema.bidiblabels.NodeLabels;
import org.bidib.jbidibc.core.schema.bidiblabels.PortLabel;
import org.bidib.jbidibc.core.schema.bidiblabels.PortLabels;
import org.bidib.wizard.api.context.ApplicationContext;
import org.bidib.wizard.migration.migrator.MigrationContext;
import org.bidib.wizard.migration.schema.nodes.Nodes;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.SAXException;
import org.xmlunit.builder.DiffBuilder;
import org.xmlunit.diff.Diff;

public class WizardLabelMigratorTest {

    private static final Logger LOGGER = LoggerFactory.getLogger(WizardLabelMigratorTest.class);

    private static final String NAMESPACE_PREFIX = "{http://www.bidib.org/schema/labels}";

    private static final String INPUT_XSL = "/migration/labels-migration.xsl";

    private static final String INPUT2_XSL = "/migration/labels-migration2.xsl";

    private static final String DATA1_XML = "/xml-test/labels/bidib2/05000DD00043ED.xml";

    private static final String DATA2_XML = "/xml-test/labels/bidib2/05000D6B009AEA.xml";

    private static final String DATA3_XML = "/xml-test/labels/wizard/NodeLabels.labels";

    private static final String DATA4_XML = "/xml-test/labels/wizard/AccessoryLabels.labels";

    private static final String DATA5_XML = "/xml-test/labels/wizard/MacroLabels.labels";

    private static final String DATA6_XML = "/xml-test/labels/wizard-wrong-type/SwitchPortLabels.labels";

    @Test
    public void migrationWithUniqueIdAsIntegerTest() {

        URL url = WizardLabelMigratorTest.class.getResource(DATA1_XML);
        Assertions.assertThat(url).isNotNull();

        String searchPath = url.getPath();

        // get the path only
        searchPath = FilenameUtils.getFullPath(searchPath);

        LOGGER.info("Prepared search path: {}", searchPath);

        final ApplicationContext context = new MigrationContext();

        Bidib2LabelMigrator migrator = new Bidib2LabelMigrator();
        NodeLabels nodeLabels = migrator.performWizardLabelsMigration(context, 0x05000DD00043EDL, searchPath);

        Assertions.assertThat(nodeLabels).isNotNull();
        Assertions.assertThat(nodeLabels.getNodeLabel()).isNotNull();

        NodeLabel nodeLabel = nodeLabels.getNodeLabel();
        Assertions.assertThat(nodeLabel.getUniqueId()).isEqualTo(0x05000DD00043EDL);
        Assertions.assertThat(nodeLabel.getUserName()).isEqualTo("NeoTest");
    }

    @Test
    @DisplayName("Test XSL transformation and compare to xml from file.")
    public void transformationWithUniqueIdAsIntegerTest()
        throws TransformerException, URISyntaxException, SAXException, IOException {

        InputStream inputXSL = WizardLabelMigratorTest.class.getResourceAsStream(INPUT_XSL);
        Assertions.assertThat(inputXSL).isNotNull();

        InputStream dataXML = WizardLabelMigratorTest.class.getResourceAsStream(DATA1_XML);
        Assertions.assertThat(dataXML).isNotNull();

        StringWriter outputXML = new StringWriter();

        try {
            WizardLabelMigrator migrator = new WizardLabelMigrator();

            Map<String, String> params = new HashMap<>();
            params.put("search_uniqueId", Long.toString(0x05000DD00043EDL));

            migrator.doTransform(params, inputXSL, dataXML, outputXML);
        }
        finally {
            if (outputXML != null) {
                try {
                    outputXML.close();
                }
                catch (IOException ex) {
                    LOGGER.warn("Close outputXML writer failed.", ex);
                }
            }

            if (inputXSL != null) {
                try {
                    inputXSL.close();
                }
                catch (IOException ex) {
                    LOGGER.warn("Close inputXSL stream failed.", ex);
                }
            }
        }

        LOGGER.info("The generated XML document is:\r\n{}", outputXML);

        Assertions.assertThat(outputXML.getBuffer().length()).isGreaterThan(0);

        // Document testDoc = XMLUnit.buildControlDocument(outputXML.toString());

        InputStream is = WizardLabelMigratorTest.class.getResourceAsStream("/xml-test/result/05000DD00043ED.xml");
        final String xmlContent = IOUtils.toString(is, "UTF-8");
        // Document controlDoc = XMLUnit.buildControlDocument(xmlContent);

        // XMLAssert.assertXMLEqual(controlDoc, testDoc);
        Diff myDiffSimilar = DiffBuilder.compare(xmlContent).withTest(outputXML.toString()).checkForSimilar().build();
        Assertions
            .assertThat(myDiffSimilar.hasDifferences()).withFailMessage("XML similar " + myDiffSimilar.toString())
            .isFalse();
    }

    @Test
    public void migrationWithUniqueIdAsHexTest() {

        URL url = WizardLabelMigratorTest.class.getResource(DATA2_XML);
        Assertions.assertThat(url).isNotNull();

        String searchPath = url.getPath();

        // get the path only
        searchPath = FilenameUtils.getFullPath(searchPath);

        LOGGER.info("Prepared search path: {}", searchPath);

        final ApplicationContext context = new MigrationContext();

        Bidib2LabelMigrator migrator = new Bidib2LabelMigrator();
        NodeLabels nodeLabels = migrator.performWizardLabelsMigration(context, 0x05000D6B009AEAL, searchPath);

        Assertions.assertThat(nodeLabels).isNotNull();
        Assertions.assertThat(nodeLabels.getNodeLabel()).isNotNull();

        NodeLabel nodeLabel = nodeLabels.getNodeLabel();
        Assertions.assertThat(nodeLabel.getUniqueId()).isEqualTo(0x05000D6B009AEAL);
        Assertions.assertThat(nodeLabel.getUserName()).isEqualTo("");
    }

    @Test
    @DisplayName("Test XSL transformation and compare to xml from file.")
    public void transformationWithUniqueIdAsHexTest()
        throws TransformerException, URISyntaxException, SAXException, IOException {

        InputStream inputXSL = WizardLabelMigratorTest.class.getResourceAsStream(INPUT_XSL);
        Assertions.assertThat(inputXSL).isNotNull();

        InputStream dataXML = WizardLabelMigratorTest.class.getResourceAsStream(DATA2_XML);
        Assertions.assertThat(dataXML).isNotNull();

        StringWriter outputXML = new StringWriter();

        try {
            WizardLabelMigrator migrator = new WizardLabelMigrator();

            Map<String, String> params = new HashMap<>();
            params.put("search_uniqueId", Long.toString(0x05000D6B009AEAL));

            migrator.doTransform(params, inputXSL, dataXML, outputXML);
        }
        finally {
            if (outputXML != null) {
                try {
                    outputXML.close();
                }
                catch (IOException ex) {
                    LOGGER.warn("Close outputXML writer failed.", ex);
                }
            }

            if (inputXSL != null) {
                try {
                    inputXSL.close();
                }
                catch (IOException ex) {
                    LOGGER.warn("Close inputXSL stream failed.", ex);
                }
            }
        }

        LOGGER.info("The generated XML document is:\r\n{}", outputXML);
        Assertions.assertThat(outputXML.getBuffer().length()).isGreaterThan(0);

        InputStream is = WizardLabelMigratorTest.class.getResourceAsStream("/xml-test/result/05000D6B009AEA.xml");
        final String xmlContent = IOUtils.toString(is, "UTF-8");

        Diff myDiffSimilar = DiffBuilder.compare(xmlContent).withTest(outputXML.toString()).checkForSimilar().build();
        Assertions
            .assertThat(myDiffSimilar.hasDifferences()).withFailMessage("XML similar " + myDiffSimilar.toString())
            .isFalse();
    }

    @Test
    @DisplayName("Test XSL transformation and compare to xml from file.")
    public void transformationOldNodeLabelsTest()
        throws TransformerException, URISyntaxException, SAXException, IOException {

        InputStream inputXSL = WizardLabelMigratorTest.class.getResourceAsStream(INPUT2_XSL);
        Assertions.assertThat(inputXSL).isNotNull();

        InputStream dataXML = WizardLabelMigratorTest.class.getResourceAsStream(DATA3_XML);
        Assertions.assertThat(dataXML).isNotNull();

        TransformerFactory factory = TransformerFactory.newInstance();
        // prevent XXE
        try {
            factory.setAttribute(XMLConstants.ACCESS_EXTERNAL_DTD, "");
        }
        catch (IllegalArgumentException e) {
            LOGGER.warn("XSL transformer implementation doesn't support {} feature", XMLConstants.ACCESS_EXTERNAL_DTD);
        }
        try {
            factory.setAttribute(XMLConstants.ACCESS_EXTERNAL_STYLESHEET, "");
        }
        catch (IllegalArgumentException e) {
            LOGGER.warn("XSL transformer implementation doesn't support {} feature", XMLConstants.ACCESS_EXTERNAL_DTD);
        }

        StreamSource xslStream = new StreamSource(inputXSL);

        Transformer transformer = factory.newTransformer(xslStream);
        transformer.setOutputProperty(OutputKeys.INDENT, "yes");
        transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "3");

        transformer.setParameter(NAMESPACE_PREFIX + "search_uniqueId", "61361602502354154");

        StreamSource in = new StreamSource(dataXML);

        StringWriter outputXML = new StringWriter();
        StreamResult out = new StreamResult(outputXML);

        transformer.transform(in, out);

        LOGGER.info("The generated XML document is:\r\n{}", outputXML);
        Assertions.assertThat(outputXML.getBuffer().length()).isGreaterThan(0);

        InputStream is = WizardLabelMigratorTest.class.getResourceAsStream("/xml-test/result/DA000D680064EA-node.xml");
        final String xmlContent = IOUtils.toString(is, "UTF-8");

        Diff myDiffSimilar = DiffBuilder.compare(xmlContent).withTest(outputXML.toString()).checkForSimilar().build();
        Assertions
            .assertThat(myDiffSimilar.hasDifferences()).withFailMessage("XML similar " + myDiffSimilar.toString())
            .isFalse();
    }

    @Test
    public void migrationOldNodeLabelsTest() {
        URL url = WizardLabelMigratorTest.class.getResource(DATA3_XML);
        Assertions.assertThat(url).isNotNull();

        String searchPath = url.getPath();
        // get the path only
        searchPath = FilenameUtils.getFullPath(searchPath);

        LOGGER.info("Prepared search path: {}", searchPath);

        Map<String, String> params = new HashMap<>();
        params.put("search_uniqueId", Long.toString(0xDA000D680064EAL));

        WizardLabelMigrator migrator = new WizardLabelMigrator();
        NodeLabels nodeLabels =
            migrator
                .performWizardLabelsMigration(params, "NodeLabels.labels", AbstractWizardLabelMigrator.INPUT2_XSL,
                    searchPath);

        Assertions.assertThat(nodeLabels).isNotNull();
        Assertions.assertThat(nodeLabels.getNodeLabel()).isNotNull();

        NodeLabel nodeLabel = nodeLabels.getNodeLabel();
        Assertions.assertThat(nodeLabel.getUniqueId()).isEqualTo(0xDA000D680064EAL);
        Assertions.assertThat(nodeLabel.getUserName()).isEqualTo("Test Booster Main");
    }

    @Test
    @DisplayName("Test XSL transformation and compare to xml from file.")
    public void transformationOldAccessoryLabelsTest()
        throws TransformerException, URISyntaxException, SAXException, IOException {

        InputStream inputXSL = WizardLabelMigratorTest.class.getResourceAsStream(INPUT2_XSL);
        Assertions.assertThat(inputXSL).isNotNull();

        InputStream dataXML = WizardLabelMigratorTest.class.getResourceAsStream(DATA4_XML);
        Assertions.assertThat(dataXML).isNotNull();

        TransformerFactory factory = TransformerFactory.newInstance();
        // prevent XXE
        try {
            factory.setAttribute(XMLConstants.ACCESS_EXTERNAL_DTD, "");
        }
        catch (IllegalArgumentException e) {
            LOGGER.warn("XSL transformer implementation doesn't support {} feature", XMLConstants.ACCESS_EXTERNAL_DTD);
        }
        try {
            factory.setAttribute(XMLConstants.ACCESS_EXTERNAL_STYLESHEET, "");
        }
        catch (IllegalArgumentException e) {
            LOGGER.warn("XSL transformer implementation doesn't support {} feature", XMLConstants.ACCESS_EXTERNAL_DTD);
        }

        StreamSource xslStream = new StreamSource(inputXSL);

        Transformer transformer = factory.newTransformer(xslStream);
        transformer.setOutputProperty(OutputKeys.INDENT, "yes");
        transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "3");

        transformer.setParameter(NAMESPACE_PREFIX + "search_uniqueId", "1464607285711414");

        StreamSource in = new StreamSource(dataXML);

        StringWriter outputXML = new StringWriter();
        StreamResult out = new StreamResult(outputXML);

        transformer.transform(in, out);

        LOGGER.info("The generated XML document is:\r\n{}", outputXML);
        Assertions.assertThat(outputXML.getBuffer().length()).isGreaterThan(0);

        InputStream is =
            WizardLabelMigratorTest.class.getResourceAsStream("/xml-test/result/5340D75001236-accessories.xml");
        final String xmlContent = IOUtils.toString(is, "UTF-8");

        Diff myDiffSimilar = DiffBuilder.compare(xmlContent).withTest(outputXML.toString()).checkForSimilar().build();
        Assertions
            .assertThat(myDiffSimilar.hasDifferences()).withFailMessage("XML similar " + myDiffSimilar.toString())
            .isFalse();
    }

    @Test
    public void migrationOldAccessoryLabelsTest() {
        URL url = WizardLabelMigratorTest.class.getResource(DATA4_XML);
        Assertions.assertThat(url).isNotNull();

        String searchPath = url.getPath();
        // get the path only
        searchPath = FilenameUtils.getFullPath(searchPath);

        LOGGER.info("Prepared search path: {}", searchPath);

        Map<String, String> params = new HashMap<>();
        params.put("search_uniqueId", Long.toString(0x5340D75001236L));

        WizardLabelMigrator migrator = new WizardLabelMigrator();
        NodeLabels nodeLabels =
            migrator
                .performWizardLabelsMigration(params, "AccessoryLabels.labels", AbstractWizardLabelMigrator.INPUT2_XSL,
                    searchPath);

        Assertions.assertThat(nodeLabels).isNotNull();
        Assertions.assertThat(nodeLabels.getNodeLabel()).isNotNull();

        NodeLabel nodeLabel = nodeLabels.getNodeLabel();
        Assertions.assertThat(nodeLabel.getUniqueId()).isEqualTo(0x5340D75001236L);
        Assertions.assertThat(nodeLabel.getUserName()).isNull();
    }

    @Test
    public void migrationOldAccessoryLabelsToNodeLabelsTest() {
        URL url = WizardLabelMigratorTest.class.getResource(DATA4_XML);
        Assertions.assertThat(url).isNotNull();

        String searchPath = url.getPath();
        // get the path only
        searchPath = FilenameUtils.getFullPath(searchPath);

        LOGGER.info("Prepared search path: {}", searchPath);

        final ApplicationContext context = new MigrationContext();

        WizardLabelMigrator migrator = new WizardLabelMigrator();
        NodeLabels nodeLabels = migrator.performWizardLabelsMigration(context, 0x5340D75001236L, searchPath);

        Assertions.assertThat(nodeLabels).isNotNull();
        Assertions.assertThat(nodeLabels.getNodeLabel()).isNotNull();

        NodeLabel nodeLabel = nodeLabels.getNodeLabel();
        Assertions.assertThat(nodeLabel.getUniqueId()).isEqualTo(0x5340D75001236L);
        Assertions.assertThat(nodeLabel.getUserName()).isEqualTo("OneControl Test 1");

        Assertions.assertThat(nodeLabels.getMacroLabels()).isNotNull();
        MacroLabels macroLabels = nodeLabels.getMacroLabels();
        Assertions.assertThat(macroLabels.getMacroLabel()).isNotNull();
        Assertions.assertThat(macroLabels.getMacroLabel().size()).isEqualTo(32);

        Assertions.assertThat(nodeLabels.getAccessoryLabels()).isNotNull();
        AccessoryLabels acecssoryLabels = nodeLabels.getAccessoryLabels();
        Assertions.assertThat(acecssoryLabels.getAccessoryLabel()).isNotNull();
        Assertions.assertThat(acecssoryLabels.getAccessoryLabel().size()).isEqualTo(20);

        Assertions.assertThat(nodeLabels.getFeedbackPortLabels()).isNotNull();
        FeedbackPortLabels feedbackPortLabels = nodeLabels.getFeedbackPortLabels();
        Assertions.assertThat(feedbackPortLabels.getPortLabel()).isNotNull();
        Assertions.assertThat(feedbackPortLabels.getPortLabel().size()).isEqualTo(1);

        Assertions.assertThat(nodeLabels.getPortLabels()).isNotNull();
        PortLabels portLabels = nodeLabels.getPortLabels();
        Assertions.assertThat(portLabels.getPortLabel()).isNotNull();
        List<PortLabel> portLabelList = portLabels.getPortLabel();

        // check total number of ports
        Assertions.assertThat(portLabelList.size()).isEqualTo(40);

        // check the ports
        Assertions.assertThat(LabelFactory.getPortsOfType(portLabelList, PortType.SERVO).size()).isEqualTo(8);
        Assertions.assertThat(LabelFactory.getPortsOfType(portLabelList, PortType.SWITCH).size()).isEqualTo(16);
        Assertions.assertThat(LabelFactory.getPortsOfType(portLabelList, PortType.INPUT).size()).isEqualTo(16);

        Assertions.assertThat(LabelFactory.getPortsOfType(portLabelList, PortType.LIGHT).size()).isEqualTo(0);

    }

    @Test
    @DisplayName("Test XSL transformation and compare to xml from file.")
    public void transformationOldMacroLabelsTest()
        throws TransformerException, URISyntaxException, SAXException, IOException {

        InputStream inputXSL = WizardLabelMigratorTest.class.getResourceAsStream(INPUT2_XSL);
        Assertions.assertThat(inputXSL).isNotNull();

        InputStream dataXML = WizardLabelMigratorTest.class.getResourceAsStream(DATA5_XML);
        Assertions.assertThat(dataXML).isNotNull();

        TransformerFactory factory = TransformerFactory.newInstance();
        // prevent XXE
        try {
            factory.setAttribute(XMLConstants.ACCESS_EXTERNAL_DTD, "");
        }
        catch (IllegalArgumentException e) {
            LOGGER.warn("XSL transformer implementation doesn't support {} feature", XMLConstants.ACCESS_EXTERNAL_DTD);
        }
        try {
            factory.setAttribute(XMLConstants.ACCESS_EXTERNAL_STYLESHEET, "");
        }
        catch (IllegalArgumentException e) {
            LOGGER.warn("XSL transformer implementation doesn't support {} feature", XMLConstants.ACCESS_EXTERNAL_DTD);
        }

        StreamSource xslStream = new StreamSource(inputXSL);

        Transformer transformer = factory.newTransformer(xslStream);
        transformer.setOutputProperty(OutputKeys.INDENT, "yes");
        transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "3");

        transformer.setParameter(NAMESPACE_PREFIX + "search_uniqueId", "1464607285711414");

        StreamSource in = new StreamSource(dataXML);

        StringWriter outputXML = new StringWriter();
        StreamResult out = new StreamResult(outputXML);

        transformer.transform(in, out);

        LOGGER.info("The generated XML document is:\r\n{}", outputXML);
        Assertions.assertThat(outputXML.getBuffer().length()).isGreaterThan(0);

        InputStream is = WizardLabelMigratorTest.class.getResourceAsStream("/xml-test/result/5340D75001236-macros.xml");
        final String xmlContent = IOUtils.toString(is, "UTF-8");

        Diff myDiffSimilar = DiffBuilder.compare(xmlContent).withTest(outputXML.toString()).checkForSimilar().build();
        Assertions
            .assertThat(myDiffSimilar.hasDifferences()).withFailMessage("XML similar " + myDiffSimilar.toString())
            .isFalse();
    }

    @Test
    public void migrationOldMacroLabelsTest() {
        URL url = WizardLabelMigratorTest.class.getResource(DATA5_XML);
        Assertions.assertThat(url).isNotNull();

        String searchPath = url.getPath();
        // get the path only
        searchPath = FilenameUtils.getFullPath(searchPath);

        LOGGER.info("Prepared search path: {}", searchPath);

        Map<String, String> params = new HashMap<>();
        params.put("search_uniqueId", Long.toString(0x5340D75001236L));

        WizardLabelMigrator migrator = new WizardLabelMigrator();
        NodeLabels nodeLabels =
            migrator
                .performWizardLabelsMigration(params, "MacroLabels.labels", AbstractWizardLabelMigrator.INPUT2_XSL,
                    searchPath);

        Assertions.assertThat(nodeLabels).isNotNull();
        Assertions.assertThat(nodeLabels.getNodeLabel()).isNotNull();

        NodeLabel nodeLabel = nodeLabels.getNodeLabel();
        Assertions.assertThat(nodeLabel.getUniqueId()).isEqualTo(0x5340D75001236L);
        Assertions.assertThat(nodeLabel.getUserName()).isNull();
        ;
    }

    @Test
    public void findAllNodesInWizardLabelsTest() {
        URL url = WizardLabelMigratorTest.class.getResource(DATA3_XML);
        Assertions.assertThat(url).isNotNull();

        String searchPath = url.getPath();
        // get the path only
        searchPath = FilenameUtils.getFullPath(searchPath);

        LOGGER.info("Prepared search path: {}", searchPath);

        WizardLabelMigrator migrator = new WizardLabelMigrator();
        Nodes nodes = migrator.findAllNodesInWizardLabels(searchPath);

        Assertions.assertThat(nodes).isNotNull();
        Assertions.assertThat(nodes.getNodeLabel()).isNotNull();

        List<org.bidib.wizard.migration.schema.nodes.NodeLabel> nodeLabels = nodes.getNodeLabel();

        LOGGER.info("Current nodeLabels: {}", nodeLabels);
        Assertions.assertThat(nodeLabels.size()).isEqualTo(6);
    }

    @Test
    public void migrateSwitchPortForcedLabelsTest() {
        URL url = WizardLabelMigratorTest.class.getResource(DATA6_XML);
        Assertions.assertThat(url).isNotNull();

        String searchPath = url.getPath();
        // get the path only
        searchPath = FilenameUtils.getFullPath(searchPath);

        LOGGER.info("Prepared search path: {}", searchPath);

        Map<String, String> params = new HashMap<>();
        params.put("search_uniqueId", Long.toString(0x5000D7500BAEAL));
        params.put("forced_porttype", "switch");

        WizardLabelMigrator migrator = new WizardLabelMigrator();
        NodeLabels nodeLabels =
            migrator
                .performWizardLabelsMigration(params, "SwitchPortLabels.labels", AbstractWizardLabelMigrator.INPUT2_XSL,
                    searchPath);

        Assertions.assertThat(nodeLabels).isNotNull();
        Assertions.assertThat(nodeLabels.getNodeLabel()).isNotNull();

        NodeLabel nodeLabel = nodeLabels.getNodeLabel();
        Assertions.assertThat(nodeLabel.getUniqueId()).isEqualTo(0x5000D7500BAEAL);
        Assertions.assertThat(nodeLabel.getUserName()).isNull();
        ;

        Assertions.assertThat(nodeLabels.getPortLabels()).isNotNull();

        PortLabels portLabels = nodeLabels.getPortLabels();

        Assertions.assertThat(portLabels.getPortLabel().size()).isEqualTo(36);

        for (PortLabel portLabel : portLabels.getPortLabel()) {
            Assertions.assertThat(portLabel.getType()).isEqualTo(PortType.SWITCH);
        }
    }
}
