package org.bidib.wizard.spy;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Optional;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.bidib.wizard.spy.config.SpyConfiguration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.WebApplicationType;
import org.springframework.boot.autoconfigure.SpringBootApplication;
import org.springframework.boot.builder.SpringApplicationBuilder;

@SpringBootApplication(scanBasePackageClasses = SpyConfiguration.class)
public class BiDiBWizardSpyApplication {

    private static final Logger LOGGER = LoggerFactory.getLogger(BiDiBWizardSpyApplication.class);

    public static void main(String[] args) {

        Locale locale = Locale.getDefault();
        LOGGER.info("Current locale: {}", locale);

        List<String> argsList = new ArrayList<>();
        argsList.addAll(Arrays.asList(args));

        final SpringApplicationBuilder builder =
                new SpringApplicationBuilder()
                        .sources(BiDiBWizardSpyApplication.class).headless(false).web(WebApplicationType.NONE)
                        .properties("wizard-app-type=spy");

        if (!"restartedMain".equals(Thread.currentThread().getName())) {
            // add these extensions if the thread is not the devutils restarted main thread
            // ConfigFileApplicationListener.CONFIG_NAME_PROPERTY

            Optional<String> wizardConfigFileName =
                    argsList.stream().filter(val -> val.startsWith("--wizard.settings.file-name")).findFirst();
            if (wizardConfigFileName.isPresent()) {
                LOGGER.info("Keep the provided data from args: {}", wizardConfigFileName.get());
                String fileName = wizardConfigFileName.get().split("=")[1];

                fileName = FilenameUtils.getBaseName(fileName);

                LOGGER.info("Add the fileName: {}", fileName);

                builder.properties("spring.config.name=application," + fileName);
            }
            else {
                builder.properties("spring.config.name=application,wizard-spy");
            }

            Optional<String> wizardConfigFileLocation =
                    argsList.stream().filter(val -> val.startsWith("--wizard.settings.file-location")).findFirst();
            if (wizardConfigFileLocation.isPresent()) {
                LOGGER.info("Keep the provided data from args: {}", wizardConfigFileLocation.get());

                String location = wizardConfigFileLocation.get().split("=")[1];
                location = location.replaceAll("\\\\", "/");
                if (!location.endsWith("/")) {
                    location += "/";
                }

                LOGGER.info("Set the optional additional-location: {}", location);

                builder.properties("spring.config.additional-location=optional:file:///" + location);
            }
            else {
                builder.properties("spring.config.additional-location=optional:file:///${user.home}/.bidib/");
            }
        }

        // get the preferences path from a jvm property (set with -Dbidib.preferencesPath=....
        String preferencesPath = System.getProperty("bidib.preferencesPath");

        // if the path is not set, use the value from the environment
        if (StringUtils.isBlank(preferencesPath)) {
            preferencesPath = System.getenv("bidib.preferencesPath");
            // if the path is not set use the user.home
            if (StringUtils.isBlank(preferencesPath)) {
                preferencesPath = System.getProperty("user.home");

                System.setProperty("bidib.preferencesPath", preferencesPath);
            }
        }

        try {
            builder.run(argsList.toArray(new String[0]));
        }
        catch (Exception ex) {
            LOGGER.warn("Start application failed.", ex);

            System.exit(1);
        }
    }
}
