package org.bidib.wizard.spy;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;

import org.apache.commons.lang3.StringUtils;
import org.bidib.jbidibc.com0com.BidibPortProvider;
import org.bidib.jbidibc.debug.DebugReaderFactory.SerialImpl;
import org.bidib.jbidibc.messages.base.BidibPortStatusListener;
import org.bidib.jbidibc.messages.base.BidibPortStatusListener.PortStatus;
import org.bidib.wizard.spy.preferences.controller.PreferencesController;
import org.bidib.wizard.spy.tray.Tray;
import org.bidib.wizard.spy.tray.TrayActionExecutor;
import org.bidib.wizard.spy.tray.TrayNotificationController;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BidibSpy implements TrayActionExecutor {

    private static final Logger LOGGER = LoggerFactory.getLogger(BidibSpy.class);

    private Tray tray;

    private String interfacePort;

    private String proxyPort;

    private String serialPortProvider;

    private BidibPortProvider provider;

    private BidibSpyConfig config = new BidibSpyConfig(null);

    private TrayNotificationController notificationController;

    private AtomicBoolean connectEnabled = new AtomicBoolean();

    public void startApp(String[] args) {

        config.loadConfig();

        try {
            UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
        }
        catch (Exception ex) {
            LOGGER.error("Change to use system look and feel failed.", ex);
        }

        // create the system tray icon
        tray = new Tray(this);
        tray.init();

        notificationController = new TrayNotificationController();
        notificationController.start(tray);

//        Runtime.getRuntime().addShutdownHook(new Thread() {
//            @Override
//            public void run() {
//                LOGGER.info("Shutdown hook is executed.");
//
//                exit();
//            }
//        });

        notificationController
            .addInfoNotification("BiDiB-Wizard-Spy", "The spy is started. Use the tray menu to connect.");
    }

    @Override
    public void exit() {

        disconnect();

        if (tray != null) {
            tray.cleanup();

            tray = null;
        }

        if (SwingUtilities.isEventDispatchThread()) {
            System.exit(0);
        }
    }

    @Override
    public void showPreferences() {
        PreferencesController controller = new PreferencesController(JOptionPane.getFrameForComponent(null));
        controller.start(config);
    }

    protected final ScheduledExecutorService reconnectProxyWorker = Executors.newScheduledThreadPool(1);

    private PortStatus latestPortStatus;

    @Override
    public void connect() {

        try {
            LOGGER.info("Create the provider the interface and proxy port.");

            provider = new BidibPortProvider();
            provider.addStatusListener(new BidibPortStatusListener() {

                @Override
                public void statusChanged(final String connectionId, final PortStatus portStatus) {
                    LOGGER
                        .info("The status of the proxy connection has been changed: {}, connectionId: {}", portStatus,
                            connectionId);
                    tray.setPortStatus(portStatus);

                    if (PortStatus.DISCONNECTED == portStatus && latestPortStatus != portStatus) {

                        latestPortStatus = portStatus;

                        if (connectEnabled.get()) {
                            LOGGER.info("Automatic reconnect of proxy port is enabled. Reconnect the proxy port.");

                            reconnectProxyWorker.schedule(() -> {
                                SwingUtilities.invokeLater(() -> provider.reconnectProxyPort());
                            }, 5, TimeUnit.SECONDS);
                        }
                        else {
                            LOGGER.info("Automatic reconnect of proxy port is not enabled.");
                        }
                    }
                    else if (latestPortStatus != portStatus) {
                        LOGGER.info("Set the latest port status: {}", portStatus);
                        latestPortStatus = portStatus;
                    }
                }
            });

            SerialImpl serialImpl = SerialImpl.SCM;
            if (StringUtils.isNotBlank(serialPortProvider)) {
                switch (serialPortProvider) {
                    case "RXTX":
                        serialImpl = SerialImpl.RXTX;

                        String osName = System.getProperty("os.name");
                        String ports = null;
                        if (osName.toLowerCase().indexOf("windows") == -1) {
                            ports = proxyPort + ":" + interfacePort;
                        }
                        else {
                            ports = proxyPort + ";" + interfacePort;
                        }
                        LOGGER.info("Set the system property: gnu.io.rxtx.SerialPorts, value: {}", ports);

                        System.setProperty("gnu.io.rxtx.SerialPorts", ports);
                        break;
                    case "PUREJAVACOMM":
                        serialImpl = SerialImpl.PUREJAVACOMM;
                        break;
                    default:
                        break;
                }
            }
            LOGGER.info("Use the serial port provider impl: {}", serialImpl);

            provider.start(interfacePort, proxyPort, serialImpl);

            connectEnabled.set(true);

            notificationController
                .addInfoNotification("BiDiB-Wizard-Spy", "Connect to interface and proxy port passed.");

            LOGGER.info("Start interface and proxy port passed.");
        }
        catch (Exception ex) {
            LOGGER.warn("Connect to interface and proxy port failed.", ex);

            notificationController
                .addErrorNotification("BiDiB-Wizard-Spy", "Connect to interface and proxy port failed.");

            stopProvider();
        }
    }

    @Override
    public void disconnect() {
        LOGGER.info("Disconnect is called from tray.");

        connectEnabled.set(false);

        stopProvider();

        notificationController
            .addInfoNotification("BiDiB-Wizard-Spy", "Disconnect from interface and proxy port passed.");
    }

    private void stopProvider() {
        try {
            if (provider != null) {
                LOGGER.info("Stop the provider.");
                provider.stop();

                provider = null;
            }
        }
        catch (Exception ex) {
            LOGGER.warn("Stop provider failed.", ex);
        }

    }

    @Override
    public boolean isConnectEnabled() {
        return (provider == null && isSpyConfigValid());
    }

    @Override
    public boolean isDisconnectEnabled() {
        return (provider != null);
    }

    private boolean isSpyConfigValid() {
        interfacePort = config.getConfig().getString("spy.interfacePort");
        proxyPort = config.getConfig().getString("spy.proxyPort");
        serialPortProvider = config.getConfig().getString("spy.serialPortProvider");

        LOGGER
            .info("Loaded properties, interfacePort: {}, proxyPort: {}, serialPortProvider: {}", interfacePort,
                proxyPort, serialPortProvider);

        if (StringUtils.isBlank(interfacePort) || StringUtils.isBlank(proxyPort)) {
            LOGGER
                .error(
                    "The configuration properties has missing valus for 'interfacePort' and/or 'proxyPort'. Add these values to properties file: {}",
                    BidibSpyConfig.DEFAULT_CONFIG_FILENAME);

            return false;
        }

        return true;
    }
}
