package org.bidib.wizard.tracer.client.preferences.view.panel;

import java.awt.BorderLayout;
import java.awt.Component;
import java.util.function.Consumer;

import javax.swing.DefaultListCellRenderer;
import javax.swing.JComboBox;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.ListModel;

import org.apache.commons.lang3.StringUtils;
import org.bidib.api.json.types.SerialPortInfo;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.common.CommPort;
import org.bidib.wizard.client.common.preferences.view.panel.AbstractSettingsPanel;
import org.bidib.wizard.client.common.preferences.view.panel.SettingsPanelInterface;
import org.bidib.wizard.common.model.settings.TracerServiceSettingsInterface;
import org.bidib.wizard.core.service.SystemInfoService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.adapter.ComboBoxAdapter;
import com.jgoodies.binding.list.SelectionInList;
import com.jgoodies.binding.value.BufferedValueModel;
import com.jgoodies.binding.value.ConverterValueModel;
import com.jgoodies.binding.value.ValueModel;
import com.jgoodies.common.collect.ArrayListModel;
import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.debug.FormDebugPanel;
import com.jgoodies.forms.factories.Paddings;

public class TracerServiceSettingsPanel extends AbstractSettingsPanel<TracerServiceSettingsInterface>
    implements SettingsPanelInterface {

    private static final Logger LOGGER = LoggerFactory.getLogger(TracerServiceSettingsPanel.class);

    private static final String ENCODED_DIALOG_COLUMN_SPECS = "pref, 3dlu, pref, 3dlu, fill:50dlu:grow, 3dlu, pref";

    private static final String ENCODED_DIALOG_ROW_SPECS =
        "pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref";

    private final TracerServiceSettingsInterface tracerServiceSettings;

    private final SystemInfoService systemInfoService;

    private JPanel contentPanel;

    public TracerServiceSettingsPanel(final TracerServiceSettingsInterface tracerServiceSettings,
        final SystemInfoService systemInfoService) {
        super(null);

        this.tracerServiceSettings = tracerServiceSettings;
        this.systemInfoService = systemInfoService;
    }

    @Override
    public JPanel createPanel(Consumer<Boolean> bufferingCallback) {

        setBufferingCallback(bufferingCallback);

        FormBuilder dialogBuilder = null;
        boolean debugDialog = false;
        if (debugDialog) {
            JPanel panel = new FormDebugPanel();
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        else {
            JPanel panel = new JPanel(new BorderLayout());
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        dialogBuilder.border(Paddings.TABBED_DIALOG);

        // use value model here and only update settings on focus lost
        preparePresentationModel(this.tracerServiceSettings);

        // prepare the panel

        // serial port provider
        final ArrayListModel<String> serialProviderList = new ArrayListModel<>();
        serialProviderList.add("RXTX");
        serialProviderList.add("SCM");
        serialProviderList.add("SPSW");
        serialProviderList.add("JSerialComm");
        serialProviderList.add("PureJavaComm");

        final BufferedValueModel selectionHolderSerialProvider =
            presentationModel.getBufferedModel(TracerServiceSettingsInterface.PROPERTY_SERIALPORT_PROVIDER);

        SelectionInList<String> serialProviderSelection =
            new SelectionInList<String>((ListModel<String>) serialProviderList);

        ComboBoxAdapter<String> comboBoxAdapterSerialProvider =
            new ComboBoxAdapter<String>(serialProviderSelection, selectionHolderSerialProvider);
        final JComboBox<String> comboSerialProvider = new JComboBox<>();
        comboSerialProvider.setModel(comboBoxAdapterSerialProvider);

        dialogBuilder.add(Resources.getString(TracerServiceSettingsPanel.class, "serialPortProvider") + ":").xy(1, 1);
        dialogBuilder.add(comboSerialProvider).xyw(3, 1, 5);

        // downstream port
        final BufferedValueModel bufferedDownstreamPortModel =
            presentationModel.getBufferedModel(TracerServiceSettingsInterface.PROPERTY_DOWNSTREAM_PORT);

        dialogBuilder.add(Resources.getString(TracerServiceSettingsPanel.class, "downstreamPort") + ":").xy(1, 3);

        ArrayListModel<SerialPortInfo> commPorts = new ArrayListModel<>();
        for (SerialPortInfo commPort : systemInfoService.getRegisteredSerialPorts()) {
            commPorts.add(commPort);
        }

        SerialPortInfo commPortNone = new SerialPortInfo();
        commPortNone.setPortName(CommPort.NONE);
        commPorts.add(0, commPortNone);
        LOGGER.info("Prepared serial ports to diplay in combobox: {}", commPorts);

        SelectionInList<SerialPortInfo> serialPortSelection =
            new SelectionInList<SerialPortInfo>((ListModel<SerialPortInfo>) commPorts);

        // use converter
        final SerialPortInfoConverter converter = new SerialPortInfoConverter(commPorts);
        final ValueModel downstreamPortConverterModel = new ConverterValueModel(bufferedDownstreamPortModel, converter);

        ComboBoxAdapter<SerialPortInfo> comboBoxAdapterSerialPortDownstream =
            new ComboBoxAdapter<SerialPortInfo>(serialPortSelection, downstreamPortConverterModel);

        final JComboBox<SerialPortInfo> commPortsComboDownstream = new JComboBox<>();
        commPortsComboDownstream.setModel(comboBoxAdapterSerialPortDownstream);
        commPortsComboDownstream.setRenderer(new SerialPortInfoListRenderer());

        dialogBuilder.add(commPortsComboDownstream).xyw(3, 3, 5);

        // upstream port
        final BufferedValueModel bufferedUpstreamPortModel =
            presentationModel.getBufferedModel(TracerServiceSettingsInterface.PROPERTY_UPSTREAM_PORT);

        dialogBuilder.add(Resources.getString(TracerServiceSettingsPanel.class, "upstreamPort") + ":").xy(1, 5);

        // use converter
        final ValueModel upstreamPortConverterModel = new ConverterValueModel(bufferedUpstreamPortModel, converter);

        ComboBoxAdapter<SerialPortInfo> comboBoxAdapterSerialPortUpstream =
            new ComboBoxAdapter<SerialPortInfo>(serialPortSelection, upstreamPortConverterModel);

        final JComboBox<SerialPortInfo> commPortsComboUpstream = new JComboBox<>();
        commPortsComboUpstream.setModel(comboBoxAdapterSerialPortUpstream);
        commPortsComboUpstream.setRenderer(new SerialPortInfoListRenderer());

        dialogBuilder.add(commPortsComboUpstream).xyw(3, 5, 5);

        contentPanel = dialogBuilder.build();
        contentPanel.setOpaque(false);
        return contentPanel;
    }

    @Override
    public String getTabTitle() {
        return Resources.getString(getClass(), "tab-tracer-service.title");
    }

    @Override
    public String getTabTooltip() {
        return Resources.getString(getClass(), "tab-tracer-service.tooltip");
    }

    private static final class SerialPortInfoListRenderer extends DefaultListCellRenderer {

        private static final long serialVersionUID = 1L;

        @Override
        public Component getListCellRendererComponent(
            JList<?> list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
            Component comp = super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);
            if (value instanceof SerialPortInfo) {
                SerialPortInfo serialPortInfo = (SerialPortInfo) value;

                if (serialPortInfo.getPortName() != null) {
                    StringBuilder sb = new StringBuilder();
                    sb.append(serialPortInfo.getPortName());
                    if (StringUtils.isNotBlank(serialPortInfo.getProductString())) {
                        sb.append(" - ").append(serialPortInfo.getProductString());
                    }
                    if (StringUtils.isNotBlank(serialPortInfo.getSerialNumber())) {
                        sb.append(" - ").append(serialPortInfo.getSerialNumber());
                    }
                    setText(sb.toString());
                }
                else {
                    setText(serialPortInfo.getPortName());
                }
            }
            else {
                // set the empty string
                setText(" ");
            }
            return comp;
        }
    }
}
