package org.bidib.wizard.tracer.service;

import java.io.ByteArrayOutputStream;
import java.util.concurrent.atomic.AtomicBoolean;

import org.bidib.jbidibc.messages.MessageProcessor;
import org.bidib.jbidibc.messages.exception.ProtocolException;
import org.bidib.jbidibc.messages.message.BidibMessageInterface;
import org.bidib.jbidibc.messages.message.BidibResponseFactory;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.jbidibc.messages.utils.MessageUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class UpstreamTracerMessageProcessor implements MessageProcessor {

    private static final Logger LOGGER = LoggerFactory.getLogger(UpstreamTracerMessageProcessor.class);

    // private static final Logger MSG_RX_LOGGER = LoggerFactory.getLogger("RX");

    private final org.bidib.jbidibc.messages.logger.Logger messageLogger;

    private final boolean checkCRC;

    private final BidibResponseFactory responseFactory;

    private final AtomicBoolean isFirstPacket = new AtomicBoolean(true);

    private final org.bidib.jbidibc.messages.logger.Logger splitMessageLogger;

    public UpstreamTracerMessageProcessor(final org.bidib.jbidibc.messages.logger.Logger messageLogger,
        boolean checkCRC) {

        this.messageLogger = messageLogger;

        this.responseFactory = new BidibResponseFactory();
        this.responseFactory.initialize();

        this.checkCRC = checkCRC;

        splitMessageLogger = new org.bidib.jbidibc.messages.logger.Logger() {

            @Override
            public void debug(String format, Object... arguments) {
                LOGGER.debug(format, arguments);
            }

            @Override
            public void info(String format, Object... arguments) {
                LOGGER.info(format, arguments);
            }

            @Override
            public void warn(String format, Object... arguments) {
                LOGGER.warn(format, arguments);
            }

            @Override
            public void error(String format, Object... arguments) {
                LOGGER.error(format, arguments);
            }
        };

        LOGGER.info("Create new ProxyMessageProcessor with checkCRC: {}", checkCRC);
    }

    @Override
    public void processMessages(ByteArrayOutputStream messageData) throws ProtocolException {

        if (messageData.size() < 1) {
            LOGGER.info("No data in provided buffer, skip process messages.");
            return;
        }

        // if a CRC error is detected in splitMessages the reading loop will terminate ...
        MessageUtils.splitBidibMessages(splitMessageLogger, messageData, checkCRC, isFirstPacket, messageArray -> processMessage(messageArray));
    }

    private void processMessage(byte[] messageArray) throws ProtocolException {
        // the first packet was successfully parsed
        this.isFirstPacket.set(false);

        BidibMessageInterface message = null;

        try {
            message = responseFactory.create(messageArray);

            StringBuilder sb = new StringBuilder("<< ");
            sb.append(message);
            sb.append(" : ");
            sb.append(ByteUtils.bytesToHex(messageArray));

            messageLogger.info(sb.toString());
        }
        catch (ProtocolException ex) {
            LOGGER.warn("Process received messages failed: {}", ByteUtils.bytesToHex(messageArray), ex);

            StringBuilder sb = new StringBuilder("<< received invalid: ");
            sb.append(message);
            sb.append(" : ");
            sb.append(ByteUtils.bytesToHex(messageArray));

            messageLogger.warn(sb.toString());

            throw ex;
        }
        catch (Exception ex) {
            LOGGER.warn("Process received messages failed: {}", ByteUtils.bytesToHex(messageArray), ex);
        }
    }

}
