package org.bidib.wizard.tracer.service;

import java.io.ByteArrayOutputStream;
import java.util.List;

import org.bidib.jbidibc.core.logger.LoggerWrapper;
import org.bidib.jbidibc.messages.BidibLibrary;
import org.bidib.jbidibc.messages.MessageProcessor;
import org.bidib.jbidibc.messages.exception.ProtocolException;
import org.bidib.jbidibc.messages.message.BidibMessageInterface;
import org.bidib.jbidibc.messages.message.BidibRequestFactory;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DownstreamTracerMessageProcessor implements MessageProcessor {

    private static final Logger LOGGER = LoggerFactory.getLogger(DownstreamTracerMessageProcessor.class);

    // private static final Logger MSG_RX_LOGGER = LoggerFactory.getLogger("RX");

    private final org.bidib.jbidibc.messages.logger.Logger messageLogger;

    private boolean checkCRC;

    private final BidibRequestFactory requestFactory;

    private boolean isFirstPacket = true;

    private final org.bidib.jbidibc.messages.logger.Logger splitMessageLogger;

    public DownstreamTracerMessageProcessor(final org.bidib.jbidibc.messages.logger.Logger messageLogger,
        boolean checkCRC) {

        this.messageLogger = messageLogger;

        this.requestFactory = new BidibRequestFactory();
        this.requestFactory.initialize();

        this.checkCRC = checkCRC;

        splitMessageLogger = new LoggerWrapper(LOGGER);

        LOGGER.info("Create new ProxyMessageProcessor with checkCRC: {}", checkCRC);
    }

    @Override
    public void processMessages(final ByteArrayOutputStream messageData, final String contextKey)
        throws ProtocolException {

        if (messageData.size() < 1) {
            LOGGER.info("No data in provided buffer, skip process messages.");
            return;
        }

        // add the magic to terminate the message packet
        messageData.write(ByteUtils.getLowByte(BidibLibrary.BIDIB_PKT_MAGIC));

        LOGGER.info("Process messages, current messageData: {}", ByteUtils.bytesToHex(messageData));

        List<BidibMessageInterface> bidibMessages = null;
        try {
            bidibMessages = requestFactory.create(messageData.toByteArray());
            for (BidibMessageInterface message : bidibMessages) {
                LOGGER.info("Current bidibMessages: {}", bidibMessages);

                StringBuilder sb = new StringBuilder(">> ");
                sb.append(bidibMessages);
                sb.append(" : ");
                sb.append(ByteUtils.bytesToHex(message.getContent()));

                messageLogger.info(sb.toString());
            }
        }
        catch (ProtocolException ex) {
            LOGGER.warn("Process received messages failed: {}", ByteUtils.bytesToHex(messageData), ex);

            StringBuilder sb = new StringBuilder(">> received invalid: ");
            sb.append(bidibMessages);
            sb.append(" : ");
            sb.append(ByteUtils.bytesToHex(messageData.toByteArray()));

            messageLogger.warn(sb.toString());

            throw ex;
        }
        catch (Exception ex) {
            LOGGER.warn("Process received messages failed: {}", ByteUtils.bytesToHex(messageData), ex);
        }
        finally {

            // reset the provided data buffer
            messageData.reset();
        }
    }

}
