package org.bidib.wizard.tracer.service;

import java.io.ByteArrayOutputStream;
import java.util.concurrent.atomic.AtomicBoolean;

import org.bidib.jbidibc.core.logger.LoggerWrapper;
import org.bidib.jbidibc.messages.MessageProcessor;
import org.bidib.jbidibc.messages.exception.ProtocolException;
import org.bidib.jbidibc.messages.message.BidibMessageInterface;
import org.bidib.jbidibc.messages.message.BidibRequestFactory;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.jbidibc.messages.utils.MessageUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class UpstreamTracerMessageProcessor implements MessageProcessor {

    private static final Logger LOGGER = LoggerFactory.getLogger(UpstreamTracerMessageProcessor.class);

    private final org.bidib.jbidibc.messages.logger.Logger messageLogger;

    private final boolean checkCRC;

    private final BidibRequestFactory requestFactory;

    private final AtomicBoolean isFirstPacket = new AtomicBoolean(true);

    private final org.bidib.jbidibc.messages.logger.Logger splitMessageLogger;

    public UpstreamTracerMessageProcessor(final org.bidib.jbidibc.messages.logger.Logger messageLogger,
        boolean checkCRC) {

        this.messageLogger = messageLogger;

        this.requestFactory = new BidibRequestFactory();
        this.requestFactory.initialize();

        this.checkCRC = checkCRC;

        splitMessageLogger = new LoggerWrapper(LOGGER);
        LOGGER.info("Create new ProxyMessageProcessor with checkCRC: {}", checkCRC);
    }

    @Override
    public void processMessages(ByteArrayOutputStream messageData, final String contextKey) throws ProtocolException {

        if (messageData.size() < 1) {
            LOGGER.info("No data in provided buffer, skip process messages.");
            return;
        }

        // if a CRC error is detected in splitMessages the reading loop will terminate ...
        MessageUtils
            .splitBidibMessages(splitMessageLogger, messageData, checkCRC, isFirstPacket,
                messageArray -> processMessage(messageArray, contextKey));
    }

    private void processMessage(byte[] messageArray, final String contextKey) throws ProtocolException {
        // the first packet was successfully parsed
        this.isFirstPacket.set(false);

        BidibMessageInterface message = null;

        try {
            message = requestFactory.createConcreteMessage(messageArray);

            StringBuilder sb = new StringBuilder("<< ");
            sb.append(message);
            sb.append(" : ");
            sb.append(ByteUtils.bytesToHex(messageArray));

            messageLogger.info(sb.toString());
        }
        catch (ProtocolException ex) {
            LOGGER.warn("Process received messages failed: {}", ByteUtils.bytesToHex(messageArray), ex);

            StringBuilder sb = new StringBuilder("<< received invalid: ");
            sb.append(message);
            sb.append(" : ");
            sb.append(ByteUtils.bytesToHex(messageArray));

            messageLogger.warn(sb.toString());

            throw ex;
        }
        catch (Exception ex) {
            LOGGER.warn("Process received messages failed: {}", ByteUtils.bytesToHex(messageArray), ex);
        }
    }

}
