package org.bidib.jbidibc.debug;

import java.io.ByteArrayOutputStream;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DebugMessageReceiver implements DebugMessageProcessor {
    private static final Logger LOGGER = LoggerFactory.getLogger(DebugMessageReceiver.class);

    protected AtomicBoolean running = new AtomicBoolean();

    private final Set<DebugMessageListener> messageListeners =
        Collections.synchronizedSet(new LinkedHashSet<DebugMessageListener>());

    public DebugMessageReceiver() {

        // enable the running flag
        running.set(true);
    }

    public void addMessageListener(DebugMessageListener listener) {
        synchronized (messageListeners) {
            if (!messageListeners.contains(listener)) {

                LOGGER.info("Add new message listener: {}", listener);
                messageListeners.add(listener);
            }
            else {
                LOGGER.warn("Message listener is already registered: {}", listener);
            }
        }
    }

    public void removeMessageListener(DebugMessageListener listener) {
        synchronized (messageListeners) {
            if (messageListeners.contains(listener)) {

                LOGGER.info("Remove message listener: {}", listener);
                messageListeners.remove(listener);
            }
            else {
                LOGGER.warn("Message listener was not registered: {}", listener);
            }
        }
    }

    @Override
    public void processMessages(final ByteArrayOutputStream output) {
        String message = output.toString();
        LOGGER.debug("processMessages, received message: {}", message);

        output.reset();

        publishMessage(message);
    }

    protected void publishMessage(final String message) {

        synchronized (messageListeners) {
            for (DebugMessageListener l : messageListeners) {
                l.debugMessage(message);
            }
        }
    }

    @Override
    public void disable() {
        running.set(false);
    }

    @Override
    public void enable() {
        running.set(true);
    }

    public static String bytesToStringUTFCustom(byte[] bytes) {
        char[] buffer = new char[bytes.length >> 1];
        for (int i = 0; i < buffer.length; i++) {
            int bpos = i << 1;
            char c = (char) (((bytes[bpos] & 0x00FF) << 8) + (bytes[bpos + 1] & 0x00FF));
            buffer[i] = c;
        }
        return new String(buffer);
    }

}
