package org.bidib.jbidibc.debug;

import java.io.ByteArrayOutputStream;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;

import org.bidib.jbidibc.messages.ConnectionListener;
import org.bidib.jbidibc.messages.helpers.Context;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractDebugReader implements DebugInterface {

    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractDebugReader.class);

    private static final Logger MSG_RAW_LOGGER = LoggerFactory.getLogger("DEBUG_RAW");

    private final DebugMessageProcessor messageReceiver;

    private ConnectionListener connectionListener;

    private String requestedPortName;

    private final AtomicBoolean sendEnabled = new AtomicBoolean();

    private BlockingQueue<byte[]> receiveQueue = new LinkedBlockingQueue<>();

    private Thread receiveQueueWorker;

    protected Thread receiverThread;

    protected AtomicBoolean receiverRunning = new AtomicBoolean();

    private AtomicLong receiveQueueWorkerThreadId = new AtomicLong();

    public AbstractDebugReader(final DebugMessageProcessor messageReceiver) {
        this.messageReceiver = messageReceiver;
    }

    @Override
    public void initialize() {

    }

    protected void setSendEnabled(boolean sendEnabled) {
        this.sendEnabled.set(sendEnabled);
    }

    protected void setRequestedPortName(String requestedPortName) {
        this.requestedPortName = requestedPortName;
    }

    protected String getRequestedPortName() {
        return requestedPortName;
    }

    @Override
    public DebugMessageProcessor getMessageReceiver() {
        return messageReceiver;
    }

    /**
     * @return the connectionListener
     */
    public ConnectionListener getConnectionListener() {
        return connectionListener;
    }

    /**
     * @param connectionListener
     *            the connectionListener to set
     */
    public void setConnectionListener(ConnectionListener connectionListener) {
        this.connectionListener = connectionListener;
    }

    public void startReceiverAndQueues(DebugMessageProcessor serialMessageReceiver, Context context) {
        LOGGER.info("Start receiver and queues.");

        if (receiverThread == null) {
            receiverThread = createReceiverThread();
        }

        if (receiverThread != null) {
            receiverThread.start();
        }

        serialMessageReceiver.enable();
        startReceiveQueueWorker();
    }

    protected Thread createReceiverThread() {
        return null;
    }

    protected void addDataToReceiveQueue(final ByteArrayOutputStream output) {

        byte[] bytes = output.toByteArray();

        byte[] buffer = new byte[bytes.length];
        System.arraycopy(bytes, 0, buffer, 0, bytes.length);

        // if (MSG_RAW_LOGGER.isInfoEnabled()) {
        // MSG_RAW_LOGGER
        // .info("<<<< len: {}, data: {}, string: {}", bytes.length, ByteUtils.bytesToHex(buffer),
        // new String(bytes));
        // }
        //
        // boolean added = receiveQueue.offer(buffer);
        // if (!added) {
        // LOGGER.error("The message was not added to the receive queue: {}", ByteUtils.bytesToHex(buffer));
        // }
        addDataToReceiveQueue(buffer);

        output.reset();
    }

    protected void addDataToReceiveQueue(byte[] data) {

        if (MSG_RAW_LOGGER.isInfoEnabled()) {
            MSG_RAW_LOGGER.info("<<<< len: {}, data: {}", data.length, ByteUtils.bytesToHex(data));
        }

        boolean added = receiveQueue.offer(data);
        if (!added) {
            LOGGER.error("The message was not added to the receive queue: {}", ByteUtils.bytesToHex(data));
        }
    }

    protected void startReceiveQueueWorker() {
        receiverRunning.set(true);

        LOGGER.info("Start the receiveQueueWorker. Current receiveQueueWorker: {}", receiveQueueWorker);
        receiveQueueWorker = new Thread(new Runnable() {

            @Override
            public void run() {
                try {
                    processReceiveQueue();
                }
                catch (Exception ex) {
                    LOGGER.warn("The processing of the receive queue was terminated with an exception!", ex);

                    // running.set(false);
                }

                LOGGER.info("Process receive queue has finished.");
            }
        }, "receiveQueueWorker");

        try {
            receiveQueueWorkerThreadId.set(receiveQueueWorker.getId());
            receiveQueueWorker.start();
        }
        catch (Exception ex) {
            LOGGER.error("Start the receiveQueueWorker failed.", ex);
        }

        LOGGER.info("Start the receiveQueueWorker finished. Current receiveQueueWorker: {}", receiveQueueWorker);
    }

    protected void stopReceiveQueueWorker() {
        LOGGER.info("Stop the receive queue worker.");
        receiverRunning.set(false);

        try {
            receiveQueueWorker.interrupt();

            receiveQueueWorker.join(1000);

            LOGGER.info("receiveQueueWorker has finished.");
        }
        catch (Exception ex) {
            LOGGER.warn("Interrupt receiveQueueWorker failed.", ex);
        }
        receiveQueueWorker = null;
    }

    private void processReceiveQueue() {
        byte[] bytes = null;
        LOGGER.info("The receiveQueueWorker is ready for processing, requestedPortName: {}", requestedPortName);

        while (receiverRunning.get()) {
            try {
                // get the message to process
                bytes = receiveQueue.take();

                if (bytes != null) {
                    // process
                    try {

                        ByteArrayOutputStream output = new ByteArrayOutputStream();
                        output.write(bytes);

                        getMessageReceiver().processMessages(output);
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Process received bytes failed.", ex);
                    }

                }
            }
            catch (InterruptedException ex) {
                LOGGER.warn("Get message from receiveQueue failed because thread was interrupted.");
            }
            catch (Exception ex) {
                LOGGER.warn("Get message from receiveQueue failed.", ex);
                bytes = null;
            }

        }

        LOGGER.info("The receiveQueueWorker has finished processing, requestedPortName: {}", requestedPortName);
        receiveQueueWorkerThreadId.set(0);
    }

    protected void closeHandle() {

    }

    protected void triggerClosePort() {
        LOGGER.warn("Close the port.");
        Thread worker = new Thread(() -> {

            LOGGER.info("Start close port because error was detected.");
            try {
                // the listeners are notified in close()
                close();
            }
            catch (Exception ex) {
                LOGGER.warn("Close after error failed.", ex);
            }
            LOGGER.warn("The port was closed.");
        });
        worker.start();
    }

    protected void stopReceiverThread() {
        LOGGER.info("Stop the receiver thread by set the running flag to false.");
        receiverRunning.set(false);

        if (receiverThread != null) {
            LOGGER.info("Wait for termination of receiver thread.");

            synchronized (receiverThread) {

                // close handle will release the event
                closeHandle();

                try {
                    receiverThread.join(5000);
                }
                catch (InterruptedException ex) {
                    LOGGER.warn("Wait for termination of receiver thread failed.", ex);
                }
            }

            LOGGER.info("Free the receiver thread.");
            receiverThread = null;
        }

    }
}
