package org.bidib.jbidibc.debug;

import java.lang.reflect.Constructor;

import org.apache.commons.lang3.SystemUtils;
import org.bidib.jbidibc.messages.exception.InvalidConfigurationException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DebugReaderFactory {

    private static final Logger LOGGER = LoggerFactory.getLogger(DebugReaderFactory.class);

    public enum SerialImpl {
        // @formatter:off
        
        FTDI("org.bidib.jbidibc.ftdi.serial.debug.DebugReaderEventWindows"), 
        RXTX("org.bidib.jbidibc.rxtx.debug.DebugReader"), 
        PUREJAVACOMM("org.bidib.jbidibc.purejavacomm.debug.DebugReader"), 
        SCM("org.bidib.jbidibc.scm.debug.DebugReader"), 
        JSERIALCOMM("org.bidib.jbidibc.jserialcomm.debug.DebugReader"), 
        SPSW_NET("org.bidib.jbidibc.spsw.debug.NetDebugReader");
        
        // @formatter:on

        String implClass;

        SerialImpl(String implClass) {
            this.implClass = implClass;
        }

        public String getImplClass() {
            return implClass;
        }
    }

    public static DebugInterface getDebugReader(SerialImpl impl, final DebugMessageProcessor messageReceiver) {
        LOGGER.info("Create debug reader, serial impl: {}", impl);

        DebugInterface debugInterface = null;

        final String clazzName;
        if (SerialImpl.FTDI == impl) {
            // check the OS
            if (SystemUtils.IS_OS_WINDOWS) {
                LOGGER.info("Create the FTDI DebugReaderEventWindows.");
                clazzName = "org.bidib.jbidibc.ftdi.serial.debug.DebugReaderEventWindows";
            }
            else if (SystemUtils.IS_OS_LINUX) {
                LOGGER.info("Create the FTDI DebugReaderEventLinux.");
                clazzName = "org.bidib.jbidibc.ftdi.serial.debug.DebugReaderEventLinux";
            }
            else {
                LOGGER.info("Create the FTDI DebugReaderBlock.");
                clazzName = "org.bidib.jbidibc.ftdi.serial.debug.DebugReaderBlock";
            }
        }
        else {
            clazzName = impl.getImplClass();
        }

        try {
            Class<?> clazz = Class.forName(clazzName);

            Constructor<?> method = clazz.getConstructor(DebugMessageProcessor.class);
            debugInterface = (DebugInterface) method.newInstance(messageReceiver);
        }
        catch (Exception ex) {
            LOGGER.error("Create debugReader instance with newInstance failed: {}", clazzName, ex);

            throw new InvalidConfigurationException("Create debugReader instance failed: " + clazzName);
        }

        debugInterface.initialize();

        return debugInterface;
    }
}
