package org.bidib.jbidibc.decoder.external;

import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.StringSelection;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.List;

import org.apache.commons.io.IOUtils;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import eu.esu._2010.lokprogrammer.metadata.FunctionType;
import eu.esu._2010.lokprogrammer.metadata.MetaType;

public class EsuXReaderTest {
    private static final Logger LOGGER = LoggerFactory.getLogger(EsuXReaderTest.class);

    @Test
    void readEsuXFile() throws FileNotFoundException, IOException {

        final InputStream esuXFile = EsuXReaderTest.class.getResourceAsStream("/external/Re620.esux");

        final EsuXReader esuXReader = new EsuXReader();

        String metaData = esuXReader.getMetaData(esuXFile);

        Assertions.assertThat(metaData).isNotBlank();

        LOGGER.info("Len of meta: {}", metaData.length());

        Assertions.assertThat(metaData).endsWith("</meta>");

        final MetaType meta = esuXReader.getMetaType(metaData);
        Assertions.assertThat(meta).isNotNull();

        LOGGER.info("Returned meta: {}", meta);
    }

    @Test
    void parseMeta() throws IOException {
        // String metaData =
        // "<?xml version=\"1.0\" encoding=\"utf-8\"?><meta
        // xmlns=\"http://www.esu.eu/2010/LOKPROGRAMMER/Metadata\"><name>SBB Re 6/6 /
        // Re620</name><address>3</address></meta>";

        String metaData = IOUtils.resourceToString("/external/Re620.xml", StandardCharsets.UTF_8);

        final EsuXReader esuXReader = new EsuXReader();

        final MetaType meta = esuXReader.getMetaType(metaData);
        Assertions.assertThat(meta).isNotNull();

        LOGGER.info("Returned meta: {}", meta);

        final String language = "de";
        // get the functions
        final List<FunctionType> functions = meta.getFunction();
        for (FunctionType function : functions) {

            int functionId = function.getId().intValue();
            // search the description by lang
            final FunctionType.Description description =
                function
                    .getDescription().stream().filter(desc -> desc.getLang().equals(language)).findFirst().orElseThrow(
                        () -> new IllegalArgumentException("No description for language available: " + language));

            LOGGER.info("Current function id: {}, description: {}", functionId, description.getValue());
        }

        StringBuilder sb = new StringBuilder();
        final String languageEn = "en";
        // get the functions
        for (FunctionType function : functions) {

            int functionId = function.getId().intValue();
            // search the description by lang
            final FunctionType.Description description =
                function
                    .getDescription().stream().filter(desc -> desc.getLang().equals(languageEn)).findFirst()
                    .orElseThrow(
                        () -> new IllegalArgumentException("No description for language available: " + languageEn));

            LOGGER.info("Current function id: {}, description: {}", functionId, description.getValue());
            sb
                .append(functionId).append("\t\t").append(description.getValue()).append("\t\t")
                .append(System.lineSeparator());
        }

        try {
            StringSelection stringSelection = new StringSelection(sb.toString());
            Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
            clipboard.setContents(stringSelection, null);
        }
        catch (Exception ex) {
            LOGGER.warn("Copy functions to clipboard failed.", ex);
        }
    }
}
