package org.bidib.jbidibc.decoder.json;

import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.InetSocketAddress;
import java.net.MalformedURLException;
import java.net.Proxy;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Properties;
import java.util.TimeZone;

import org.apache.commons.collections4.IterableUtils;
import org.apache.commons.collections4.Predicate;
import org.assertj.core.api.Assertions;
import org.bidib.jbidibc.decoder.decoderdb.AbstractDecoderDbAccess;
import org.bidib.jbidibc.decoder.decoderdb.AbstractDecoderDbAccess.DataFormat;
import org.bidib.jbidibc.decoder.decoderdb.ProxyUtils;
import org.bidib.jbidibc.decoder.json.DecoderDbStatusResponse.DecoderStatusResponse;
import org.bidib.jbidibc.decoder.json.DecoderDbStatusResponse.FirmwareStatusResponse;
import org.bidib.jbidibc.decoder.json.DecoderDbStatusResponse.ManufacturersStatusResponse;
import org.bidib.jbidibc.decoder.schema.manufacturers.ManufacturerType;
import org.bidib.jbidibc.decoder.schema.manufacturers.ManufacturersList;
import org.bidib.jbidibc.decoder.schema.manufacturers.ManufacturersType;
import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.MethodOrderer;
import org.junit.jupiter.api.Order;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestMethodOrder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.module.jaxb.JaxbAnnotationModule;

@TestMethodOrder(MethodOrderer.OrderAnnotation.class)
public class DecoderDbStatusResponseTest {

    private static final Logger LOGGER = LoggerFactory.getLogger(DecoderDbStatusResponseTest.class);

    private static final String REST_SERVICE_URI = "https://www.decoderdb.de";

    @Test
    @Order(1)
    public void listAll() throws JsonParseException, JsonMappingException, IOException {

        ObjectMapper mapper = new ObjectMapper(); // can reuse, share globally
        JaxbAnnotationModule module = new JaxbAnnotationModule();
        // configure as necessary
        mapper.registerModule(module);

        InputStream is = DecoderDbStatusResponseTest.class.getResourceAsStream("/json/listAll.json");

        DecoderDbStatusResponse listAll = mapper.readValue(is, DecoderDbStatusResponse.class);
        LOGGER.info("Loaded listAll: {}", listAll);

        Assertions.assertThat(listAll).isNotNull();
        Assertions.assertThat(listAll.getManufacturers()).isNotNull();
        Assertions.assertThat(listAll.getManufacturers().getFilename()).isEqualTo("Manufacturers.xml");
        Calendar cal = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
        cal.set(2016, Calendar.MAY, 19, 00, 00, 00);
        cal.set(Calendar.MILLISECOND, 0);
        Assertions.assertThat(listAll.getManufacturers().getNmraListDate()).isEqualTo(cal.getTime());
        Assertions.assertThat(listAll.getManufacturers().isValid()).isTrue();

        Assertions.assertThat(listAll.getDecoderDetections()).isNotNull();
        Assertions.assertThat(listAll.getDecoderDetections().getFilename()).isEqualTo("decoderDetection.xml");
        Assertions
            .assertThat(listAll.getDecoderDetections().getLink())
            .isEqualTo("https://www.decoderdb.de/?decoderdetection");
        Assertions.assertThat(listAll.getDecoderDetections().isValid()).isTrue();

        Assertions.assertThat(listAll.getDecoder()).isNotNull();
        Assertions.assertThat(listAll.getDecoder().length).isEqualTo(15);

        DecoderStatusResponse product = listAll.getDecoder()[0];
        Assertions.assertThat(product).isNotNull();
        Assertions.assertThat(product.getFilename()).isEqualTo("Product_0_NMRA-Standard.xml");
        Assertions.assertThat(product.getManufacturerId()).isEqualTo(Integer.valueOf(0));
        Assertions.assertThat(product.getManufacturerExtendedId()).isNull();
        Assertions.assertThat(product.getName()).isEqualTo("NMRA-Standard");

        product = listAll.getDecoder()[1];
        Assertions.assertThat(product).isNotNull();
        Assertions.assertThat(product.getFilename()).isEqualTo("Product_13-256_DecoderDBLok.xml");
        Assertions.assertThat(product.getManufacturerId()).isEqualTo(Integer.valueOf(13));
        Assertions.assertThat(product.getManufacturerExtendedId()).isEqualTo(Integer.valueOf(256));
        Assertions.assertThat(product.getName()).isEqualTo("Decoder DB Lok");
    }

    @Test
    @Disabled
    @Order(2)
    public void listManufacturersRemote() throws MalformedURLException, URISyntaxException {
        LOGGER.info("listManufacturersRemote");

        ManufacturersList listManufacturers =
            AbstractDecoderDbAccess
                .fetch("cv", "cv".toCharArray(), REST_SERVICE_URI + "/?manufacturers", ManufacturersList.class,
                    DataFormat.xml);

        LOGGER.info("Retrieved listManufacturers: {}", listManufacturers);

        Assertions.assertThat(listManufacturers).isNotNull();

        Assertions.assertThat(listManufacturers.getManufacturers()).isNotNull();
        Assertions.assertThat(listManufacturers.getManufacturers().getManufacturer()).isNotNull();
        Assertions.assertThat(listManufacturers.getManufacturers().getManufacturer()).isNotEmpty();

        // find the DIY manufacturer
        final short ID = 13;

        List<ManufacturerType> manufacturers = listManufacturers.getManufacturers().getManufacturer();
        ManufacturerType manufacturerDiy = IterableUtils.find(manufacturers, new Predicate<ManufacturerType>() {

            @Override
            public boolean evaluate(ManufacturerType manufacturer) {
                return manufacturer.getId() == ID;
            }
        });

        Assertions.assertThat(manufacturerDiy).isNotNull();
        LOGGER.info("Found DIY manufacturer: {}", manufacturerDiy);

        // find the OpenDCC manufacturer
        final Integer EXTENDED_ID_OPENDCC = Integer.valueOf(258);

        ManufacturerType manufacturerOpenDcc = IterableUtils.find(manufacturers, new Predicate<ManufacturerType>() {

            @Override
            public boolean evaluate(ManufacturerType manufacturer) {
                return manufacturer.getId() == ID && EXTENDED_ID_OPENDCC.equals(manufacturer.getExtendedId());
            }
        });

        Assertions.assertThat(manufacturerOpenDcc).isNotNull();
        LOGGER.info("Found OpenDCC manufacturer: {}", manufacturerOpenDcc);
    }

    @Test
    @Disabled
    @Order(2)
    public void listAllRemote() throws MalformedURLException, URISyntaxException {
        LOGGER.info("listAllRemote");

        DecoderDbStatusResponse listAll =
            AbstractDecoderDbAccess
                .fetch("cv", "cv".toCharArray(), REST_SERVICE_URI + "/?listAll", DecoderDbStatusResponse.class,
                    DataFormat.json);

        LOGGER.info("Retrieved listAll: {}", listAll);

        ManufacturersStatusResponse manufacturersStatusResponse = listAll.getManufacturers();
        Assertions.assertThat(manufacturersStatusResponse).isNotNull();

        Date nmraListDate = manufacturersStatusResponse.getNmraListDate();
        Assertions.assertThat(nmraListDate).isNotNull();
        Date lastUpdate = manufacturersStatusResponse.getLastUpdate();
        Assertions.assertThat(lastUpdate).isNotNull();

        // get the decoders
        DecoderStatusResponse[] decoders = listAll.getDecoder();
        Assertions.assertThat(decoders).isNotNull();
        Assertions.assertThat(decoders.length > 0).isTrue();

        LOGGER.info("Current decoders: {}", new Object[] { decoders });

        // get the firmware items
        FirmwareStatusResponse[] firmwareItems = listAll.getFirmware();
        Assertions.assertThat(firmwareItems).isNotNull();
        Assertions.assertThat(firmwareItems.length > 0).isTrue();

        LOGGER.info("Current firmwareItems: {}", new Object[] { firmwareItems });

        // get the manufacturers
        String link = manufacturersStatusResponse.getLink();
        String filename = manufacturersStatusResponse.getFilename();
        LOGGER
            .info("Current nmraListDate: {}, lastUpdate: {}, filename: {}, link: {}", nmraListDate, lastUpdate,
                filename, link);

        ManufacturersList manufacturersList =
            AbstractDecoderDbAccess.fetch("cv", "cv".toCharArray(), link, ManufacturersList.class, DataFormat.xml);

        Assertions.assertThat(manufacturersList).isNotNull();

        LOGGER.info("Current manufacturersList: {}", manufacturersList);

        Assertions.assertThat(manufacturersList.getManufacturers()).isNotNull();
        ManufacturersType manufacturersType = manufacturersList.getManufacturers();
        List<ManufacturerType> manufacturers = manufacturersType.getManufacturer();

        Assertions.assertThat(manufacturers).hasSizeGreaterThan(0);
    }

    @Test
    public void proxyTest() throws URISyntaxException, MalformedURLException {

        URL url = new URL("http://www.bidib.org");

        Proxy proxy = ProxyUtils.findProxy(url.toURI());

        LOGGER.info("Current proxy: {}", proxy);

        if (!Proxy.NO_PROXY.equals(proxy)) {
            try {
                InetSocketAddress addr = (InetSocketAddress) proxy.address();
                Properties systemSettings = System.getProperties();
                systemSettings.put("proxySet", "true");
                systemSettings.put("http.proxyHost", addr.getHostName());
                systemSettings.put("http.proxyPort", Integer.toString(addr.getPort()));
                HttpURLConnection con = (HttpURLConnection) url.openConnection();
                LOGGER.info(con.getResponseCode() + " : " + con.getResponseMessage());
                LOGGER.info("HTTP_OK: {} ", con.getResponseCode() == HttpURLConnection.HTTP_OK);
            }
            catch (Exception e) {
                LOGGER.warn("Check connection failed.", e);
            }
        }
    }
}
