package org.bidib.jbidibc.decoder.json;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;

import org.apache.commons.collections4.IterableUtils;
import org.apache.commons.collections4.Predicate;
import org.assertj.core.api.Assertions;
import org.bidib.jbidibc.decoder.decoderdb.DecoderFirmwareDefinitionAccess;
import org.bidib.jbidibc.decoder.schema.manufacturers.ManufacturerType;
import org.bidib.jbidibc.decoder.schema.manufacturers.ManufacturersList;
import org.bidib.jbidibc.decoder.schema.manufacturers.ManufacturersType;
import org.junit.jupiter.api.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;

public class ManufacturerAccessTest {

    private static final Logger LOGGER = LoggerFactory.getLogger(ManufacturerAccessTest.class);

    @Test
    public void fetchJson() {
        LOGGER.info("Fetch manufacturers from DecoderDB.");

        DecoderFirmwareDefinitionAccess manufacturerAccess = new DecoderFirmwareDefinitionAccess();
        ManufacturersList listManufacturers = manufacturerAccess.fetchManufacturers("cv", "cv");

        LOGGER.info("Retrieved listManufacturers: {}", listManufacturers);

        Assertions.assertThat(listManufacturers).isNotNull();

        Assertions.assertThat(listManufacturers.getManufacturers()).isNotNull();
        Assertions.assertThat(listManufacturers.getManufacturers().getManufacturer()).isNotNull();
        Assertions.assertThat(listManufacturers.getManufacturers().getManufacturer()).isNotEmpty();

        // find the DIY manufacturer
        final short ID = 13;

        List<ManufacturerType> manufacturers = listManufacturers.getManufacturers().getManufacturer();
        ManufacturerType manufacturerDiy = IterableUtils.find(manufacturers, new Predicate<ManufacturerType>() {

            @Override
            public boolean evaluate(ManufacturerType manufacturer) {
                return manufacturer.getId() == ID;
            }
        });

        Assertions.assertThat(manufacturerDiy).isNotNull();
        LOGGER.info("Found DIY manufacturer: {}", manufacturerDiy);

        // find the OpenDCC manufacturer
        final Integer EXTENDED_ID_OPENDCC = Integer.valueOf(258);

        ManufacturerType manufacturerOpenDcc = IterableUtils.find(manufacturers, new Predicate<ManufacturerType>() {

            @Override
            public boolean evaluate(ManufacturerType manufacturer) {
                return manufacturer.getId() == ID && EXTENDED_ID_OPENDCC.equals(manufacturer.getExtendedId());
            }
        });

        Assertions.assertThat(manufacturerOpenDcc).isNotNull();
        LOGGER.info("Found OpenDCC manufacturer: {}", manufacturerOpenDcc);
    }

    @Test
    public void loadManufacturersJsonLocal() throws JsonParseException, JsonMappingException, IOException {

        final ObjectMapper mapper =
            new ObjectMapper().registerModule(new Jdk8Module()).registerModule(new JavaTimeModule());
        SimpleModule module = new SimpleModule();
        module.addDeserializer(ManufacturersType.class, new ManufacturersTypeDeserializer());
        mapper.registerModule(module);

        InputStream is = ManufacturerAccessTest.class.getResourceAsStream("/manufacturers/Manufacturers.json");
        Assertions.assertThat(is).isNotNull();

        ManufacturersList listAll = mapper.reader().forType(ManufacturersList.class).readValue(is);
        LOGGER.info("Loaded Manufacturers.json: {}", listAll);

        Assertions.assertThat(listAll).isNotNull();
        Assertions.assertThat(listAll.getManufacturers()).isNotNull();
        Assertions.assertThat(listAll.getManufacturers().getManufacturer()).hasSizeGreaterThan(10);
    }
}
