package org.bidib.jbidibc.decoder.decoderdb;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.List;

import org.apache.commons.collections4.IterableUtils;
import org.apache.commons.collections4.Predicate;
import org.apache.commons.io.IOUtils;
import org.assertj.core.api.Assertions;
import org.bidib.jbidibc.decoder.json.DecoderDbStatusResponseTest;
import org.bidib.jbidibc.decoder.schema.commontypes.ProtocolTypeType;
import org.bidib.jbidibc.decoder.schema.decoderdetection.DecoderDetection;
import org.bidib.jbidibc.decoder.schema.decoderdetection.DecoderDetectionProtocolType;
import org.bidib.jbidibc.decoder.schema.decoderdetection.ManufacturerType;
import org.bidib.jbidibc.decoderdetection.DecoderDetectionFactory;
import org.junit.jupiter.api.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DecoderDetectionAccessTest {

    private static final Logger LOGGER = LoggerFactory.getLogger(DecoderDetectionAccessTest.class);

    @Test
    // @Disabled
    public void fetch() {
        LOGGER.info("Fetch decoderDetection from DecoderDB.");

        DecoderDetectionAccess decoderDetectionAccess = new DecoderDetectionAccess();
        DecoderDetection decoderDetection = decoderDetectionAccess.fetch("cv", "cv".toCharArray());

        LOGGER.info("Retrieved decoderDetection: {}", decoderDetection);

        Assertions.assertThat(decoderDetection).isNotNull();

        Assertions.assertThat(decoderDetection.getProtocols()).isNotNull();
        Assertions.assertThat(decoderDetection.getProtocols().getProtocol()).isNotNull();
        Assertions.assertThat(decoderDetection.getProtocols().getProtocol()).isNotEmpty();

        // find DCC protocol

        List<DecoderDetectionProtocolType> protocols = decoderDetection.getProtocols().getProtocol();
        DecoderDetectionProtocolType dccProtocol =
            IterableUtils.find(protocols, new Predicate<DecoderDetectionProtocolType>() {

                @Override
                public boolean evaluate(DecoderDetectionProtocolType protocol) {
                    return ProtocolTypeType.DCC.equals(protocol.getType());
                }
            });

        Assertions.assertThat(dccProtocol).isNotNull();

        // find the DIY manufacturer
        final short ID = 13;

        List<ManufacturerType> manufacturers = dccProtocol.getManufacturer();
        ManufacturerType manufacturerDiy = IterableUtils.find(manufacturers, new Predicate<ManufacturerType>() {

            @Override
            public boolean evaluate(ManufacturerType manufacturer) {
                return manufacturer.getId() == ID;
            }
        });

        Assertions.assertThat(manufacturerDiy).isNotNull();
        LOGGER.info("Found DIY manufacturer: {}", manufacturerDiy);

        // find the OpenCar manufacturer
        final Integer EXTENDED_ID_OPENCAR = Integer.valueOf(257);

        ManufacturerType manufacturerOpenDcc = IterableUtils.find(manufacturers, new Predicate<ManufacturerType>() {

            @Override
            public boolean evaluate(ManufacturerType manufacturer) {
                return manufacturer.getId() == ID && EXTENDED_ID_OPENCAR.equals(manufacturer.getExtendedId());
            }
        });

        Assertions.assertThat(manufacturerOpenDcc).isNotNull();
        LOGGER.info("Found OpenCar manufacturer: {}", manufacturerOpenDcc);
    }

    @Test
    public void fetchLocal() throws IOException {

        InputStream is =
            DecoderDbStatusResponseTest.class.getResourceAsStream("/decoderDetection/DecoderDetection.decdb");
        Assertions.assertThat(is).isNotNull();

        String content = IOUtils.toString(is, StandardCharsets.UTF_8);
        DecoderDetection decoderDetection = new DecoderDetectionFactory().parseDecoderDetection(content);

        Assertions.assertThat(decoderDetection).isNotNull();

        Assertions.assertThat(decoderDetection.getProtocols()).isNotNull();
        Assertions.assertThat(decoderDetection.getProtocols().getProtocol()).isNotNull();
        Assertions.assertThat(decoderDetection.getProtocols().getProtocol()).isNotEmpty();

        // find DCC protocol

        List<DecoderDetectionProtocolType> protocols = decoderDetection.getProtocols().getProtocol();
        DecoderDetectionProtocolType dccProtocol =
            IterableUtils.find(protocols, new Predicate<DecoderDetectionProtocolType>() {

                @Override
                public boolean evaluate(DecoderDetectionProtocolType protocol) {
                    return ProtocolTypeType.DCC.equals(protocol.getType());
                }
            });

        Assertions.assertThat(dccProtocol).isNotNull();

        // find the DIY manufacturer
        final short ID = 13;

        List<ManufacturerType> manufacturers = dccProtocol.getManufacturer();
        ManufacturerType manufacturerDiy = IterableUtils.find(manufacturers, new Predicate<ManufacturerType>() {

            @Override
            public boolean evaluate(ManufacturerType manufacturer) {
                return manufacturer.getId() == ID;
            }
        });

        Assertions.assertThat(manufacturerDiy).isNotNull();
        LOGGER.info("Found DIY manufacturer: {}", manufacturerDiy);

        // find the OpenCar manufacturer
        final Integer EXTENDED_ID_OPENCAR = Integer.valueOf(257);

        ManufacturerType manufacturerOpenDcc = IterableUtils.find(manufacturers, new Predicate<ManufacturerType>() {

            @Override
            public boolean evaluate(ManufacturerType manufacturer) {
                return manufacturer.getId() == ID && EXTENDED_ID_OPENCAR.equals(manufacturer.getExtendedId());
            }
        });

        Assertions.assertThat(manufacturerOpenDcc).isNotNull();
        LOGGER.info("Found OpenCar manufacturer: {}", manufacturerOpenDcc);
    }
}
