package org.bidib.jbidibc.decoder.json;

import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.InetSocketAddress;
import java.net.MalformedURLException;
import java.net.Proxy;
import java.net.URISyntaxException;
import java.net.URL;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.Month;
import java.util.List;
import java.util.Properties;

import org.apache.commons.collections4.IterableUtils;
import org.apache.commons.collections4.Predicate;
import org.assertj.core.api.Assertions;
import org.bidib.jbidibc.decoder.decoderdb.AbstractDecoderDbAccess;
import org.bidib.jbidibc.decoder.decoderdb.AbstractDecoderDbAccess.DataFormat;
import org.bidib.jbidibc.decoder.decoderdb.ProxyUtils;
import org.bidib.jbidibc.decoder.json.DecoderDbStatusResponse.DecoderDetectionsStatusResponse;
import org.bidib.jbidibc.decoder.json.DecoderDbStatusResponse.DecoderStatusResponse;
import org.bidib.jbidibc.decoder.json.DecoderDbStatusResponse.FirmwareStatusResponse;
import org.bidib.jbidibc.decoder.json.DecoderDbStatusResponse.ManufacturersStatusResponse;
import org.bidib.jbidibc.decoder.schema.decoderdetection.DecoderDetection;
import org.bidib.jbidibc.decoder.schema.decoderdetection.DecoderDetectionProtocolType;
import org.bidib.jbidibc.decoder.schema.decoderdetection.ProtocolsType;
import org.bidib.jbidibc.decoder.schema.manufacturers.ManufacturerType;
import org.bidib.jbidibc.decoder.schema.manufacturers.ManufacturersList;
import org.bidib.jbidibc.decoderdetection.DecoderDetectionFactory;
import org.junit.jupiter.api.MethodOrderer;
import org.junit.jupiter.api.Order;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestMethodOrder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;

@TestMethodOrder(MethodOrderer.OrderAnnotation.class)
public class DecoderDbStatusResponseTest {

    private static final Logger LOGGER = LoggerFactory.getLogger(DecoderDbStatusResponseTest.class);

    // private static final String REST_SERVICE_URI = "https://www.decoderdb.de";

    private static final String REST_SERVICE_URI = "https://www.fichtelbahn.de/files/decoderDB";

    @Test
    @Order(1)
    public void loadRepositoryJsonLocal() throws JsonParseException, JsonMappingException, IOException {

        // final ObjectMapper mapper = new ObjectMapper(); // can reuse, share globally
        // final JakartaXmlBindAnnotationModule module = new JakartaXmlBindAnnotationModule();
        // // configure as necessary
        // mapper.registerModule(module);
        final ObjectMapper mapper =
            new ObjectMapper().registerModule(new Jdk8Module()).registerModule(new JavaTimeModule());

        InputStream is = DecoderDbStatusResponseTest.class.getResourceAsStream("/json/repository.json");
        Assertions.assertThat(is).isNotNull();

        DecoderDbStatusResponse listAll = mapper.readValue(is, DecoderDbStatusResponse.class);
        LOGGER.info("Loaded repository.json: {}", listAll);

        Assertions.assertThat(listAll).isNotNull();
        Assertions.assertThat(listAll.getManufacturers()).isNotNull();
        Assertions.assertThat(listAll.getManufacturers().getFilename()).isEqualTo("Manufacturers.decdb");
        LocalDate cal = LocalDate.of(2021, Month.MARCH, 6);
        Assertions.assertThat(listAll.getManufacturers().getNmraListDate()).isEqualTo(cal);

        Assertions.assertThat(listAll.getDecoderDetections()).isNotNull();
        Assertions.assertThat(listAll.getDecoderDetections().getFilename()).isEqualTo("DecoderDetection.decdb");
        Assertions
            .assertThat(listAll.getDecoderDetections().getLink())
            .isEqualTo("https://www.fichtelbahn.de/files/decoderDB/DecoderDetection.decdb");

        Assertions.assertThat(listAll.getDecoder()).isNotNull();
        Assertions.assertThat(listAll.getDecoder().length).isGreaterThan(220);

        DecoderStatusResponse product = listAll.getDecoder()[0];
        Assertions.assertThat(product).isNotNull();
        Assertions.assertThat(product.getFilename()).isEqualTo("Decoder_0_NMRA-Standard.decdb");
        Assertions.assertThat(product.getManufacturerId()).isEqualTo(Integer.valueOf(0));
        Assertions.assertThat(product.getManufacturerExtendedId()).isEqualTo(0);
        Assertions.assertThat(product.getName()).isEqualTo("NMRA-Standard");

        product = listAll.getDecoder()[1];
        Assertions.assertThat(product).isNotNull();
        Assertions.assertThat(product.getFilename()).isEqualTo("Decoder_0_StandardSUSI.decdb");
        Assertions.assertThat(product.getManufacturerId()).isEqualTo(0);
        Assertions.assertThat(product.getManufacturerExtendedId()).isEqualTo(0);
        Assertions.assertThat(product.getName()).isEqualTo("Standard SUSI");
    }

    @Test
    // @Disabled
    @Order(2)
    public void listManufacturersRemote() throws MalformedURLException, URISyntaxException {
        LOGGER.info("listManufacturersRemote");

        ManufacturersList listManufacturers =
            AbstractDecoderDbAccess
                .fetch("cv", "cv".toCharArray(), REST_SERVICE_URI + "/Manufacturers.decdb", ManufacturersList.class,
                    DataFormat.xml);

        LOGGER.info("Retrieved listManufacturers: {}", listManufacturers);

        Assertions.assertThat(listManufacturers).isNotNull();

        Assertions.assertThat(listManufacturers.getManufacturers()).isNotNull();
        Assertions.assertThat(listManufacturers.getManufacturers().getManufacturer()).isNotNull();
        Assertions.assertThat(listManufacturers.getManufacturers().getManufacturer()).isNotEmpty();

        // find the DIY manufacturer
        final short ID = 13;

        List<ManufacturerType> manufacturers = listManufacturers.getManufacturers().getManufacturer();
        ManufacturerType manufacturerDiy = IterableUtils.find(manufacturers, new Predicate<ManufacturerType>() {

            @Override
            public boolean evaluate(ManufacturerType manufacturer) {
                return manufacturer.getId() == ID;
            }
        });

        Assertions.assertThat(manufacturerDiy).isNotNull();
        LOGGER.info("Found DIY manufacturer: {}", manufacturerDiy);

        // find the OpenDCC manufacturer
        final Integer EXTENDED_ID_OPENDCC = Integer.valueOf(258);

        ManufacturerType manufacturerOpenDcc = IterableUtils.find(manufacturers, new Predicate<ManufacturerType>() {

            @Override
            public boolean evaluate(ManufacturerType manufacturer) {
                return manufacturer.getId() == ID && EXTENDED_ID_OPENDCC.equals(manufacturer.getExtendedId());
            }
        });

        Assertions.assertThat(manufacturerOpenDcc).isNotNull();
        LOGGER.info("Found OpenDCC manufacturer: {}", manufacturerOpenDcc);
    }

    @Test
    // @Disabled
    @Order(2)
    public void loadRepositoryJsonRemote() throws MalformedURLException, URISyntaxException {
        LOGGER.info("loadRepositoryJsonRemote");

        DecoderDbStatusResponse repository =
            AbstractDecoderDbAccess
                .fetch("cv", "cv".toCharArray(), REST_SERVICE_URI + "/repository.json", DecoderDbStatusResponse.class,
                    DataFormat.json);

        LOGGER.info("Retrieved listAll: {}", repository);

        // get the manufacturers
        ManufacturersStatusResponse manufacturersStatusResponse = repository.getManufacturers();
        Assertions.assertThat(manufacturersStatusResponse).isNotNull();

        LocalDate nmraListDate = manufacturersStatusResponse.getNmraListDate();
        Assertions.assertThat(nmraListDate).isNotNull();
        LocalDateTime lastUpdate = manufacturersStatusResponse.getLastUpdate();
        Assertions.assertThat(lastUpdate).isNotNull();

        // get the decoderDetection
        DecoderDetectionsStatusResponse decoderDetections = repository.getDecoderDetections();
        Assertions.assertThat(decoderDetections).isNotNull();
        Assertions.assertThat(decoderDetections.getLink()).isNotNull();
        Assertions
            .assertThat(decoderDetections.getLink())
            .isEqualTo("https://www.fichtelbahn.de/files/decoderDB/DecoderDetection.decdb");
        Assertions.assertThat(decoderDetections.getFilename()).isNotNull();
        Assertions.assertThat(decoderDetections.getFilename()).isEqualTo("DecoderDetection.decdb");

        // get the decoders
        DecoderStatusResponse[] decoders = repository.getDecoder();
        Assertions.assertThat(decoders).isNotNull();
        Assertions.assertThat(decoders.length).isGreaterThan(220);
        // Assertions.assertThat(decoders.length > 0).isTrue();

        LOGGER.info("Current decoders: {}", new Object[] { decoders });

        // get the firmware items
        FirmwareStatusResponse[] firmwareItems = repository.getFirmware();
        Assertions.assertThat(firmwareItems).isNotNull();
        Assertions.assertThat(firmwareItems.length).isGreaterThanOrEqualTo(180);

        LOGGER.info("Current firmwareItems: {}", new Object[] { firmwareItems });

        // get the manufacturers
        String link = manufacturersStatusResponse.getLink();
        String filename = manufacturersStatusResponse.getFilename();
        LOGGER
            .info("Current nmraListDate: {}, lastUpdate: {}, filename: {}, link: {}", nmraListDate, lastUpdate,
                filename, link);

        // ManufacturersList manufacturersList =
        // AbstractDecoderDbAccess.fetch("cv", "cv".toCharArray(), link, ManufacturersList.class, DataFormat.xml);
        //
        // Assertions.assertThat(manufacturersList).isNotNull();
        //
        // LOGGER.info("Current manufacturersList: {}", manufacturersList);
        //
        // Assertions.assertThat(manufacturersList.getManufacturers()).isNotNull();
        // ManufacturersType manufacturersType = manufacturersList.getManufacturers();
        // List<ManufacturerType> manufacturers = manufacturersType.getManufacturer();
        //
        // Assertions.assertThat(manufacturers).hasSizeGreaterThanOrEqualTo(175);

        // the decoderDetection needs a special mapper
        link = decoderDetections.getLink();
        DecoderDetection decoderDetection = AbstractDecoderDbAccess.fetch("cv", "cv".toCharArray(), link, val -> {
            return new DecoderDetectionFactory().parseDecoderDetection(val);
        });

        Assertions.assertThat(decoderDetection).isNotNull();

        LOGGER.info("Current decoderDetection: {}", decoderDetection);

        Assertions.assertThat(decoderDetection.getProtocols()).isNotNull();
        ProtocolsType protocolsType = decoderDetection.getProtocols();
        List<DecoderDetectionProtocolType> protocols = protocolsType.getProtocol();

        Assertions.assertThat(protocols).hasSizeGreaterThanOrEqualTo(3);

    }

    @Test
    public void proxyTest() throws URISyntaxException, MalformedURLException {

        URL url = new URL("http://www.bidib.org");

        Proxy proxy = ProxyUtils.findProxy(url.toURI());

        LOGGER.info("Current proxy: {}", proxy);

        if (!Proxy.NO_PROXY.equals(proxy)) {
            try {
                InetSocketAddress addr = (InetSocketAddress) proxy.address();
                Properties systemSettings = System.getProperties();
                systemSettings.put("proxySet", "true");
                systemSettings.put("http.proxyHost", addr.getHostName());
                systemSettings.put("http.proxyPort", Integer.toString(addr.getPort()));
                HttpURLConnection con = (HttpURLConnection) url.openConnection();
                LOGGER.info(con.getResponseCode() + " : " + con.getResponseMessage());
                LOGGER.info("HTTP_OK: {} ", con.getResponseCode() == HttpURLConnection.HTTP_OK);
            }
            catch (Exception e) {
                LOGGER.warn("Check connection failed.", e);
            }
        }
    }
}
