package org.bidib.jbidibc.gateway;

import java.io.IOException;
import java.net.InetAddress;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import org.bidib.jbidibc.core.AbstractBidib;
import org.bidib.jbidibc.core.BidibInterface;
import org.bidib.jbidibc.core.BidibMessageProcessor;
import org.bidib.jbidibc.core.ConnectionListener;
import org.bidib.jbidibc.core.MessageListener;
import org.bidib.jbidibc.core.MessageReceiver;
import org.bidib.jbidibc.core.NodeListener;
import org.bidib.jbidibc.core.exception.PortNotFoundException;
import org.bidib.jbidibc.core.exception.PortNotOpenedException;
import org.bidib.jbidibc.core.helpers.Context;
import org.bidib.jbidibc.core.node.NodeRegistry;
import org.bidib.jbidibc.core.node.listener.TransferListener;
import org.bidib.jbidibc.core.utils.ByteUtils;
import org.bidib.jbidibc.net.NetBidib;
import org.bidib.jbidibc.net.NetBidibPort;
import org.bidib.jbidibc.net.NetBidibServerTcpPort;
import org.bidib.jbidibc.net.NetBidibUdpPort;
import org.bidib.jbidibc.net.NetMessageHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GatewayBidib extends AbstractBidib {

    private static final Logger LOGGER = LoggerFactory.getLogger(GatewayBidib.class);

    private static GatewayBidib instance;

    private NetBidibPort port;

    private Thread portWorker;

    private NetMessageHandler netMessageHandler;

    private String connectedPortName;

    private InetAddress address;

    private int portNumber;

    private String protocol;

    public static synchronized BidibInterface getInstance() {
        if (instance == null) {
            instance = new GatewayBidib();
            instance.initialize();
        }
        return instance;
    }

    @Override
    protected BidibMessageProcessor createMessageReceiver(NodeRegistry nodeFactory) {
        return new GatewayMessageReceiver(nodeFactory);
    }

    private MessageReceiver getNetMessageReceiver() {
        return (MessageReceiver) getMessageReceiver();
    }

    @Override
    public void open(
        String portName, ConnectionListener connectionListener, Set<NodeListener> nodeListeners,
        Set<MessageListener> messageListeners, Set<TransferListener> transferListeners, Context context)
        throws PortNotFoundException, PortNotOpenedException {

        LOGGER.info("Open port: {}", portName);

        setConnectionListener(connectionListener);

        // register the listeners
        registerListeners(nodeListeners, messageListeners, transferListeners);

        if (port == null) {
            LOGGER.info("Open port with name: {}", portName);
            if (portName == null || portName.trim().isEmpty()) {
                throw new PortNotFoundException("");
            }

            if (portName.indexOf(":") < 0) {
                portName += ":" + NetBidib.BIDIB_UDP_PORT_NUMBER;
                LOGGER.info("Added portnumber to portName: {}", portName);
            }

            try {
                close();
                port = internalOpen(portName, context);
                connectedPortName = portName;

                LOGGER.info("Port is opened, send the magic. The connected port is: {}", connectedPortName);
                // sendMagic();
                getRootNode();
            }
            catch (Exception ex) {
                LOGGER.warn("Open port and send magic failed.", ex);

                throw new PortNotOpenedException(portName, PortNotOpenedException.UNKNOWN);
            }
        }
        else {
            LOGGER.warn("Port is already opened.");
        }
    }

    private NetBidibPort internalOpen(String portName, final Context context) throws IOException {
        LOGGER.info("Internal open port: {}", portName);

        String[] hostAndPort = portName.split(":");

        if (hostAndPort.length > 2) {
            // protocol provided
            protocol = hostAndPort[0];
            address = InetAddress.getByName(hostAndPort[1]);
            portNumber = Integer.parseInt(hostAndPort[2]);
        }
        else {
            protocol = "tcp";
            address = InetAddress.getByName(hostAndPort[0]);
            portNumber = Integer.parseInt(hostAndPort[1]);
        }

        LOGGER.info("Configured address: {}, portNumber: {}, protocol: {}", address, portNumber, protocol);

        final BidibMessageProcessor messageReceiver = getMessageReceiver();

        if (context != null) {
            Boolean ignoreWrongMessageNumber = context.get("ignoreWrongMessageNumber", Boolean.class, Boolean.FALSE);
            messageReceiver.setIgnoreWrongMessageNumber(ignoreWrongMessageNumber);
        }

        // enable the message receiver before the event listener is added
        getNetMessageReceiver().enable();

        netMessageHandler = new GatewayNetMessageHandler(messageReceiver);

        // open the port
        NetBidibPort netBidibPort = null;

        if ("tcp".equalsIgnoreCase(protocol)) {
            LOGGER.info("Create a NetBidibTcpPort with the portnumber: {}", NetBidib.BIDIB_UDP_PORT_NUMBER);
            netBidibPort = new NetBidibServerTcpPort(NetBidib.BIDIB_UDP_PORT_NUMBER, null, netMessageHandler);
        }
        else {
            LOGGER.info("Create a NetBidibUdpPort with the portnumber: {}", NetBidib.BIDIB_UDP_PORT_NUMBER);
            netBidibPort = new NetBidibUdpPort(NetBidib.BIDIB_UDP_PORT_NUMBER, netMessageHandler);
        }
        LOGGER.info("Prepare and start the port worker for netBidibPort: {}", netBidibPort);

        startReceiverAndQueues(messageReceiver, context);

        portWorker = new Thread(netBidibPort);
        portWorker.start();

        return netBidibPort;
    }

    @Override
    public boolean isOpened() {
        return port != null;
    }

    @Override
    public void close() {
        LOGGER.info("Close the port.");

        if (port != null) {
            LOGGER.info("Stop the port.");
            port.stop();

            if (portWorker != null) {
                synchronized (portWorker) {
                    try {
                        portWorker.join(5000L);
                    }
                    catch (InterruptedException ex) {
                        LOGGER.warn("Wait for termination of port worker failed.", ex);
                    }
                    portWorker = null;
                }
            }

            port = null;
        }

        stopReceiverAndQueues(null);

        if (getConnectionListener() != null) {
            getConnectionListener().closed(connectedPortName);
        }

        // clear the connectedPortName
        connectedPortName = null;

        LOGGER.info("Close the port finished.");
    }

    @Override
    public List<String> getPortIdentifiers() {
        return Collections.emptyList();
    }

    @Override
    protected void sendData(byte[] data) {
        // TODO Auto-generated method stub
        LOGGER.info("Send data to host/client: {}, netMessageHandler: {}", ByteUtils.bytesToHex(data),
            netMessageHandler);
        netMessageHandler.send(port, data);
    }
}
