package org.bidib.jbidibc.net.serialovertcp;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.InetAddress;

import org.bidib.jbidibc.messages.ConnectionListener;
import org.bidib.jbidibc.messages.MessageReceiver;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultNetMessageHandler implements NetMessageHandler {
    private static final Logger LOGGER = LoggerFactory.getLogger(DefaultNetMessageHandler.class);

    private MessageReceiver messageReceiverDelegate;

    private BidibNetAddress remoteAddress;

    private final ConnectionListener connectionListener;

    /**
     * Creates a new instance of DefaultNetMessageHandler.
     * 
     * @param messageReceiverDelegate
     *            the delegate message receiver that processes the BiDiB messages
     * @param address
     *            the address of the master to connect to
     * @param port
     *            the port of the master to connect to
     * @param connectionListener
     *            the connection listener
     */
    public DefaultNetMessageHandler(MessageReceiver messageReceiverDelegate, InetAddress address, int port,
        final ConnectionListener connectionListener) {
        this.messageReceiverDelegate = messageReceiverDelegate;
        this.connectionListener = connectionListener;

        LOGGER.info("Set the remote address: {}, port: {}", address, port);
        remoteAddress = new BidibNetAddress(address, port);
    }

    @Override
    public void receive(final DataPacket packet) {
        // a data packet was received ... process the envelope and extract the message
        if (LOGGER.isTraceEnabled()) {
            LOGGER
                .trace("Received a packet from address: {}, port: {}, data: {}", packet.getAddress(), packet.getPort(),
                    ByteUtils.bytesToHex(packet.getData()));
        }

        // remove the UDP paket wrapper data and forward to the MessageReceiver
        ByteArrayOutputStream output = new ByteArrayOutputStream();
        output.write(packet.getData(), 0, packet.getData().length);

        LOGGER.debug("Forward received message to messageReceiverDelegate: {}", messageReceiverDelegate);
        try {
            messageReceiverDelegate.receive(output);
        }
        catch (Exception ex) {
            LOGGER.warn("Process messages failed.", ex);
            throw new RuntimeException(ex);
        }
    }

    @Override
    public void send(NetBidibPort port, byte[] bytes) {
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("Send message to port: {}, message: {}", port, ByteUtils.bytesToHex(bytes));
        }

        if (remoteAddress == null) {
            LOGGER.warn("### No remote addresses available. The message will not be sent!");
            return;
        }

        if (port != null) {

            try {
                LOGGER
                    .info("Send message to remote address, address: {}, port: {}", remoteAddress.getAddress(),
                        remoteAddress.getPortNumber());

                port.send(bytes, remoteAddress.getAddress(), remoteAddress.getPortNumber());
            }
            catch (IOException ex) {
                LOGGER.warn("Send message to port failed.", ex);
                throw new RuntimeException("Send message to datagram socket failed.", ex);
            }
        }
        else {
            LOGGER.warn("Send not possible, the port is closed.");
        }
    }

    @Override
    public void acceptClient(String remoteHost) {
        LOGGER.info("Accept client with address: {}", remoteHost);
    }

    @Override
    public void cleanup(String remoteHost) {
        LOGGER.info("Cleanup client with address: {}", remoteHost);

        connectionListener.closed(remoteHost);
    }

}
