package org.bidib.jbidibc.net.serialovertcp;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.ConnectException;
import java.net.InetAddress;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import org.bidib.jbidibc.core.AbstractBidib;
import org.bidib.jbidibc.core.BidibInterface;
import org.bidib.jbidibc.core.MessageListener;
import org.bidib.jbidibc.core.NodeListener;
import org.bidib.jbidibc.core.node.BidibNode;
import org.bidib.jbidibc.core.node.NodeRegistry;
import org.bidib.jbidibc.core.node.listener.TransferListener;
import org.bidib.jbidibc.messages.ConnectionListener;
import org.bidib.jbidibc.messages.base.AbstractBaseBidib;
import org.bidib.jbidibc.messages.base.RawMessageListener;
import org.bidib.jbidibc.messages.exception.InvalidConfigurationException;
import org.bidib.jbidibc.messages.exception.PortNotFoundException;
import org.bidib.jbidibc.messages.exception.PortNotOpenedException;
import org.bidib.jbidibc.messages.exception.PortNotOpenedException.FailureReason;
import org.bidib.jbidibc.messages.exception.ProtocolException;
import org.bidib.jbidibc.messages.helpers.Context;
import org.bidib.jbidibc.messages.message.BidibResponseFactory;
import org.bidib.jbidibc.messages.message.ResponseFactory;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.jbidibc.serial.SerialMessageEncoder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NetBidib extends AbstractBidib<NetMessageReceiver> {

    private static final Logger LOGGER = LoggerFactory.getLogger(NetBidib.class);

    public static final int BIDIB_UDP_PORT_NUMBER = 62875;

    private NetBidibPort port;

    private NetMessageHandler netMessageHandler;

    private Thread portWorker;

    private String connectedPortName;

    private InetAddress address;

    private int portNumber;

    private String protocol;

    private ConnectionListener connectionListener;

    private NetConnector connector;

    private ByteArrayOutputStream output = new ByteArrayOutputStream(100);

    private class NetConnector extends AbstractBaseBidib<NetMessageReceiver> {
        @Override
        protected void sendData(final ByteArrayOutputStream data, final RawMessageListener rawMessageListener) {

            if (port != null) {
                // byte[] bytes = data.toByteArray();
                LOGGER.info("Send message to net message handler: {}, port: {}", ByteUtils.bytesToHex(data), port);

                // forward the message to the netMessageReceiver
                try {

                    SerialMessageEncoder.encodeMessage(data, output);

                    LOGGER.info("Send, after encoding: {}", ByteUtils.bytesToHex(output));

                    if (rawMessageListener != null) {
                        rawMessageListener.notifySend(output.toByteArray());
                    }

                    netMessageHandler.send(port, output.toByteArray());
                }
                catch (Exception ex) {
                    LOGGER.warn("Forward message to send with netMessageReceiver failed.", ex);
                    throw new RuntimeException("Forward message to send with netMessageReceiver failed.", ex);
                }
                finally {
                    output.reset();
                }
            }
            else {
                LOGGER.warn("Send not possible, the port is closed.");
            }
        }

        @Override
        protected void internalOpen(String portName, Context context)
            throws PortNotFoundException, PortNotOpenedException {

            LOGGER.info("Internal open port: {}", portName);

            try {
                String[] hostAndPort = portName.split(":");
                if (hostAndPort.length > 2) {
                    // protocol provided
                    protocol = hostAndPort[0];
                    address = InetAddress.getByName(hostAndPort[1]);
                    portNumber = Integer.parseInt(hostAndPort[2]);
                }
                else {
                    protocol = "tcp";
                    address = InetAddress.getByName(hostAndPort[0]);
                    portNumber = Integer.parseInt(hostAndPort[1]);
                }
            }
            catch (Exception ex) {
                LOGGER.warn("Prepare address and portnumber for netBidibPort failed.", ex);
                throw new InvalidConfigurationException("Prepare address and portnumber for netBidibPort failed.");
            }

            LOGGER.info("Configured address: {}, portNumber: {}, protocol: {}", address, portNumber, protocol);

            final NetMessageReceiver messageReceiver = getMessageReceiver();

            // enable the message receiver before the event listener is added
            messageReceiver.enable();

            netMessageHandler = new DefaultNetMessageHandler(messageReceiver, address, portNumber, connectionListener);

            // open the port
            NetBidibPort netBidibPort = null;

            try {
                netBidibPort = new NetBidibPlainTcpPort(address, portNumber, netMessageHandler);
            }
            catch (IOException ex) {
                LOGGER.warn("Configure NetBidibPlainTcpPort failed.", ex);
                throw new InvalidConfigurationException("Configure NetBidibPlainTcpPort failed.");
            }

            LOGGER.info("Prepare and start the port worker for netBidibPort: {}", netBidibPort);

            // startReceiverAndQueues(getBidibMessageProcessor(), context);

            portWorker = new Thread(netBidibPort);
            portWorker.start();

            port = netBidibPort;

        }
    }

    protected NetBidib() {
        LOGGER.info("Create new instance of plain tcp NetBidib.");
    }

    @Override
    protected NetMessageReceiver createMessageReceiver(
        final NodeRegistry nodeFactory, final RawMessageListener rawMessageListener, final Context context) {

        final ResponseFactory responseFactory = new BidibResponseFactory();
        responseFactory.initialize();

        // the serial over tcp needs CRC check
        NetMessageReceiver messageReceiver = new NetMessageReceiver(nodeFactory, responseFactory, true);
        messageReceiver.setRawMessageListener(rawMessageListener);
        messageReceiver.init(context);
        return messageReceiver;
    }

    @Override
    public void setConnectionListener(final ConnectionListener connectionListener) {

        this.connectionListener = new ConnectionListener() {

            @Override
            public void status(String messageKey, final Context context) {
                LOGGER.info("The status was signalled: {}, context: {}", messageKey, context);
                connectionListener.status(messageKey, context);
            }

            @Override
            public void opened(String port) {
                LOGGER.info("The port was opened: {}", port);
                connectionListener.opened(port);
            }

            @Override
            public void closed(String port) {
                LOGGER.info("The port was closed: {}", port);

                connectionListener.closed(port);
            }

            @Override
            public void stall(boolean stall) {
                connectionListener.stall(stall);
            }
        };

        super.setConnectionListener(this.connectionListener);
    }

    /**
     * Get a new initialized instance of NetBidib.
     *
     * @return the instance of NetBidib
     */
    public static BidibInterface createInstance(final Context context) {
        LOGGER.info("Create new instance of NetBidib.");

        NetBidib instance = new NetBidib();
        instance.initialize(context);

        return instance;
    }

    @Override
    public void initialize(final Context context) {
        LOGGER.info("Initialize. Create the connector.");
        super.initialize(context);

        connector = new NetConnector();

        final NetMessageReceiver messageReceiver = getMessageReceiver();
        connector.setMessageReceiver(messageReceiver);

        initializeConnector(connector);
    }

    @Override
    public void open(
        String portName, ConnectionListener connectionListener, Set<NodeListener> nodeListeners,
        Set<MessageListener> messageListeners, Set<TransferListener> transferListeners, final Context context)
        throws PortNotFoundException, PortNotOpenedException {

        LOGGER.info("Open port: {}", portName);

        setConnectionListener(connectionListener);

        // register the listeners
        registerListeners(nodeListeners, messageListeners, transferListeners);

        if (port == null) {
            LOGGER.info("Open port with name: {}", portName);
            if (portName == null || portName.trim().isEmpty()) {
                throw new PortNotFoundException("");
            }

            if (portName.indexOf(":") < 0) {
                portName += ":" + NetBidib.BIDIB_UDP_PORT_NUMBER;
                LOGGER.info("Added portnumber to portName: {}", portName);
            }

            try {
                // close();
                port = internalOpen(portName, context);
                connectedPortName = portName;

                LOGGER.info("Port is opened, send the magic. The connected port is: {}", connectedPortName);
                sendMagic();

                LOGGER.info("Startup sequence is finished. Notify the communiction that we are finished.");
                // notify the communication
                getConnectionListener().opened(portName);

            }
            catch (ConnectException ex) {
                LOGGER.warn("Open port failed because connect failed.", ex);

                throw new PortNotOpenedException(portName, PortNotOpenedException.CONNECT_FAILED)
                    .withFailureReason(FailureReason.CONNECT_FAILED);
            }
            catch (Exception ex) {
                LOGGER.warn("Open port and send magic failed.", ex);

                throw new PortNotOpenedException(portName, PortNotOpenedException.UNKNOWN)
                    .withFailureReason(FailureReason.UNKNOWN);
            }
            LOGGER.info("Open port passed: {}", portName);
        }
        else {
            LOGGER.warn("Port is already opened.");
        }
    }

    private NetBidibPort internalOpen(String portName, final Context context) throws IOException {
        LOGGER.info("Internal open port: {}", portName);

        String[] hostAndPort = portName.split(":");

        if (hostAndPort.length > 2) {
            // protocol provided
            protocol = hostAndPort[0];
            address = InetAddress.getByName(hostAndPort[1]);
            portNumber = Integer.parseInt(hostAndPort[2]);
        }
        else {
            protocol = "tcp";
            address = InetAddress.getByName(hostAndPort[0]);
            portNumber = Integer.parseInt(hostAndPort[1]);
        }

        LOGGER.info("Configured address: {}, portNumber: {}, protocol: {}", address, portNumber, protocol);

        final NetMessageReceiver messageReceiver = getMessageReceiver();

        // enable the message receiver before the event listener is added
        messageReceiver.enable();

        netMessageHandler = new DefaultNetMessageHandler(messageReceiver, address, portNumber, connectionListener);

        // open the port
        NetBidibPort netBidibPort = null;

        netBidibPort = new NetBidibPlainTcpPort(address, portNumber, netMessageHandler);

        LOGGER.info("Prepare and start the port worker for netBidibPort: {}", netBidibPort);

        connector.startReceiverAndQueues(getMessageReceiver(), context);

        portWorker = new Thread(netBidibPort);
        portWorker.start();

        return netBidibPort;
    }

    @Override
    public boolean isOpened() {
        return port != null;
    }

    @Override
    public void close() {
        LOGGER.info("Close the port.");

        if (port != null) {
            LOGGER.info("Stop the port.");

            final NetBidibPort portToClose = this.port;
            this.port = null;

            portToClose.stop();

            if (portWorker != null) {
                LOGGER.info("Wait for the port worker to finish.");
                synchronized (portWorker) {
                    try {
                        portWorker.join(5000L);
                    }
                    catch (InterruptedException ex) {
                        LOGGER.warn("Wait for termination of port worker failed.", ex);
                    }
                    portWorker = null;
                    LOGGER.info("The port worker has finished.");
                }
            }

            // port = null;
        }
        else {
            LOGGER.info("No port to close available.");
        }

        connector.stopReceiverAndQueues(null);

        if (connectedPortName != null) {

            String connectedPortNameToClose = this.connectedPortName;
            // clear the connectedPortName
            this.connectedPortName = null;
            fireConnectionClosed(connectedPortNameToClose);
        }
        else {
            LOGGER.info("The connectedPortName is not available.");
        }

        super.close();

        cleanupAfterClose(null);

        LOGGER.info("Close the port has finished.");
    }

    /**
     * Get the magic from the root node
     * 
     * @return the magic provided by the root node
     * @throws ProtocolException
     */
    private int sendMagic() throws ProtocolException {
        BidibNode rootNode = getRootNode();

        // Ignore the first exception ...
        int magic = -1;
        try {
            magic = rootNode.getMagic(15000);
        }
        catch (Exception e) {
            magic = rootNode.getMagic(15000);
        }
        LOGGER.info("The node returned magic: {}", magic);
        return magic;
    }

    @Override
    public void setResponseTimeout(int timeout) {
        LOGGER.info("Set the response timeout to: {}", timeout);
        super.setResponseTimeout(timeout);
    }

    @Override
    public List<String> getPortIdentifiers() {
        return Collections.emptyList();
    }

    @Override
    public void send(byte[] data) {
        connector.send(data);
    }

    @Override
    protected int contactInterface() {
        return 0;
    }
}
