package org.bidib.jbidibc.netbidib.server;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.SocketAddress;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Function;

import org.apache.commons.lang3.StringUtils;
import org.bidib.jbidibc.messages.BidibLibrary;
import org.bidib.jbidibc.messages.BidibMessagePublisher;
import org.bidib.jbidibc.messages.ConnectionListener;
import org.bidib.jbidibc.messages.HostAdapter;
import org.bidib.jbidibc.messages.Node;
import org.bidib.jbidibc.messages.StringData;
import org.bidib.jbidibc.messages.enums.PairingResult;
import org.bidib.jbidibc.messages.exception.InvalidConfigurationException;
import org.bidib.jbidibc.messages.exception.ProtocolException;
import org.bidib.jbidibc.messages.exception.ProtocolInvalidContentException;
import org.bidib.jbidibc.messages.helpers.Context;
import org.bidib.jbidibc.messages.helpers.DefaultContext;
import org.bidib.jbidibc.messages.message.BidibCommand;
import org.bidib.jbidibc.messages.message.BidibMessage;
import org.bidib.jbidibc.messages.message.BidibMessageInterface;
import org.bidib.jbidibc.messages.message.BidibRequestFactory;
import org.bidib.jbidibc.messages.message.BidibResponseFactory;
import org.bidib.jbidibc.messages.message.LocalBidibUpResponse;
import org.bidib.jbidibc.messages.message.LocalLogoffMessage;
import org.bidib.jbidibc.messages.message.LocalLogonAckMessage;
import org.bidib.jbidibc.messages.message.LocalLogonMessage;
import org.bidib.jbidibc.messages.message.StringResponse;
import org.bidib.jbidibc.messages.message.SysPVersionResponse;
import org.bidib.jbidibc.messages.message.SysUniqueIdResponse;
import org.bidib.jbidibc.messages.message.netbidib.LocalLinkMessage;
import org.bidib.jbidibc.messages.message.netbidib.LocalProtocolSignatureMessage;
import org.bidib.jbidibc.messages.message.netbidib.NetBidibCommandMessage;
import org.bidib.jbidibc.messages.message.netbidib.NetBidibLinkData;
import org.bidib.jbidibc.messages.message.netbidib.NetBidibLinkData.LogonStatus;
import org.bidib.jbidibc.messages.message.netbidib.NetBidibLinkData.PairingStatus;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.jbidibc.messages.utils.MessageUtils;
import org.bidib.jbidibc.messages.utils.NodeUtils;
import org.bidib.jbidibc.messages.utils.ThreadFactoryBuilder;
import org.bidib.jbidibc.netbidib.ConnectionUpdateEvent;
import org.bidib.jbidibc.netbidib.NetBidibContextKeys;
import org.bidib.jbidibc.netbidib.exception.AccessDeniedException;
import org.bidib.jbidibc.netbidib.exception.PairingDeniedException;
import org.bidib.jbidibc.netbidib.pairingstore.LocalPairingStore.PairingLookupResult;
import org.bidib.jbidibc.netbidib.pairingstore.PairingStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.SimpleChannelInboundHandler;
import io.netty.channel.group.ChannelGroup;

public abstract class NetBidibServerHandler<T> extends SimpleChannelInboundHandler<ByteBuf>
    implements BidibMessagePublisher<T> {

    private static final Logger LOGGER = LoggerFactory.getLogger(NetBidibServerHandler.class);

    // private static final Logger MSG_RX_LOGGER = LoggerFactory.getLogger("RX");

    // protected static final Logger MSG_TX_LOGGER = LoggerFactory.getLogger("TX");

    // private static final Logger MSG_RAW_LOGGER = LoggerFactory.getLogger("RAW");

    private static final Logger MSG_RX_NET_LOGGER = LoggerFactory.getLogger("RX_NET");

    private static final Logger MSG_TX_NET_LOGGER = LoggerFactory.getLogger("TX_NET");

    private BidibRequestFactory bidibRequestFactory;

    protected BidibResponseFactory responseFactory;

    private final HostAdapter<T> hostAdapter;

    private final String backendPortName;

    protected final Object pairedPartnerLock = new Object();

    /**
     * The link data of the paired remote partner.
     */
    protected NetBidibLinkData pairedPartner;

    /**
     * The link data of the server partner (ourself).
     */
    protected final NetBidibLinkData serverLinkData;

    private PairingStore pairingStore;

    /**
     * The channel group to pool all TCP connections. It is used to guaranty that no more than one connection is opened.
     */
    private ChannelGroup channelGroup;

    protected ChannelHandlerContext ctx;

    private Set<ConnectionListener> remoteConnectionListeners = new HashSet<>();

    private final Consumer<NetBidibServerHandler<T>> lazyInitializationCallback;

    private final Function<BidibMessageInterface, T> messageContentSupplier;

    private RoleTypeEnum roleType;

    private final org.bidib.jbidibc.messages.logger.Logger splitMessageLogger;

    private final AtomicBoolean isFirstPacket = new AtomicBoolean(true);

    /**
     * Constructor
     *
     * @param channelGroup
     *            the channel group to pool all TCP connections.
     * @param hostAdapter
     *            the host adapter
     * @param backendPortName
     *            the port identifier of the backend, e.g. COM8
     * @param serverLinkData
     *            the server link data instance to use
     */
    public NetBidibServerHandler(final ChannelGroup channelGroup, final HostAdapter<T> hostAdapter,
        final String backendPortName, final NetBidibLinkData serverLinkData,
        final Consumer<NetBidibServerHandler<T>> lazyInitializationCallback,
        Function<BidibMessageInterface, T> messageContentSupplier, final RoleTypeEnum roleType,
        final NetBidibLinkData pairedPartner) {

        LOGGER.info("Create new NetBidibServerHandler instance. Provided roleType: {}", roleType);

        this.channelGroup = channelGroup;
        this.hostAdapter = hostAdapter;
        this.backendPortName = backendPortName;

        this.serverLinkData = serverLinkData;
        this.lazyInitializationCallback = lazyInitializationCallback;
        this.messageContentSupplier = messageContentSupplier;

        this.roleType = roleType;
        this.pairedPartner = pairedPartner;

        splitMessageLogger = new org.bidib.jbidibc.messages.logger.Logger() {

            @Override
            public void debug(String format, Object... arguments) {
                LOGGER.debug(format, arguments);
            }

            @Override
            public void info(String format, Object... arguments) {
                LOGGER.info(format, arguments);
            }

            @Override
            public void warn(String format, Object... arguments) {
                LOGGER.warn(format, arguments);
            }

            @Override
            public void error(String format, Object... arguments) {
                LOGGER.error(format, arguments);
            }
        };
    }

    protected boolean handleLocalBidibUpResponse() {
        return this.roleType == RoleTypeEnum.INTERFACE;
    }

    public void addRemoteConnectionListener(final ConnectionListener remoteConnectionListener) {
        synchronized (this.remoteConnectionListeners) {

            this.remoteConnectionListeners.add(remoteConnectionListener);
        }
    }

    public void removeRemoteConnectionListener(final ConnectionListener remoteConnectionListener) {
        synchronized (this.remoteConnectionListeners) {

            this.remoteConnectionListeners.remove(remoteConnectionListener);
        }
    }

    public void setPairingStore(final PairingStore pairingStore) {
        this.pairingStore = pairingStore;
    }

    protected void performLazyInitialization() {
        LOGGER.info("Perform the lazy initialization.");

        if (this.bidibRequestFactory == null) {
            this.bidibRequestFactory = new BidibRequestFactory();
            this.bidibRequestFactory.setEscapeMagic(false);
            this.bidibRequestFactory.initialize();
        }

        if (this.responseFactory == null) {
            this.responseFactory = new BidibResponseFactory();
        }

        if (this.lazyInitializationCallback != null) {
            LOGGER.info("Call the lazy initialization callback.");
            // this will set the toHostPublisher
            this.lazyInitializationCallback.accept(this);
        }
    }

    @Override
    public void channelRegistered(ChannelHandlerContext ctx) throws Exception {

        final SocketAddress remoteAddress = ctx.channel().remoteAddress();
        LOGGER.info("Session created. IP: {}, channel: {}", remoteAddress, ctx.channel());

        // prevent multiple simultaneous connections
        if (hasActiveConnection()) {
            LOGGER.warn("More than one session: reject!");

            // send 'SHUTDOWN' message, then close the Channel
            ctx.channel().writeAndFlush(Unpooled.EMPTY_BUFFER).addListener(ChannelFutureListener.CLOSE);
            return;
        }

        MSG_RX_NET_LOGGER.info("Connected");

        performLazyInitialization();

        // add to channel group and set the state
        channelGroup.add(ctx.channel());
        ctx.channel().attr(ConnectionState.STATE_KEY).set(ConnectionState.Phase.NOT_CONNECTED);
        if (remoteAddress != null) {
            ctx.channel().attr(ConnectionState.REMOTE_ADDRESS_KEY).set(remoteAddress.toString());
        }
        else {
            final SocketAddress localAddress = ctx.channel().localAddress();
            LOGGER.info("No remoteAddress available. Use localAddress: {}", localAddress);
            // ctx.channel().attr(ConnectionState.REMOTE_ADDRESS_KEY).set(localAddress.toString());
        }
        super.channelRegistered(ctx);

        LOGGER.info("Store the channelHandlerContext: {}", ctx);
        this.ctx = ctx;

        synchronized (this.remoteConnectionListeners) {
            for (ConnectionListener connectionListener : this.remoteConnectionListeners) {
                try {
                    connectionListener
                        .opened(ctx.channel().remoteAddress() != null ? ctx.channel().remoteAddress().toString()
                            : "unknown");
                }
                catch (Exception ex) {
                    LOGGER.warn("Notify that the client connection was closed failed.", ex);
                }
            }
        }

    }

    private void sendInitialLocalProtocolSignatureMessage() throws ProtocolException {

        String requestorName = this.serverLinkData.getRequestorName();

        LOGGER
            .info("Send the initial LocalProtocolSignatureMessage to the client. Current requestorName: {}",
                requestorName);

        if (StringUtils.isBlank(requestorName)
            || !requestorName.startsWith(LocalProtocolSignatureMessage.EMITTER_PREFIX_BIDIB)) {
            LOGGER.warn("Invalid requestor name provided: {}", requestorName);

            throw new IllegalArgumentException("Invalid requestor name provided.");
        }

        // use BiDiB emitter
        NetBidibCommandMessage message =
            bidibRequestFactory
                .createLocalProtocolSignature(requestorName /* LocalProtocolSignatureMessage.EMITTER_PREFIX_BIDIB */);
        publishMessage(ctx, message);

        final Context context = new DefaultContext();
        context.register(NetBidibContextKeys.KEY_PORT, backendPortName);

        final ConnectionListener connectionListener = new ConnectionListener() {

            @Override
            public void opened(String port) {
                LOGGER.info("The port was opened: {}", port);

            }

            @Override
            public void closed(String port) {

            }

            @Override
            public void status(String messageKey, final Context context) {

            }

            @Override
            public void stall(boolean stall) {
                // TODO Auto-generated method stub
            }

            @Override
            public void pairingFinished(PairingResult pairingResult) {
                LOGGER.info("The pairing process has finished. Current pairingResult: {}", pairingResult);

                // TODO Auto-generated method stub

            }
        };
        context.register(NetBidibContextKeys.KEY_CONNECTION_LISTENER, connectionListener);

        try {
            LOGGER.info("Initialize the host adapter.");

            hostAdapter.initialize(context);

            LOGGER.info("Connect to the backend. Provided context: {}", context);
            // this call will connect to the backend
            hostAdapter.signalConnectionOpened(context);

            // // TODO wait until the root node is available
            //
            // // disable spontaneous events from backend
            // LOGGER.info("Disable all spontaneous events from backend.");
            // BidibCommand sysDisableMessage = bidibRequestFactory.createSysDisable();
            // sysDisableMessage.setAddr(Node.ROOTNODE_ADDR);
            // hostAdapter.forwardMessageToBackend(messageContentSupplier.apply(sysDisableMessage));

            // ask the backend for the names and uniqueId
            LOGGER.info("Get the productname, username, uniqueId and protocol version from the root node.");

            BidibCommand stringGetMessage =
                bidibRequestFactory.createStringGet(StringData.NAMESPACE_NODE, StringData.INDEX_PRODUCTNAME);
            stringGetMessage.setAddr(Node.ROOTNODE_ADDR);
            sendLocalBidibDownMessage(stringGetMessage);

            stringGetMessage =
                bidibRequestFactory.createStringGet(StringData.NAMESPACE_NODE, StringData.INDEX_USERNAME);
            stringGetMessage.setAddr(Node.ROOTNODE_ADDR);
            sendLocalBidibDownMessage(stringGetMessage);

            // get the unique id from the backend
            BidibCommand getUniqueIdMessage = bidibRequestFactory.createSysGetUniqueId();
            getUniqueIdMessage.setAddr(Node.ROOTNODE_ADDR);
            sendLocalBidibDownMessage(getUniqueIdMessage);

            // get the product version from the backend
            BidibCommand getPVersionMessage = bidibRequestFactory.createSysGetPVersion();
            getUniqueIdMessage.setAddr(Node.ROOTNODE_ADDR);
            sendLocalBidibDownMessage(getPVersionMessage);
        }
        catch (Exception ex) {
            LOGGER.warn("Connect to backend failed.", ex);

            throw new InvalidConfigurationException("Connect to backend failed.");
        }
    }

    private void sendLocalBidibDownMessage(final BidibCommand bidibCommand) {

        BidibCommand localBidibDownMessage = bidibRequestFactory.createLocalBidibDown(bidibCommand);
        localBidibDownMessage.setAddr(Node.ROOTNODE_ADDR);

        LOGGER.info("Prepared the localBidibDownMessage to forward to backend: {}", localBidibDownMessage);

        hostAdapter.forwardMessageToBackend(messageContentSupplier.apply(localBidibDownMessage));
    }

    @Override
    public void channelUnregistered(ChannelHandlerContext ctx) throws Exception {
        SocketAddress remoteAddress = ctx.channel().remoteAddress();
        LOGGER.info("Session closed. IP: {}", remoteAddress);

        if (remoteAddress == null) {
            final SocketAddress localAddress = ctx.channel().localAddress();
            LOGGER.info("No remoteAddress available. Use localAddress: {}", localAddress);

            remoteAddress = localAddress;
        }

        String connectedRemoteAddress = ctx.channel().attr(ConnectionState.REMOTE_ADDRESS_KEY).get();
        LOGGER.info("The connected remote address: {}", connectedRemoteAddress);

        // only close for primary connection
        if (hasActiveConnection() && !Objects.equals(remoteAddress.toString(), connectedRemoteAddress)) {
            LOGGER.info("The primary connection is still established.");
            return;
        }

        // set the state and remove from channel group (by super call)
        ctx.channel().attr(ConnectionState.STATE_KEY).set(ConnectionState.Phase.NOT_CONNECTED);
        super.channelUnregistered(ctx);

        final Context context = new DefaultContext();
        if (pairedPartner.getUniqueId() != null) {
            context.register(Context.UNIQUE_ID, pairedPartner.getUniqueId());
        }

        LOGGER.info("Signal that the connection to the guest was closed.");
        try {
            hostAdapter.signalConnectionClosed(context);
        }
        catch (Exception ex) {
            LOGGER.warn("Signal that the connection to the guest was closed failed.", ex);
        }

        try {
            LOGGER.info("Clear the saved serverLinkData.");
            serverLinkData.clear(false);
        }
        catch (Exception ex) {
            LOGGER.warn("Remove uniqueid from serverLinkData failed.", ex);
        }

        // release the pairedPartner
        synchronized (pairedPartnerLock) {
            LOGGER.info("Clear the paired partner link data: {}", pairedPartner);
            pairedPartner.clear(true);
        }

        LOGGER.info("Release the stored channelHandlerContext: {}", ctx);
        this.ctx = null;

        // only trigger the "no connection" event when no active connection exists
        if (!hasActiveConnection()) {
            ctx.fireUserEventTriggered(new ConnectionUpdateEvent(ConnectionState.NOT_CONNECTED));
        }

        synchronized (this.remoteConnectionListeners) {
            for (ConnectionListener connectionListener : this.remoteConnectionListeners) {
                try {
                    connectionListener
                        .closed(ctx.channel().remoteAddress() != null ? ctx.channel().remoteAddress().toString()
                            : "unknown");
                }
                catch (Exception ex) {
                    LOGGER.warn("Notify that the client connection was closed failed.", ex);
                }
            }
        }

        MSG_RX_NET_LOGGER.info("Disconnected");
    }

    @Override
    public void channelRead0(ChannelHandlerContext ctx, ByteBuf in) {

        // process the data that is received from the remote partner over netBidib

        byte[] messageContent = new byte[in.readableBytes()];
        in.readBytes(messageContent);

        LOGGER.info("<<net<<  : {}", ByteUtils.bytesToHex(messageContent));

        if (MSG_RX_NET_LOGGER.isInfoEnabled()) {
            MSG_RX_NET_LOGGER.info("<<netraw<< {}", ByteUtils.bytesToHex(messageContent));
        }

        try {

            // the message: LEN ADDR SEQ MSG_TYPE DATA
            int len = ByteUtils.getInt(messageContent[0]);
            LOGGER.info("Current message len: {}", len);

            final ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
            try {
                outputStream.write(messageContent);
            }
            catch (IOException ex2) {
                LOGGER.warn("Write data to output stream failed.", ex2);
            }

            // no crc check for netBidib messages
            MessageUtils.splitBidibMessages(splitMessageLogger, outputStream, false, messageArray -> processReceivedMessage(messageArray), isFirstPacket);
        }
        catch (ProtocolException ex) {
            LOGGER.warn("Create messages from provided data failed.", ex);
        }
    }

    private void processReceivedMessage(byte[] messageArray) throws ProtocolException {

        BidibMessage message = null;

        try {
            // let the request factory create the commands
            List<BidibMessageInterface> commands = bidibRequestFactory.create(messageArray);
            LOGGER.info("Received commands: {}", commands);

            for (BidibMessageInterface bidibCommand : commands) {

                if (MSG_RX_NET_LOGGER.isInfoEnabled()) {

                    StringBuilder sb = new StringBuilder("<<net<< ");
                    sb.append(bidibCommand);
                    sb.append(" : ");
                    sb.append(ByteUtils.bytesToHex(messageArray));

                    MSG_RX_NET_LOGGER.info(sb.toString());
                }

                switch (ByteUtils.getInt(bidibCommand.getType())) {
                    case BidibLibrary.MSG_LOCAL_PROTOCOL_SIGNATURE:
                        // respond the protocol signature
                        final LocalProtocolSignatureMessage localProtocolSignatureMessage =
                                (LocalProtocolSignatureMessage) bidibCommand;

                        processMsgLocalProtocolSignature(ctx, localProtocolSignatureMessage);
                        break;
                    case BidibLibrary.MSG_LOCAL_LINK:
                        LocalLinkMessage localLinkMessage = (LocalLinkMessage) bidibCommand;
                        processMsgLocalLink(ctx, localLinkMessage);
                        break;

                    case BidibLibrary.MSG_LOCAL_LOGON:
                        // forward the command to the backend
                        if (hostAdapter != null) {
                            // the bidibCommand encodes the raw message
                            hostAdapter.forwardMessageToBackend(messageContentSupplier.apply(bidibCommand));
                        }
                        else {
                            LOGGER.warn("No hostAdapter assigned.");
                        }
                        break;

                    case BidibLibrary.MSG_LOCAL_LOGON_ACK:
                        // if MSG_LOCAL_LOGON_ACK is received here we are connected to the remote partner
                        // without bidib-distributed
                        LocalLogonAckMessage localLogonAckMessage = (LocalLogonAckMessage) bidibCommand;
                        processMsgLocalLogonAck(ctx, localLogonAckMessage);
                        break;

                    case BidibLibrary.MSG_LOCAL_LOGON_REJECTED:
                        LOGGER.warn("Process the MSG_LOCAL_LOGON_REJECTED !!!");
                        break;

                    default:
                        LOGGER.debug("Processing BiDiB node related command: {}", bidibCommand);

                        final LogonStatus logonStatus = this.pairedPartner.getLogonStatus();

                        if (LogonStatus.LOGGED_ON == logonStatus) {
                            // forward the command to the backend
                            if (hostAdapter != null) {
                                // the bidibCommand encodes the raw message
                                hostAdapter.forwardMessageToBackend(messageContentSupplier.apply(bidibCommand));
                            }
                            else {
                                LOGGER.warn("No hostAdapter assigned.");
                            }
                        }
                        else {
                            LOGGER.warn("The logon status for the paired partner is not LOGGED_ON.");

                            // TODO init the disconnect
                            throw new IllegalArgumentException("The paired partner is not LOGGED_ON.");
                        }
                        break;
                }
            }
        }
        catch (ProtocolException ex) {
            LOGGER.warn("Process received messages failed: {}", ByteUtils.bytesToHex(messageArray), ex);

            StringBuilder sb = new StringBuilder("<<net<< received invalid: ");
            sb.append(message);
            sb.append(" : ");
            sb.append(ByteUtils.bytesToHex(messageArray));

            MSG_RX_NET_LOGGER.warn(sb.toString());

            throw ex;
        }
        catch (AccessDeniedException ex) {
            LOGGER.warn("Access was denied.", ex);
            throw ex;
        }
        catch (PairingDeniedException ex) {
            LOGGER.warn("Pairing was denied.", ex);
            throw ex;
        }
        catch (Exception ex) {
            LOGGER.warn("Process received messages failed: {}", ByteUtils.bytesToHex(messageArray), ex);

            throw new RuntimeException(ex);
        }
    }

    private void processMsgLocalProtocolSignature(
        final ChannelHandlerContext ctx, final LocalProtocolSignatureMessage localProtocolSignatureMessage)
        throws ProtocolException {
        String requestorName = localProtocolSignatureMessage.getRequestorName();
        LOGGER.info("Received MSG_LOCAL_PROTOCOL_SIGNATURE from requestor: {}", requestorName);

        if (StringUtils.isBlank(requestorName)
            || !requestorName.startsWith(LocalProtocolSignatureMessage.EMITTER_PREFIX_BIDIB)) {
            LOGGER.warn("No requestor provided or the requestor does not start with 'BiDiB'. Abort connection.");

            throw new AccessDeniedException("No requestor provided or the requestor does not start with 'BiDiB'.");
        }

        LOGGER.info("Update the pairedPartner with requestorName: {}", requestorName);
        pairedPartner.setRequestorName(requestorName);

        sendInitialLocalProtocolSignatureMessage();
    }

    /**
     * Process the MSG_LOCAL_LINK netBidib message that was received from the remote partner.
     * 
     * @param ctx
     *            the netty context
     * @param localLinkMessage
     *            the message
     * @throws ProtocolException
     */
    private void processMsgLocalLink(final ChannelHandlerContext ctx, final LocalLinkMessage localLinkMessage)
        throws ProtocolException {
        LOGGER.info("Received MSG_LOCAL_LINK: {}", localLinkMessage);

        switch (localLinkMessage.getLinkDescriptor()) {
            case BidibLibrary.BIDIB_LINK_DESCRIPTOR_UID:
                LOGGER
                    .info("Received the partner UID of the host: {}",
                        ByteUtils.formatHexUniqueId(localLinkMessage.getSenderUniqueId()));

                boolean isPaired = false;

                if (this.pairingStore != null) {
                    try {
                        LOGGER
                            .info("Check if the UID is already in the pairing store: {}",
                                ByteUtils.formatHexUniqueId(localLinkMessage.getSenderUniqueId()));
                        PairingLookupResult pairingLookupResult =
                            this.pairingStore.isPaired(localLinkMessage.getSenderUniqueId());

                        isPaired = PairingLookupResult.PAIRED == pairingLookupResult;

                        LOGGER.info("Result of pairingStore lookup, isPaired: {}", isPaired);
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Get the paired status from the pairing store failed.", ex);
                    }
                }
                else {
                    LOGGER.warn("No pairing store configured.");
                }

                synchronized (pairedPartnerLock) {

                    if (pairedPartner.getUniqueId() == null) {

                        LOGGER
                            .info("Update the pairedPartner with uniqueId: {}",
                                ByteUtils.formatHexUniqueId(localLinkMessage.getSenderUniqueId()));

                        // pairedPartner = new NetBidibLinkData(PartnerType.REMOTE);
                        pairedPartner.setUniqueId(localLinkMessage.getSenderUniqueId());
                        if (isPaired) {
                            // LOGGER.info("Set the pairedPartner as paired based on the pairing store entry.");
                            // pairedPartner.setPairingStatus(PairingStatus.PAIRED);
                            LOGGER.info("Set the serverLinkData as paired based on the pairing store entry.");
                            serverLinkData.setPairingStatus(PairingStatus.PAIRED);
                        }
                        else {
                            LOGGER.info("Set the serverLinkData as unpaired based on the pairing store entry.");
                            serverLinkData.setPairingStatus(PairingStatus.UNPAIRED);
                        }

                        LOGGER.info("Created new pairedPartner: {}", pairedPartner);
                    }
                    else {
                        // check if the paired partner has the matching uniqueId
                        if (pairedPartner.getUniqueId() == null
                            || localLinkMessage.getSenderUniqueId() != pairedPartner.getUniqueId()) {
                            LOGGER
                                .warn(
                                    "The provided uniqueId from the sender does not match the stored uniqueId in the pairedPartner. Do not answer and ignore the message.");
                            // TODO how shall we handle this?

                            return;
                        }
                    }

                    // send only if available already
                    if (serverLinkData.getUniqueId() != null) {
                        LOGGER.info("Current pairingStatus: {}", serverLinkData.getPairingStatus());

                        // send my UID to the remote partner
                        LocalLinkMessage myUID =
                            new LocalLinkMessage(new byte[] { 0 }, 0, serverLinkData.getUniqueId());
                        LOGGER.info("Publish uniqueId of server: {}", myUID);
                        publishMessage(ctx, myUID);

                        switch (serverLinkData.getPairingStatus()) {
                            case UNPAIRED:
                                LOGGER.info("The partner is not paired. Send the STATUS_UNPAIRED.");
                                try {
                                    publishPairedStatus(ctx, BidibLibrary.BIDIB_LINK_STATUS_UNPAIRED);
                                }
                                catch (PairingDeniedException ex) {
                                    LOGGER.info("Pairing is denied but we ignore it here: {}", ex.getMessage());
                                }
                                break;
                            case PAIRED:
                                LOGGER.info("The partner is paired. Send the STATUS_PAIRED.");
                                publishPairedStatus(ctx, BidibLibrary.BIDIB_LINK_STATUS_PAIRED);
                                break;
                            default:
                                break;
                        }
                    }
                    else {
                        LOGGER.info("No uniqueId from connected backend available yet.");
                    }

                    if (serverLinkData.getProdString() != null) {
                        LocalLinkMessage myProd =
                            new LocalLinkMessage(new byte[] { 0 }, 0, BidibLibrary.BIDIB_LINK_DESCRIPTOR_PROD_STRING,
                                serverLinkData.getProdString());
                        LOGGER.info("Publish product string of server: {}", myProd);
                        publishMessage(ctx, myProd);
                    }
                    else {
                        LOGGER.info("No prod string from connected backend available yet.");
                    }

                    if (serverLinkData.getUserString() != null) {
                        LocalLinkMessage myUser =
                            new LocalLinkMessage(new byte[] { 0 }, 0, BidibLibrary.BIDIB_LINK_DESCRIPTOR_USER_STRING,
                                serverLinkData.getUserString());
                        LOGGER.info("Publish user string of server: {}", myUser);
                        publishMessage(ctx, myUser);
                    }
                    else {
                        LOGGER.info("No user string from connected backend available yet.");
                    }

                    if (serverLinkData.getProtocolVersion() != null) {
                        LocalLinkMessage myProtocolVersion =
                            new LocalLinkMessage(new byte[] { 0 }, 0, BidibLibrary.BIDIB_LINK_DESCRIPTOR_P_VERSION,
                                serverLinkData.getProtocolVersion());
                        LOGGER.info("Publish protocol version of server: {}", myProtocolVersion);
                        publishMessage(ctx, myProtocolVersion);
                    }
                    else {
                        LOGGER.info("No protocol version from connected backend available yet.");
                    }

                }

                break;

            case BidibLibrary.BIDIB_LINK_DESCRIPTOR_P_VERSION:
                LOGGER.info("Received the partner P_VERSION: {}", localLinkMessage.getProtocolVersion());
                synchronized (pairedPartnerLock) {
                    pairedPartner.setProtocolVersion(localLinkMessage.getProtocolVersion());
                }
                break;
            case BidibLibrary.BIDIB_LINK_DESCRIPTOR_PROD_STRING:
                LOGGER.info("Received the partner PROD_STRING: {}", localLinkMessage.getProdString());
                synchronized (pairedPartnerLock) {
                    pairedPartner.setProdString(localLinkMessage.getProdString());
                }
                break;
            case BidibLibrary.BIDIB_LINK_DESCRIPTOR_USER_STRING:
                LOGGER.info("Received the partner USER_STRING: {}", localLinkMessage.getProdString());
                synchronized (pairedPartnerLock) {
                    pairedPartner.setUserString(localLinkMessage.getProdString());
                }
                break;
            case BidibLibrary.BIDIB_LINK_STATUS_PAIRED:
                LOGGER
                    .info("Received a paired message, senderUID: {}, receiverUID: {}",
                        ByteUtils.formatHexUniqueId(localLinkMessage.getSenderUniqueId()),
                        ByteUtils.formatHexUniqueId(localLinkMessage.getReceiverUniqueId()));

                synchronized (pairedPartnerLock) {
                    pairedPartner.setPairingStatus(PairingStatus.PAIRED);
                }

                if (this.pairingStore != null) {
                    try {
                        LOGGER.info("Update the pairing store.");
                        this.pairingStore
                            .setPaired(localLinkMessage.getSenderUniqueId(), pairedPartner.getRequestorName(),
                                pairedPartner.getProdString(), pairedPartner.getUserString(),
                                pairedPartner.getProtocolVersion(), true);
                        this.pairingStore.store();
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Store the pairing status failed.", ex);
                    }
                }
                else {
                    LOGGER.warn("No pairing store configured.");
                }

                // check if we want really want to pair
                if (serverLinkData.getPairingStatus() == PairingStatus.UNPAIRED) {
                    LOGGER.info("Send the UNPAIRED status.");
                    publishPairedStatus(ctx, BidibLibrary.BIDIB_LINK_STATUS_UNPAIRED);

                    LOGGER.info("Send the Logoff message.");
                    LocalLogoffMessage localLogoffMessage =
                        new LocalLogoffMessage(new byte[] { 0 }, 0, serverLinkData.getUniqueId());
                    publishMessage(ctx, localLogoffMessage);
                }
                else {
                    // if we are in the interface role we must wait for the logon from the node
                    if (this.roleType == RoleTypeEnum.NODE) {
                        LOGGER.info("Send the logon message after receive PAIRED message from remote partner.");
                        synchronized (pairedPartnerLock) {
                            if (PairingStatus.PAIRED == pairedPartner.getPairingStatus()
                                && LogonStatus.LOGGED_OFF == pairedPartner.getLogonStatus()) {
                                LOGGER
                                    .info(
                                        "The pairedPartner accepted the pairing already but is LOGGED_OFF. Send the LOGON.");

                                LocalLogonMessage localLogonMessage =
                                    new LocalLogonMessage(new byte[] { 0 }, 0, serverLinkData.getUniqueId());
                                publishMessage(ctx, localLogonMessage);

                                pairedPartner.setLogonStatus(LogonStatus.LOGGED_ON);
                            }
                        }
                    }
                    else {
                        LOGGER.info("Do not sent logon because role type INTERFACE is configured.");
                    }
                }
                break;

            case BidibLibrary.BIDIB_LINK_STATUS_UNPAIRED:
                LOGGER
                    .info("Received an UNPAIRED message, senderUID: {}, receiverUID: {}",
                        ByteUtils.formatHexUniqueId(localLinkMessage.getSenderUniqueId()),
                        ByteUtils.formatHexUniqueId(localLinkMessage.getReceiverUniqueId()));

                synchronized (pairedPartnerLock) {
                    pairedPartner.setPairingStatus(PairingStatus.UNPAIRED);
                }

                // TODO we must send a pairing request

                // NetBidibCommandMessage pairingRequestMessage =
                // bidibRequestFactory
                // .createLocalLinkPairingRequest(serverLinkData.getUniqueId(),
                // localLinkMessage.getSenderUniqueId(), pairingTimeout.intValue());
                // publishMessage(ctx, pairingRequestMessage);
                // serverLinkData.setPairingStatus(PairingStatus.PAIRING_REQUESTED);
                break;

            case BidibLibrary.BIDIB_LINK_PAIRING_REQUEST:
                LOGGER
                    .info("Received a pairing request, senderUID: {}, receiverUID: {}, pairingTimeout: {}",
                        ByteUtils.formatHexUniqueId(localLinkMessage.getSenderUniqueId()),
                        ByteUtils.formatHexUniqueId(localLinkMessage.getReceiverUniqueId()),
                        localLinkMessage.getPairingTimeout());

                synchronized (pairedPartnerLock) {
                    pairedPartner.setUniqueId(localLinkMessage.getSenderUniqueId());
                    pairedPartner.setPairingStatus(PairingStatus.PAIRING_REQUESTED);
                }

                final Integer pairingTimeout = localLinkMessage.getPairingTimeout();

                /**
                 * try { // send the unpaired result to the client publishPairedStatus(ctx,
                 * BidibLibrary.BIDIB_LINK_STATUS_UNPAIRED); } catch (PairingDeniedException ex) { LOGGER.info("Ignore
                 * the pairing denied exception."); }
                 */
                // use a callback to let the user confirm the pairing process the pairing request

                pairingWorker.schedule(() -> {
                    Boolean paired = Boolean.FALSE;

                    if (pairingCallback != null) {
                        LOGGER.info("Use the pairing callback to get the paired result.");
                        synchronized (pairedPartnerLock) {
                            paired = pairingCallback.apply(pairedPartner, pairingTimeout);
                        }
                    }
                    else {
                        LOGGER.warn("No pairingCallback available. Accept every client.");
                        paired = Boolean.TRUE;
                    }

                    LOGGER
                        .info("After pairing callback. Pairing success: {}, pairedPartner: {}", paired, pairedPartner);
                    try {
                        if (Boolean.TRUE.equals(paired)) {
                            publishPairedStatus(ctx, BidibLibrary.BIDIB_LINK_STATUS_PAIRED);
                        }
                        else {
                            publishPairedStatus(ctx, BidibLibrary.BIDIB_LINK_STATUS_UNPAIRED);
                        }
                    }
                    catch (ProtocolException ex) {
                        LOGGER.warn("Publish paired status failed.", ex);
                    }
                    catch (PairingDeniedException ex) {
                        LOGGER.warn("Pairing was denied. We must close the connection.", ex);
                        // TODO: handle exception

                        channelGroup.close();
                    }
                }, 5, TimeUnit.MILLISECONDS);

                break;
            default:
                LOGGER.warn("Unhandled message: {}", localLinkMessage);
                break;
        }
    }

    private final ScheduledExecutorService pairingWorker =
        Executors
            .newScheduledThreadPool(1, new ThreadFactoryBuilder().setNameFormat("pairingWorkers-thread-%d").build());

    private BiFunction<NetBidibLinkData, Integer, Boolean> pairingCallback;

    public void setPairingCallback(final BiFunction<NetBidibLinkData, Integer, Boolean> pairingCallback) {
        LOGGER.info("Set the pairing callback: {}", pairingCallback);
        this.pairingCallback = pairingCallback;
    }

    /**
     * Process the MSG_LOCAL_LOGON_ACK netBidib message that was received from the remote partner.
     * 
     * @param ctx
     *            the netty context
     * @param localLogonAckMessage
     *            the message
     */
    private void processMsgLocalLogonAck(
        final ChannelHandlerContext ctx, final LocalLogonAckMessage localLogonAckMessage) {
        LOGGER.info("Received MSG_LOCAL_LOGON_ACK: {}", localLogonAckMessage);

        int nodeAddress = localLogonAckMessage.getNodeAddress();
        long uniqueId = localLogonAckMessage.getSenderUniqueId();

        LOGGER.info("Current nodeAddress: {}, uniqueId: {}", nodeAddress, ByteUtils.formatHexUniqueId(uniqueId));

        // TODO use the provided nodeAddress as the local bidib address

        if (serverLinkData.getUniqueId() == uniqueId) {
            synchronized (pairedPartnerLock) {
                pairedPartner.setLogonStatus(LogonStatus.LOGGED_ON);

                LOGGER.info("Current pairedPartner: {}", pairedPartner);
            }
        }
        else {
            LOGGER
                .warn("The provided uniqueId does not match, provided: {}, expected: {}",
                    ByteUtils.formatHexUniqueId(uniqueId), ByteUtils.formatHexUniqueId(serverLinkData.getUniqueId()));
        }
    }

    protected void publishPairedStatus(final ChannelHandlerContext ctx, int descriptor) throws ProtocolException {

        LOGGER.info("Publish the paired status: {}", descriptor);
        if (BidibLibrary.BIDIB_LINK_STATUS_PAIRED == descriptor) {
            LOGGER.info("The BIDIB_LINK_STATUS_PAIRED is published.");

            // set the state to paired
            ctx.channel().attr(ConnectionState.STATE_KEY).set(ConnectionState.Phase.PAIRED);

            LOGGER.info("Send the status pairing status PAIRED message to the client.");

            NetBidibCommandMessage statusPaired =
                bidibRequestFactory
                    .createLocalLinkStatusPaired(serverLinkData.getUniqueId(), pairedPartner.getUniqueId());
            publishMessage(ctx, statusPaired);

            serverLinkData.setPairingStatus(PairingStatus.PAIRED);

            LOGGER.info("Current pairedPartner: {}, serverLinkData: {}", pairedPartner, serverLinkData);

            if (PairingStatus.PAIRED == pairedPartner.getPairingStatus()
                && PairingStatus.PAIRED == serverLinkData.getPairingStatus()
                && LogonStatus.LOGGED_OFF == pairedPartner.getLogonStatus()) {
                LOGGER.info("The pairedPartner accepted the pairing already. Send the LOGON.");

                LocalLogonMessage localLogonMessage =
                    new LocalLogonMessage(new byte[] { 0 }, 0, serverLinkData.getUniqueId());
                publishMessage(ctx, localLogonMessage);

                pairedPartner.setLogonStatus(LogonStatus.LOGGED_ON);
            }
            else {
                LOGGER.info("LOGON was not sent.");
            }

        }
        else {
            LOGGER.info("The BIDIB_LINK_STATUS_UNPAIRED is published.");

            // set the state to unpaired
            ctx.channel().attr(ConnectionState.STATE_KEY).set(ConnectionState.Phase.UNPAIRED);

            NetBidibCommandMessage statusUnpaired =
                bidibRequestFactory
                    .createLocalLinkStatusUnpaired(serverLinkData.getUniqueId(), pairedPartner.getUniqueId());

            publishMessage(ctx, statusUnpaired);

            serverLinkData.setPairingStatus(PairingStatus.UNPAIRED);

            LOGGER
                .info("Throw PairingDeniedException to signal the connection is not paired. Current uniqueId: {}",
                    ByteUtils.formatHexUniqueId(pairedPartner.getUniqueId()));

            throw new PairingDeniedException(
                "Pairing denied for uniqueId: " + ByteUtils.formatHexUniqueId(pairedPartner.getUniqueId()));
        }
    }

    /**
     * Process the {@code LocalBidibUpResponse} and publish {@code LocalLinkMessage} messages to the connected partner.
     * 
     * @param localBidibUpResponse
     *            the localBidibUpResponse message received from the backend
     * @return {@code true}: send the pairing status to the connected partner, {@code false}: don't send the pairing
     *         status to the connected partner
     * @throws ProtocolException
     *             thrown if illegal message content is detected or processing of message failed.
     */
    protected boolean processLocalBidibUpResponseFromBackend(final LocalBidibUpResponse localBidibUpResponse)
        throws ProtocolException {

        boolean sendPairingStatusIfRequired = false;

        try {
            // add missing header info to the wrapped message

            byte[] wrapped = localBidibUpResponse.getWrappedMessage();

            byte[] raw = ByteUtils.concat(new byte[] { ByteUtils.getLowByte(wrapped.length + 2), 0, 0 }, wrapped);

            BidibMessageInterface message = responseFactory.create(raw);
            LOGGER.info("Received wrapped message: {}", message);

            boolean publish = false;
            switch (ByteUtils.getInt(message.getType())) {
                case BidibLibrary.MSG_SYS_UNIQUE_ID:
                    SysUniqueIdResponse response = (SysUniqueIdResponse) message;

                    synchronized (pairedPartnerLock) {
                        publish = serverLinkData.getUniqueId() == null;
                        serverLinkData.setUniqueId(NodeUtils.getUniqueId(response.getUniqueId()));

                        if (pairedPartner != null && pairedPartner.getUniqueId() != null && publish) {
                            LOGGER.info("Publish the uniqueId: {}", serverLinkData.getUniqueId());
                            LocalLinkMessage myUID =
                                new LocalLinkMessage(new byte[] { 0 }, 0, serverLinkData.getUniqueId());
                            publishMessage(ctx, myUID);
                        }
                    }
                    break;
                case BidibLibrary.MSG_STRING:
                    StringResponse stringResponse = (StringResponse) message;
                    StringData stringData = stringResponse.getStringData();

                    switch (stringData.getIndex()) {
                        case StringData.INDEX_PRODUCTNAME:
                            synchronized (pairedPartnerLock) {
                                publish = serverLinkData.getProdString() == null;

                                serverLinkData.setProdString(stringData.getValue());

                                // publish if the paired partner is available
                                if (pairedPartner != null && pairedPartner.getUniqueId() != null && publish) {
                                    LocalLinkMessage myProd =
                                        new LocalLinkMessage(new byte[] { 0 }, 0,
                                            BidibLibrary.BIDIB_LINK_DESCRIPTOR_PROD_STRING,
                                            serverLinkData.getProdString());
                                    LOGGER.info("Publish the product string: {}", myProd);
                                    publishMessage(ctx, myProd);
                                }
                            }
                            break;
                        case StringData.INDEX_USERNAME:
                            synchronized (pairedPartnerLock) {
                                publish = serverLinkData.getUserString() == null;

                                serverLinkData.setUserString(stringData.getValue());

                                if (pairedPartner != null && pairedPartner.getUniqueId() != null && publish) {
                                    LocalLinkMessage myUser =
                                        new LocalLinkMessage(new byte[] { 0 }, 0,
                                            BidibLibrary.BIDIB_LINK_DESCRIPTOR_USER_STRING,
                                            serverLinkData.getUserString());
                                    LOGGER.info("Publish the user string: {}", myUser);
                                    publishMessage(ctx, myUser);
                                }
                            }
                            break;
                        default:
                            break;
                    }

                    break;
                case BidibLibrary.MSG_SYS_P_VERSION:
                    SysPVersionResponse pVersionResponse = (SysPVersionResponse) message;

                    synchronized (pairedPartnerLock) {
                        publish = serverLinkData.getProtocolVersion() == null;
                        serverLinkData.setProtocolVersion(pVersionResponse.getVersion());

                        if (pairedPartner != null && pairedPartner.getUniqueId() != null && publish) {
                            LocalLinkMessage myProtocolVersion =
                                new LocalLinkMessage(new byte[] { 0 }, 0, BidibLibrary.BIDIB_LINK_DESCRIPTOR_P_VERSION,
                                    serverLinkData.getProtocolVersion());
                            LOGGER.info("Publish the protocol version: {}", myProtocolVersion);
                            publishMessage(ctx, myProtocolVersion);
                        }

                        sendPairingStatusIfRequired = true;
                    }
                    break;
                default:
                    break;
            }
        }
        catch (ProtocolException ex) {
            LOGGER.warn("Create message from wrapped message data failed: {}", ex.getMessage());
            throw new ProtocolInvalidContentException("Create message from wrapped message data failed.", ex);
        }
        catch (Exception ex) {
            LOGGER.warn("Process the LocalBidibUpResponse message failed.", ex);
            throw new ProtocolException("Process the LocalBidibUpResponse message failed.");
        }

        return sendPairingStatusIfRequired;
    }

    /**
     * Publish the message to the connected host.
     * 
     * @param ctx
     *            the context
     * @param message
     *            the message
     */
    private void publishMessage(final ChannelHandlerContext ctx, final BidibMessageInterface message) {
        LOGGER.info("Publish the message to channel: {}", message);

        byte[] msg = message.getContent();

        if (MSG_TX_NET_LOGGER.isInfoEnabled()) {
            // log the bidib message and the content of the "output" stream
            StringBuilder sb = new StringBuilder(">>net>> ");
            sb.append(message);
            sb.append(" : ");
            sb.append(ByteUtils.bytesToHex(msg));
            MSG_TX_NET_LOGGER.info(sb.toString());
        }

        // this.socketChannel.writeAndFlush(Unpooled.copiedBuffer(msg));
        channelGroup.iterator().next().writeAndFlush(Unpooled.copiedBuffer(msg));

        LOGGER.info("Write message to socketChannel has finished, msg: {}", ByteUtils.bytesToHex(msg));
        // ctx.channel().writeAndFlush(Unpooled.copiedBuffer(msg));
    }

    @Override
    public void channelReadComplete(ChannelHandlerContext ctx) throws Exception {

        LOGGER.debug("channelReadComplete.");

        // ctx.writeAndFlush(Unpooled.EMPTY_BUFFER).addListener(ChannelFutureListener.CLOSE);
    }

    @Override
    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) throws Exception {
        LOGGER.warn("Exception in handler. Close the channelHandlerContext.", cause);
        if (cause instanceof IOException) {
            LOGGER.warn("IOException: {}", cause.getMessage());
        }
        else {
            LOGGER.warn("Exception caught in server handler. Will close connection.", cause);
        }
        ctx.channel().attr(ConnectionState.STATE_KEY).set(ConnectionState.Phase.NOT_CONNECTED);
        ctx.close();

        LOGGER.info("Clear the info of the paired partner.");
        synchronized (pairedPartnerLock) {
            pairedPartner.clear(true);
        }
    }

    /**
     * Returns true if currently an active connection exists.
     *
     * @return channel group has an active connection
     */
    public boolean hasActiveConnection() {
        return !channelGroup.isEmpty();
    }

    // public void setPairedPartner(NetBidibLinkData pairedPartner) {
    // LOGGER.info("Set the new paired partner: {}", pairedPartner);
    //
    // synchronized (pairedPartnerLock) {
    // this.pairedPartner = pairedPartner;
    // }
    //
    // }

}
