package org.bidib.jbidibc.netbidib.server;

import java.util.function.Consumer;

import org.bidib.jbidibc.messages.BidibLibrary;
import org.bidib.jbidibc.messages.HostAdapter;
import org.bidib.jbidibc.messages.SequenceNumberProvider;
import org.bidib.jbidibc.messages.message.BidibMessageInterface;
import org.bidib.jbidibc.messages.message.LocalBidibUpResponse;
import org.bidib.jbidibc.messages.message.netbidib.NetBidibLinkData;
import org.bidib.jbidibc.messages.message.netbidib.NetBidibLinkData.LogonStatus;
import org.bidib.jbidibc.messages.message.netbidib.NetBidibLinkData.PairingStatus;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.jbidibc.netbidib.exception.PairingDeniedException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.netty.buffer.Unpooled;
import io.netty.channel.group.ChannelGroup;

public class NetBidibServerHandlerByteArray extends NetBidibServerHandler<byte[]> {

    private static final Logger LOGGER = LoggerFactory.getLogger(NetBidibServerHandlerByteArray.class);

    private static final Logger MSG_TX_NET_LOGGER = LoggerFactory.getLogger("TX_NET");

    public NetBidibServerHandlerByteArray(ChannelGroup channelGroup, HostAdapter<byte[]> hostAdapter,
        String backendPortName, NetBidibLinkData serverLinkData,
        Consumer<NetBidibServerHandler<byte[]>> lazyInitializationCallback, final RoleTypeEnum roleType,
        final NetBidibLinkData pairedPartner) {
        super(channelGroup, hostAdapter, backendPortName, serverLinkData, lazyInitializationCallback,
            message -> message.getContent(), roleType, pairedPartner);
    }

    @Override
    public void publishBidibMessage(final SequenceNumberProvider proxyNode, final byte[] content) {
        LOGGER.info("Received message to publish to host: {}", ByteUtils.bytesToHex(content));

        // do some inspection and filter out the MSG_BIDIB_LOCAL_UP
        if (handleLocalBidibUpResponse()) {
            try {
                BidibMessageInterface message = responseFactory.create(content);
                if (message instanceof LocalBidibUpResponse) {
                    LocalBidibUpResponse localBidibUpResponse = (LocalBidibUpResponse) message;
                    LOGGER.info("Do not deliver LocalBidibUpResponse: {}", localBidibUpResponse);

                    boolean sendPairingStatusIfRequired = processLocalBidibUpResponseFromBackend(localBidibUpResponse);

                    if (sendPairingStatusIfRequired) {
                        LOGGER.info("Check if we must send the current pairing status: {}", serverLinkData);
                        // TODO
                        if (serverLinkData.getUniqueId() != null) {
                            try {

                                if (serverLinkData.getPairingStatus() == null
                                    || serverLinkData.getPairingStatus() == PairingStatus.UNKNOWN) {
                                    LOGGER.warn("Set the pairing status from UNKNOWN to UNPAIRED.");
                                    serverLinkData.setPairingStatus(PairingStatus.UNPAIRED);
                                }

                                LOGGER
                                    .info("Current pairingStatus stored in serverLinkData: {}",
                                        serverLinkData.getPairingStatus());

                                switch (serverLinkData.getPairingStatus()) {
                                    case UNPAIRED:
                                        LOGGER.info("The partner is not paired. Send the STATUS_UNPAIRED.");
                                        publishPairedStatus(ctx, BidibLibrary.BIDIB_LINK_STATUS_UNPAIRED);
                                        break;
                                    case PAIRED:
                                        LOGGER.info("The partner is paired. Send the STATUS_PAIRED.");
                                        publishPairedStatus(ctx, BidibLibrary.BIDIB_LINK_STATUS_PAIRED);
                                        break;
                                    default:
                                        break;
                                }
                            }
                            catch (PairingDeniedException pde) {
                                LOGGER.info("The pairing is denied.", pde);
                            }
                        }
                        else {
                            LOGGER.info("The uniqueId of the server link is not yet available.");
                        }
                    }
                    return;
                }
            }
            catch (Exception ex) {
                LOGGER.warn("Check if message is local response failed.");
            }
        }

        synchronized (pairedPartnerLock) {
            if (pairedPartner == null || LogonStatus.LOGGED_OFF == pairedPartner.getLogonStatus()) {
                LOGGER
                    .warn(
                        "The paired partner is not available or not logged on. Discard the message. Current pairedPartner: {}",
                        pairedPartner);
                return;
            }
        }

        LOGGER.info("Publish the message data to channel: {}", ByteUtils.bytesToHex(content));

        if (MSG_TX_NET_LOGGER.isInfoEnabled()) {
            // log the bidib message and the content of the "output" stream
            StringBuilder sb = new StringBuilder(">>net>>");
            sb.append("BiDiB");
            sb.append(" : ");
            sb.append(ByteUtils.bytesToHex(content));
            MSG_TX_NET_LOGGER.info(sb.toString());
        }

        ctx.writeAndFlush(Unpooled.copiedBuffer(content));
    }

}
