package org.bidib.jbidibc.netbidib.server;

import java.io.IOException;
import java.util.function.Consumer;

import org.bidib.jbidibc.messages.HostAdapter;
import org.bidib.jbidibc.messages.ProtocolVersion;
import org.bidib.jbidibc.messages.message.netbidib.NetBidibLinkData;
import org.bidib.jbidibc.messages.message.netbidib.NetBidibLinkData.LogonStatus;
import org.bidib.jbidibc.messages.message.netbidib.NetBidibLinkData.PairingStatus;
import org.bidib.jbidibc.messages.message.netbidib.NetBidibLinkData.PartnerType;
import org.bidib.jbidibc.netbidib.server.adapter.ScmSerialHostAdapter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.netty.channel.group.ChannelGroup;

public class NetBidibServerByteArray extends NetBidibServer<byte[]> {

    private static final Logger LOGGER = LoggerFactory.getLogger(NetBidibServerByteArray.class);

    public NetBidibServerByteArray(final HostAdapter<byte[]> hostAdapter, final String backendPortName,
        final NetBidibLinkData serverLinkData, RoleTypeEnum roleType, final NetBidibLinkData pairedPartner) {
        this(DEFAULT_HOSTNAME, DEFAULT_PORTNUM, hostAdapter, backendPortName, serverLinkData, roleType, pairedPartner);
    }

    /**
     * Create a {@code NetBidibServer} instance with the provided hostname and portNumber.
     * 
     * @param hostName
     *            the hostname
     * @param portNumber
     *            the port number
     */
    public NetBidibServerByteArray(String hostName, int portNumber, final HostAdapter<byte[]> hostAdapter,
        final String backendPortName, final NetBidibLinkData serverLinkData, RoleTypeEnum roleType,
        final NetBidibLinkData pairedPartner) {
        super(hostName, portNumber, hostAdapter, backendPortName, serverLinkData, roleType, pairedPartner);
    }

    /**
     * Create the {@code NetBidibServerHandler} instance.
     * <p>
     * Note: Override this method to do more initialization of the {@code NetBidibServerHandler}.
     * </p>
     * 
     * @param channelGroup
     *            the netty channel group
     * @return the serverHandler instance
     */
    @Override
    protected NetBidibServerHandler<byte[]> createNetBidibServerHandler(
        final ChannelGroup channelGroup, final NetBidibLinkData serverLinkData, final HostAdapter<byte[]> hostAdapter,
        final String backendPortName, final Consumer<NetBidibServerHandler<byte[]>> lazyInitializationCallback,
        RoleTypeEnum roleType, final NetBidibLinkData pairedPartner) {
        NetBidibServerHandler<byte[]> serverHandler =
            new NetBidibServerHandlerByteArray(channelGroup, hostAdapter, backendPortName, serverLinkData,
                lazyInitializationCallback, roleType, pairedPartner);

        return serverHandler;
    }

    /**
     * Main to test the local server part without bidib messages for nodes.
     * 
     * @param args
     *            the args
     * @throws InterruptedException
     * @throws IOException
     */
    public static void main(String[] args) throws InterruptedException, IOException {

        String backendPortName = null;
        if (args.length > 0) {
            backendPortName = args[0];
        }
        else {
            LOGGER.warn("Provide the backend port name as argument, e.g. COM5");
            return;
        }

        LOGGER.info("Start the server. The current backendPortName: {}", backendPortName);

        final NetBidibLinkData serverLinkData = new NetBidibLinkData(PartnerType.LOCAL);

        // // create my unique id
        // byte[] uniqueId = new byte[] { 0x00, 0x00, 0x0D, (byte) 0xFA, 0x01, 0x03, (byte) 0xF1 };
        // serverLinkData.setUniqueId(ByteUtils.convertUniqueIdToLong(uniqueId));
        // serverLinkData.setProdString("netBiDiB-Gateway");
        // serverLinkData.setUserString("Central-Gateway");
        serverLinkData.setProtocolVersion(ProtocolVersion.VERSION_0_8);

        final ScmSerialHostAdapter<byte[]> hostAdapter = new ScmSerialHostAdapter<>(message -> message.getContent());

        RoleTypeEnum roleType = RoleTypeEnum.INTERFACE;
        final NetBidibLinkData pairedPartner = new NetBidibLinkData(PartnerType.REMOTE);
        pairedPartner.setPairingStatus(PairingStatus.UNKNOWN);
        pairedPartner.setLogonStatus(LogonStatus.LOGGED_OFF);

        NetBidibServer<byte[]> netBidibServer =
            new NetBidibServerByteArray(NetBidibServer.DEFAULT_HOSTNAME, NetBidibServer.DEFAULT_PORTNUM, hostAdapter,
                backendPortName, serverLinkData, roleType, pairedPartner);

        netBidibServer.setShutdownHook(new Thread(() -> {
            LOGGER.info("Run shutdown hook.");

            try {
                netBidibServer.stop();
            }
            catch (Exception ex) {
                LOGGER.warn("Stop the netBidibServer failed.", ex);
            }
        }));
        Runtime.getRuntime().addShutdownHook(netBidibServer.getShutdownHook());

        netBidibServer.startServer();

        LOGGER.info("Wait for server startup.");

        try {
            Thread.sleep(500);

            LOGGER.info("Wait for shutdown.");
            final Object shutdownLock = netBidibServer.getShutdownLock();
            synchronized (shutdownLock) {
                shutdownLock.wait();
            }
        }
        finally {
            LOGGER.info("Stop the netBidibServer.");
            try {
                netBidibServer.stop();
            }
            catch (Exception ex) {
                LOGGER.warn("Stop the netBidibServer failed.", ex);
            }
        }

    }

}
