package org.bidib.jbidibc.netbidib.server;

import java.io.ByteArrayOutputStream;
import java.util.Arrays;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Function;
import java.util.function.Supplier;

import org.apache.commons.lang3.StringUtils;
import org.bidib.jbidibc.core.BidibMessageEvaluator;
import org.bidib.jbidibc.messages.BidibMessagePublisher;
import org.bidib.jbidibc.messages.ConnectionListener;
import org.bidib.jbidibc.messages.Node;
import org.bidib.jbidibc.messages.SequenceNumberProvider;
import org.bidib.jbidibc.messages.enums.NetBidibSocketType;
import org.bidib.jbidibc.messages.enums.PairingResult;
import org.bidib.jbidibc.messages.exception.ProtocolException;
import org.bidib.jbidibc.messages.helpers.Context;
import org.bidib.jbidibc.messages.message.BidibCommand;
import org.bidib.jbidibc.messages.message.BidibCommandMessage;
import org.bidib.jbidibc.messages.message.BidibMessageInterface;
import org.bidib.jbidibc.messages.message.BidibRequestFactory;
import org.bidib.jbidibc.messages.message.netbidib.NetBidibLinkData;
import org.bidib.jbidibc.messages.message.netbidib.NetBidibLinkData.PartnerType;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.jbidibc.messages.utils.ThreadFactoryBuilder;
import org.bidib.jbidibc.netbidib.client.pairingstates.DefaultPairingStateHandler;
import org.bidib.jbidibc.netbidib.client.pairingstates.NetBidibMessageSender;
import org.bidib.jbidibc.netbidib.client.pairingstates.PairingInteractionPublisher;
import org.bidib.jbidibc.netbidib.client.pairingstates.PairingStateHandler;
import org.bidib.jbidibc.netbidib.pairingstore.PairingStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.netty.buffer.Unpooled;
import io.netty.channel.ChannelHandlerContext;

/**
 * The {@code AbstractNetBidibServerHandler} is a netty channel handler that is called when a data packet is received.
 *
 * @param <T>
 *            The type of message content that is produced.
 */
public abstract class AbstractNetBidibServerHandler<T> implements BidibMessagePublisher<T> {

    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractNetBidibServerHandler.class);

    private static final Logger MSG_TX_LOGGER = LoggerFactory.getLogger("TX");

    private BidibRequestFactory bidibRequestFactory;

    protected final Object pairedPartnerLock = new Object();

    private PairingStore pairingStore;

    /**
     * The map that holds the pairing state handlers per netty context.
     */
    private final Map<String, ChannelHandlerContext> channelHandlerMap = new LinkedHashMap<>();

    private final Map<String, PairingStateHandler> pairingStateHandlerMap = new LinkedHashMap<>();

    private Set<ConnectionListener> remoteConnectionListeners = new HashSet<>();

    private RoleTypeEnum roleType;

    /**
     * our own link data
     */
    private final NetBidibLinkData serverLinkData;

    private ServerNetMessageReceiver netMessageReceiver;

    private final ScheduledExecutorService logonReceivedPublisherWorker;

    private final Supplier<BidibMessageEvaluator> bidibMessageEvaluatorSupplier;

    private NetBidibMessageSender netBidibMessageSender;

    private PairingInteractionPublisher pairingInteractionPublisher;

    /**
     * Constructor
     *
     * @param channelGroup
     *            the channel group to pool all TCP connections.
     * @param hostAdapter
     *            the host adapter
     * @param connectionId
     *            the connection name
     * @param serverLinkData
     *            the server link data instance to use (with default values provided from config)
     */
    public AbstractNetBidibServerHandler(final NetBidibLinkData serverLinkData,
        final Function<BidibMessageInterface, T> messageContentSupplier, final RoleTypeEnum roleType,
        final Supplier<BidibMessageEvaluator> bidibMessageEvaluatorSupplier) {

        LOGGER.info("Create new NetBidibServerHandler instance. Provided roleType: {}", roleType);

        this.serverLinkData = serverLinkData;

        this.roleType = roleType;
        this.bidibMessageEvaluatorSupplier = bidibMessageEvaluatorSupplier;

        this.logonReceivedPublisherWorker =
            Executors
                .newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().setNameFormat("publisherWorkers-thread-%d").build());
    }

    public void setNetMessageReceiver(final ServerNetMessageReceiver netMessageReceiver) {
        LOGGER.info("Set the netMessageReceiver: {}", netMessageReceiver);
        this.netMessageReceiver = netMessageReceiver;

        LOGGER.info("Set the message listener map and the message evaluator in the message receiver.");
        this.netMessageReceiver.setNetBidibLocalMessageListenerMap(pairingStateHandlerMap);
        this.netMessageReceiver.setBidibMessageEvaluator(bidibMessageEvaluatorSupplier.get());

    }

    public void initialize() {
        LOGGER.info("Initialize the NetBidibServerHandler.");

        this.bidibRequestFactory = new BidibRequestFactory();
        this.bidibRequestFactory.setEscapeMagic(false);
        this.bidibRequestFactory.initialize();

        this.netBidibMessageSender = new NetBidibMessageSender() {

            @Override
            public void publishNetBidibMessage(final String contextKey, final BidibCommand message)
                throws ProtocolException {
                LOGGER.info("publishNetBidibMessage, contextKey: {}, message: {}", contextKey, message);
                try {
                    sendNetBidibMessage(contextKey, message);
                }
                catch (Exception ex) {
                    LOGGER.warn("Send the netBidibMessage failed.", ex);
                    throw new ProtocolException("Send the netBidibMessage failed.");
                }
            }

            @Override
            public void publishLocalLogonRejected(String contextKey, long uniqueId) {
                LOGGER
                    .info("publishLocalLogonRejected, contextKey: {}, uniqueId: {}", contextKey,
                        ByteUtils.formatHexUniqueId(uniqueId));

                sendLocalLogonRejected(contextKey, uniqueId);
            }
        };

        this.pairingInteractionPublisher = new PairingInteractionPublisher() {

            @Override
            public void publishUserAction(final String actionKey, final Context context) {
                LOGGER.info("Publish the user action, actionKey: {}, context: {}", actionKey, context);

                logonReceivedPublisherWorker
                    .submit(() -> AbstractNetBidibServerHandler.this.publishUserAction(actionKey, context));
            }

            @Override
            public void publishPairingFinished(final PairingResult pairingResult, long uniqueId) {
                LOGGER.info("Publish the pairing result: {}", pairingResult);

                logonReceivedPublisherWorker
                    .submit(() -> AbstractNetBidibServerHandler.this.publishPairingFinished(pairingResult, uniqueId));
            }

            @Override
            public void publishLocalLogon(int localNodeAddr, long uniqueId) {
                LOGGER
                    .info("Publish the logon received from a different thread, localNodeAddr: {}, uniqueId: {}",
                        localNodeAddr, ByteUtils.formatHexUniqueId(uniqueId));

                logonReceivedPublisherWorker
                    .submit(() -> AbstractNetBidibServerHandler.this.publishLogonReceived(localNodeAddr, uniqueId));
            }

            @Override
            public void publishLocalLogoff(long uniqueId) {
                LOGGER.info("Publish the logoff received from a different thread.");

                logonReceivedPublisherWorker
                    .submit(() -> AbstractNetBidibServerHandler.this.publishLogoffReceived(uniqueId));
            }

            @Override
            public void handleError(RuntimeException ex) {
                logonReceivedPublisherWorker.submit(() -> AbstractNetBidibServerHandler.this.handleError(ex));
            }
        };
    }

    private PairingStateHandler createPairingStateHandler(final String contextKey) {
        LOGGER.info("Create new pairing state handler for contextKey: {}", contextKey);

        // create the pairing state handler
        PairingStateHandler netBidibPairingStateHandler =
            new DefaultPairingStateHandler(this.netBidibMessageSender, this.pairingInteractionPublisher,
                this.bidibRequestFactory, contextKey);
        netBidibPairingStateHandler.setNetBidibSocketType(NetBidibSocketType.serverSocket);

        NetBidibLinkData remotePartnerLinkData = new NetBidibLinkData(PartnerType.REMOTE);
        NetBidibLinkData serverLinkData = new NetBidibLinkData(this.serverLinkData);

        netBidibPairingStateHandler.initialize(remotePartnerLinkData, serverLinkData, this.pairingStore);

        return netBidibPairingStateHandler;
    }

    public PairingStateHandler getPairingStateHandler(final Long uniqueId) {
        LOGGER.info("Get the pairing state handler for remote partner uniqueId: {}", uniqueId);

        Optional<PairingStateHandler> pairingStateHandler =
            this.pairingStateHandlerMap
                .values().stream().filter(psh -> uniqueId.equals(psh.getRemotePartnerLinkData().getUniqueId()))
                .findFirst();
        return pairingStateHandler
            .orElseThrow(() -> new IllegalArgumentException(
                "No pairingStateHandler found for remote partner uniqueId: " + ByteUtils.formatHexUniqueId(uniqueId)));
    }

    protected boolean handleLocalBidibUpResponse() {
        return this.roleType == RoleTypeEnum.INTERFACE;
    }

    public void addRemoteConnectionListener(final ConnectionListener remoteConnectionListener) {
        synchronized (this.remoteConnectionListeners) {

            this.remoteConnectionListeners.add(remoteConnectionListener);
        }
    }

    public void removeRemoteConnectionListener(final ConnectionListener remoteConnectionListener) {
        synchronized (this.remoteConnectionListeners) {

            this.remoteConnectionListeners.remove(remoteConnectionListener);
        }
    }

    public void setPairingStore(final PairingStore pairingStore) {
        this.pairingStore = pairingStore;
    }

    private void sendLocalLogonRejected(String contextKey, long uniqueId) {

        BidibCommandMessage localLogonRejected = bidibRequestFactory.createLocalLogonRejected(uniqueId);
        localLogonRejected.setAddr(Node.ROOTNODE_ADDR);
        sendNetBidibMessage(contextKey, localLogonRejected);

        final ChannelHandlerContext ctx = this.channelHandlerMap.get(contextKey);
        if (ctx != null) {
            LOGGER.info("Disconnect the channel, contextKey: {}", contextKey);
            ctx.channel().disconnect();
        }
        else {
            LOGGER.warn("Disconnect channel failed because the context is not registered, contextKey: {}", contextKey);
        }
    }

    private void sendNetBidibMessage(final String contextKey, final BidibCommand message) {
        LOGGER.info("Send the netBidibMessage, contextKey: {}, message: {}", contextKey, message);

        // get the correct netty context

        final ChannelHandlerContext ctx = this.channelHandlerMap.get(contextKey);
        if (ctx != null) {
            publishMessage(ctx, message);
        }
        else {
            LOGGER
                .warn("Publish message failed because the context is not registered, contextKey: {}, message: {}",
                    contextKey, message);
        }
    }

    public void processMessages(final ByteArrayOutputStream messageData, String contextKey) throws ProtocolException {
        // let the NetMessageReceiver instance process the messages
        this.netMessageReceiver.processMessages(messageData, contextKey);
    }

    public void channelRegistered(final String contextKey, final ChannelHandlerContext ctx) {
        LOGGER.info("Channel registered, contextKey: {}", contextKey);

        this.channelHandlerMap.put(contextKey, ctx);
        LOGGER.info("Store the channelHandlerContext: {}", ctx);

        // create the pairing state handler for the new contextKey
        this.pairingStateHandlerMap.put(contextKey, createPairingStateHandler(contextKey));

        this.netMessageReceiver.notifyConnectionOpened(contextKey);
    }

    public void cleanupHandlerContext(final String contextKey) {
        LOGGER.info("Cleanup the maps. Fetched the contextKey: {}", contextKey);

        if (StringUtils.isBlank(contextKey)) {
            this.channelHandlerMap.clear();
            this.pairingStateHandlerMap.clear();
        }
        else {
            this.channelHandlerMap.remove(contextKey);
            this.pairingStateHandlerMap.remove(contextKey);
        }

        // let the message receiver remove the nodes from this connection
        this.netMessageReceiver.notifyConnectionClosed(contextKey);
    }

    /**
     * Publish the bidib message to all channels.
     * 
     * @param sequenceNumberProvider
     *            the sequence number provider for the message
     * @param message
     *            the message content
     */
    protected void publishBidibMessage(
        final String contextKey, final SequenceNumberProvider sequenceNumberProvider, BidibMessageInterface message) {
        LOGGER.info("Received message to publish to the guest: {}", message);

        // get the local address of the node by the contextKey and add it to the provided address
        byte[] addr = message.getAddr();

        final Integer localNodeAddress = this.bidibMessageEvaluatorSupplier.get().getLocalNodeAddress(contextKey);
        if (localNodeAddress == null) {
            LOGGER.warn("No local node address found for contextKey: {}, message: {}", contextKey, message);
            // throw new ProtocolException("No local node address found for contextKey: " + contextKey);
            return;
        }

        if (Arrays.equals(addr, new byte[] { ByteUtils.getLowByte(localNodeAddress) })) {
            message.setAddr(Node.ROOTNODE_ADDR);
        }
        else if (!Arrays.equals(addr, Node.ROOTNODE_ADDR)) {
            message.setAddr(ByteUtils.subArray(addr, 1));
        }

        LOGGER.info("Change address of message to publish to the guest: {}", message);

        byte[] content = message.getContent();

        logTX(message, content);

        publishMessage(contextKey, content);
    }

    /**
     * Publish the bidib message to all channels.
     * 
     * @param sequenceNumberProvider
     *            the sequence number provider for the message
     * @param message
     *            the message content
     */
    protected void publishBidibMessage(
        final String contextKey, final SequenceNumberProvider sequenceNumberProvider, final byte[] message) {
        LOGGER.info("Received message to publish to the guest: {}", ByteUtils.bytesToHex(message));

        publishMessage(contextKey, message);
    }

    protected void logTX(final BidibMessageInterface bidibCommand, byte[] content) {

        if (MSG_TX_LOGGER.isInfoEnabled()) {

            StringBuilder sb = new StringBuilder(">>net>> ");
            sb.append(bidibCommand);
            sb.append(" : ");
            sb.append(ByteUtils.bytesToHex(content));

            MSG_TX_LOGGER.info(sb.toString());
        }
    }

    private void publishMessage(final String contextKey, final byte[] message) {

        // write the message content to the context / connection to the remote partner

        final ChannelHandlerContext ctx = this.channelHandlerMap.get(contextKey);

        try {
            ctx.writeAndFlush(Unpooled.copiedBuffer(message));
        }
        catch (Exception ex) {
            LOGGER.warn("Write message to channel failed.", ex);
        }
    }

    /**
     * Publish the message to the connected host.
     * 
     * @param ctx
     *            the context
     * @param message
     *            the message
     */
    private void publishMessage(final ChannelHandlerContext ctx, final BidibMessageInterface message) {
        LOGGER.info("Publish the message to channel: {}, message: {}", ctx, message);

        byte[] content = message.getContent();

        logTX(message, content);

        ctx.writeAndFlush(Unpooled.copiedBuffer(content));

        LOGGER.info("Write message to socketChannel has finished, msg: {}", ByteUtils.bytesToHex(content));
    }

    /**
     * Publish the action to the user. This will call the user to perform an action.
     */
    private void publishUserAction(String actionKey, final Context context) {
        LOGGER.info("Publish the user action, actionKey: {}, context: {}", actionKey, context);

        final List<ConnectionListener> connectionListeners = getSafeConnectionListeners();

        for (ConnectionListener connectionListener : connectionListeners) {
            try {
                connectionListener.actionRequired(actionKey, context);
            }
            catch (Exception ex) {
                LOGGER.warn("Notify that action is required failed, actionKey: {}", actionKey, ex);
            }
        }
    }

    private List<ConnectionListener> getSafeConnectionListeners() {
        final List<ConnectionListener> connectionListeners = new LinkedList<>();
        synchronized (this.remoteConnectionListeners) {
            connectionListeners.addAll(this.remoteConnectionListeners);
        }
        LOGGER.info("Publish to connectionListeners: {}", connectionListeners);

        return connectionListeners;
    }

    private void publishPairingFinished(final PairingResult pairingResult, long uniqueId) {
        LOGGER.info("Publish the pairing result: {}", pairingResult, ByteUtils.formatHexUniqueId(uniqueId));

        final List<ConnectionListener> connectionListeners = getSafeConnectionListeners();
        for (ConnectionListener connectionListener : connectionListeners) {
            try {
                connectionListener.pairingFinished(pairingResult, uniqueId);
            }
            catch (Exception ex) {
                LOGGER.warn("Notify the pairing result failed, pairingResult: {}", pairingResult, ex);
            }
        }
    }

    private void publishLogonReceived(int localNodeAddr, long uniqueId) {
        LOGGER
            .info("Publish the logon received, localNodeAddr: {}, uniqueId: {}", localNodeAddr,
                ByteUtils.formatHexUniqueId(uniqueId));

        final List<ConnectionListener> connectionListeners = getSafeConnectionListeners();
        for (ConnectionListener connectionListener : connectionListeners) {
            try {
                connectionListener.logonReceived(localNodeAddr, uniqueId);
            }
            catch (Exception ex) {
                LOGGER.warn("Notify the logon received failed.", ex);
            }
        }
    }

    private void publishLogoffReceived(long uniqueId) {
        LOGGER.info("Publish the logoff received, uniqueId: {}", ByteUtils.formatHexUniqueId(uniqueId));

        final List<ConnectionListener> connectionListeners = getSafeConnectionListeners();
        for (ConnectionListener connectionListener : connectionListeners) {
            try {
                connectionListener.logoffReceived(uniqueId);
            }
            catch (Exception ex) {
                LOGGER.warn("Notify the logoff received failed.", ex);
            }
        }
    }

    private void handleError(RuntimeException ex) {
        final List<ConnectionListener> connectionListeners = getSafeConnectionListeners();
        for (ConnectionListener connectionListener : connectionListeners) {
            try {
                connectionListener.handleError(ex);
            }
            catch (Exception ex1) {
                LOGGER.warn("Handle the error in connectionListener failed.", ex1);
            }
        }
    }

    public void setConnectionListener(ConnectionListener connectionListener) {
        LOGGER.info("Set the connection listener: {}", connectionListener);

        // add the connection listener for the remote connection to the client
        addRemoteConnectionListener(new ConnectionListener() {

            @Override
            public void actionRequired(String messageKey, Context context) {
                connectionListener.actionRequired(messageKey, context);
            }

            @Override
            public void pairingFinished(final PairingResult pairingResult, long uniqueId) {
                connectionListener.pairingFinished(pairingResult, uniqueId);
            }

            @Override
            public void logonReceived(int localNodeAddr, long uniqueId) {
                connectionListener.logonReceived(localNodeAddr, uniqueId);
            }

            @Override
            public void logoffReceived(long uniqueId) {
                connectionListener.logoffReceived(uniqueId);
            }

            @Override
            public void status(String messageKey, final Context context) {
                LOGGER.info("Connection status received, messageKey: {}, context: {}", messageKey, context);

                connectionListener.status(messageKey, context);
            }

            @Override
            public void opened(String port) {
                LOGGER.info("Remote connection opened, port: {}", port);
                connectionListener.opened(port);
            }

            @Override
            public void closed(String port) {
                LOGGER.info("The connection to the client was closed: {}", port);
                connectionListener.closed(port);
            }

            @Override
            public void stall(boolean stall) {
                connectionListener.stall(stall);
            }
        });

    }
}
