package org.bidib.jbidibc.netbidib.server;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.SocketAddress;

import org.bidib.jbidibc.messages.exception.ProtocolException;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.netty.buffer.ByteBuf;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.SimpleChannelInboundHandler;

public class NetBidibChannelInboundHandler<T> extends SimpleChannelInboundHandler<ByteBuf> {

    private static final Logger LOGGER = LoggerFactory.getLogger(NetBidibChannelInboundHandler.class);

    private final AbstractNetBidibServerHandler<T> netBidibServerHandler;

    private String contextKey;

    public NetBidibChannelInboundHandler(final AbstractNetBidibServerHandler<T> netBidibServerHandler) {
        this.netBidibServerHandler = netBidibServerHandler;
    }

    /**
     * Prepare the context key with the remote host address and the port.
     * 
     * @param remoteAddress
     *            the remote address
     * @return the context key
     */
    private static String prepareKey(final InetSocketAddress remoteAddress) {
        String key = remoteAddress.getAddress().getHostAddress() + ":" + remoteAddress.getPort();

        return key;
    }

    @Override
    public void channelRegistered(ChannelHandlerContext ctx) throws Exception {

        final InetSocketAddress remoteAddress = (InetSocketAddress) ctx.channel().remoteAddress();
        LOGGER
            .info("Session created. Remote IP: {}, port: {}, channel: {}", remoteAddress.getAddress().getHostAddress(),
                remoteAddress.getPort(), ctx.channel());

        ctx.channel().attr(ConnectionState.STATE_KEY).set(ConnectionState.Phase.NOT_CONNECTED);
        if (remoteAddress != null) {
            ctx.channel().attr(ConnectionState.REMOTE_ADDRESS_KEY).set(remoteAddress.toString());
        }

        super.channelRegistered(ctx);

        // register the channel handler context
        final String contextKey = prepareKey(remoteAddress);
        this.contextKey = contextKey;

        ctx.channel().attr(ConnectionState.CONTEXT_KEY).set(contextKey);

        this.netBidibServerHandler.channelRegistered(contextKey, ctx);
    }

    @Override
    public void channelUnregistered(final ChannelHandlerContext ctx) throws Exception {
        SocketAddress remoteAddress = ctx.channel().remoteAddress();
        LOGGER.info("Session closed. IP: {}", remoteAddress);

        if (remoteAddress == null) {
            final SocketAddress localAddress = ctx.channel().localAddress();
            LOGGER.info("No remoteAddress available. Use localAddress: {}", localAddress);

            remoteAddress = localAddress;
        }

        String connectedRemoteAddress = ctx.channel().attr(ConnectionState.REMOTE_ADDRESS_KEY).get();
        LOGGER.info("The connected remote address: {}", connectedRemoteAddress);

        // set the state and remove from channel group (by super call)
        ctx.channel().attr(ConnectionState.STATE_KEY).set(ConnectionState.Phase.NOT_CONNECTED);
        super.channelUnregistered(ctx);

        cleanupHandlerContext(ctx);

        LOGGER.info("Disconnected.");
    }

    private void cleanupHandlerContext(final ChannelHandlerContext ctx) {
        // get the channel handler context key
        final String contextKey = ctx.channel().attr(ConnectionState.CONTEXT_KEY).get();
        LOGGER.info("Cleanup the maps. Fetched the contextKey: {}", contextKey);

        this.netBidibServerHandler.cleanupHandlerContext(contextKey);

        this.contextKey = null;
    }

    @Override
    public void channelInactive(ChannelHandlerContext ctx) throws Exception {

        LOGGER.info("channelInactive, ctx: {}", ctx);

        // TODO Auto-generated method stub
        super.channelInactive(ctx);
    }

    @Override
    public void handlerAdded(ChannelHandlerContext ctx) throws Exception {

        LOGGER.info("handlerAdded, ctx: {}", ctx);
        // TODO Auto-generated method stub
        super.handlerAdded(ctx);
    }

    @Override
    public void handlerRemoved(ChannelHandlerContext ctx) throws Exception {

        LOGGER.info("handlerRemoved, ctx: {}", ctx);
        // TODO Auto-generated method stub
        super.handlerRemoved(ctx);
    }

    @Override
    public void channelReadComplete(ChannelHandlerContext ctx) throws Exception {

        LOGGER.debug("channelReadComplete.");
        super.channelReadComplete(ctx);

        // ctx.writeAndFlush(Unpooled.EMPTY_BUFFER).addListener(ChannelFutureListener.CLOSE);
    }

    @Override
    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) throws Exception {
        // LOGGER.warn("Exception in handler. Close the channelHandlerContext.", cause);
        if (cause instanceof IOException) {
            LOGGER.warn("IOException: {}", cause.getMessage());
        }
        else {
            LOGGER.warn("Exception caught in server handler. Will close connection.", cause);
        }
        ctx.channel().attr(ConnectionState.STATE_KEY).set(ConnectionState.Phase.NOT_CONNECTED);
        ctx.close();

        // LOGGER.info("Clear the info of the paired partner.");
        // synchronized (pairedPartnerLock) {
        // pairedPartner.clear(true);
        // }
    }

    // re-use the instance to keep data over packets
    private final ByteArrayOutputStream messageData = new ByteArrayOutputStream(2048);

    @Override
    public void channelRead0(ChannelHandlerContext ctx, ByteBuf in) {

        // process the data that is received from the remote partner over netBidib

        byte[] messageContent = new byte[in.readableBytes()];
        in.readBytes(messageContent);

        LOGGER.info("<<net<< {} : {}", this.contextKey, ByteUtils.bytesToHex(messageContent));

        try {

            // the message: LEN ADDR SEQ MSG_TYPE DATA
            int len = ByteUtils.getInt(messageContent[0]);
            LOGGER.debug("Current message len: {}", len);

            try {
                messageData.write(messageContent);
            }
            catch (IOException ex2) {
                LOGGER.warn("Write data to output stream failed.", ex2);
            }

            final String contextKey = ctx.channel().attr(ConnectionState.CONTEXT_KEY).get();
            this.netBidibServerHandler.processMessages(messageData, contextKey);

        }
        catch (ProtocolException ex) {
            LOGGER.warn("Create messages from provided data failed.", ex);
        }
    }

}
