package org.bidib.jbidibc.netbidib.server;

import java.net.InetSocketAddress;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.time.StopWatch;
import org.bidib.jbidibc.core.schema.BidibFactory;
import org.bidib.jbidibc.messages.ConnectionListener;
import org.bidib.jbidibc.messages.enums.PairingResult;
import org.bidib.jbidibc.messages.helpers.Context;
import org.bidib.jbidibc.messages.message.netbidib.BidibLinkData;
import org.bidib.jbidibc.messages.message.netbidib.NetBidibLinkData;
import org.bidib.jbidibc.messages.message.netbidib.NetBidibLinkData.PairingStatus;
import org.bidib.jbidibc.messages.utils.ThreadFactoryBuilder;
import org.bidib.jbidibc.netbidib.NetBidibContextKeys;
import org.bidib.jbidibc.netbidib.client.pairingstates.PairingStateHandler;
import org.bidib.jbidibc.netbidib.client.pairingstates.PairingStateInteractionHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.netty.bootstrap.ServerBootstrap;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelInitializer;
import io.netty.channel.ChannelOption;
import io.netty.channel.EventLoopGroup;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.SocketChannel;
import io.netty.channel.socket.nio.NioServerSocketChannel;
import io.netty.handler.logging.LogLevel;
import io.netty.handler.logging.LoggingHandler;
import io.netty.util.concurrent.DefaultEventExecutorGroup;
import io.netty.util.concurrent.EventExecutorGroup;
import io.netty.util.concurrent.Future;

public abstract class NetBidibNettyServer<T> {

    private static final Logger LOGGER = LoggerFactory.getLogger(NetBidibNettyServer.class);

    public static final String BIND_ADDRESS_ALL_INTERFACES = "0.0.0.0";

    public static final int DEFAULT_PORTNUM = 62875;

    public static final String DEFAULT_HOSTNAME = "localhost";

    private int portNumber = DEFAULT_PORTNUM;

    private String hostName;

    private String connectionName;

    private EventLoopGroup group;

    private EventLoopGroup workerGroup;

    private final EventExecutorGroup eventExecutorGroup = new DefaultEventExecutorGroup(3);

    private ChannelFuture channelFuture;

    private final Object shutdownLock = new Object();

    private Thread shutdownHook;

    private NetBidibLinkData serverLinkData;

    private ScheduledExecutorService serverWorkers;

    private final RoleTypeEnum roleType;

    private AbstractNetBidibServerHandler<T> netBidibServerHandler;

    public NetBidibNettyServer(final String connectionName, final NetBidibLinkData serverLinkData,
        RoleTypeEnum roleType, final AbstractNetBidibServerHandler<T> netBidibServerHandler) {
        this(DEFAULT_HOSTNAME, DEFAULT_PORTNUM, connectionName, serverLinkData, roleType, netBidibServerHandler);
    }

    /**
     * Create a {@code NetBidibServer} instance with the provided hostname and portNumber.
     * 
     * @param hostName
     *            the hostname
     * @param portNumber
     *            the port number
     */
    public NetBidibNettyServer(String hostName, int portNumber, final String connectionName,
        final NetBidibLinkData serverLinkData, final RoleTypeEnum roleType,
        final AbstractNetBidibServerHandler<T> netBidibServerHandler) {
        this.hostName = hostName;
        this.portNumber = portNumber;
        this.connectionName = connectionName;
        this.serverLinkData = serverLinkData;
        this.roleType = roleType;
        this.netBidibServerHandler = netBidibServerHandler;
    }

    public void setNetMessageReceiver(final ServerNetMessageReceiver netMessageReceiver) {
        LOGGER.info("Set the netMessageReceiver: {}", netMessageReceiver);

        this.netBidibServerHandler.setNetMessageReceiver(netMessageReceiver);
    }

    private void createServerWorkers() {

        final ThreadFactory namedThreadFactory =
            new ThreadFactoryBuilder().setNameFormat("netBidibServerWorkers-thread-%d").build();
        this.serverWorkers = Executors.newScheduledThreadPool(1, namedThreadFactory);
    }

    public String getConnectionParams() {
        return this.hostName + ":" + this.portNumber;
    }

    /**
     * @return the shutdownHook
     */
    public Thread getShutdownHook() {
        return shutdownHook;
    }

    /**
     * @param shutdownHook
     *            the shutdownHook to set
     */
    public void setShutdownHook(Thread shutdownHook) {
        this.shutdownHook = shutdownHook;
    }

    public Object getShutdownLock() {
        return shutdownLock;
    }

    /**
     * Create the {@code NetBidibServerHandler} instance. This netty channel handler wil be added as last handler to the
     * pipeline and will be called when a packet is received.
     * <p>
     * Note: Override this method to do more initialization of the {@code NetBidibServerHandler}.
     * </p>
     *
     * @param channelGroup
     *            the netty channel group
     * @return the serverHandler instance
     */
    protected abstract NetBidibChannelInboundHandler<T> createNetBidibChannelInboundHandler(
        final NetBidibLinkData serverLinkData, final String connectionName, final RoleTypeEnum roleType,
        final ConnectionListener connectionListener);

    /**
     * Start the server.
     */
    public void startServer(final ConnectionListener connectionListener) {
        LOGGER.info("Start the server.");

        this.netBidibServerHandler.setConnectionListener(connectionListener);

        final CountDownLatch startLatch = new CountDownLatch(1);

        try {
            LOGGER.info("Create ServerBootstrap with hostname: {}, portNumber: {}", hostName, portNumber);

            this.group = new NioEventLoopGroup(1);
            this.workerGroup = new NioEventLoopGroup();

            createServerWorkers();

            final ServerBootstrap serverBootstrap = new ServerBootstrap();

            serverBootstrap.group(this.group, this.workerGroup);
            serverBootstrap.channel(NioServerSocketChannel.class);
            serverBootstrap.localAddress(new InetSocketAddress(hostName, portNumber));

            serverBootstrap.option(ChannelOption.SO_BACKLOG, 100);
            serverBootstrap.childOption(ChannelOption.SO_KEEPALIVE, true);

            // serverBootstrap.option(ChannelOption.SO_REUSEADDR, true);

            serverBootstrap.childOption(ChannelOption.TCP_NODELAY, true);

            serverBootstrap.handler(new LoggingHandler(LogLevel.INFO));

            serverBootstrap.childHandler(new ChannelInitializer<SocketChannel>() {
                @Override
                protected void initChannel(final SocketChannel socketChannel) throws Exception {
                    LOGGER.info("Init the socketChannel: {}", socketChannel);

                    final NetBidibChannelInboundHandler<T> netBidibChannelInboundHandler =
                        createNetBidibChannelInboundHandler(serverLinkData, connectionName, roleType,
                            connectionListener);

                    socketChannel
                        .pipeline().addLast(eventExecutorGroup, "netBidibHandler", netBidibChannelInboundHandler);
                }
            });

            LOGGER.info("Start pre-loading the messageTypes.");

            StopWatch sw = StopWatch.createStarted();

            BidibFactory.getMessageTypes();

            sw.stop();
            LOGGER.info("Finished pre-loading the messageTypes, duration: {}ms", sw.getTime());

            final Runnable serverThread = () -> {
                try {

                    LOGGER.info("Start the TCP server.");
                    ChannelFuture channelFuture = serverBootstrap.bind().sync();

                    this.channelFuture = channelFuture;

                    try {
                        connectionListener
                            .opened(this.channelFuture.channel().remoteAddress() != null
                                ? this.channelFuture.channel().remoteAddress().toString() : "local");
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Notify that the client connection was opened failed.", ex);
                    }

                    final Channel channel = channelFuture.channel();

                    // let the starting thread know we have started
                    startLatch.countDown();

                    channel.closeFuture().sync();

                    LOGGER.info("The TCP server channel was closed.");
                }
                catch (Exception ex) {
                    LOGGER.warn("Process socket with netty failed.", ex);
                }
                finally {
                    try {
                        group.shutdownGracefully().sync();
                    }
                    catch (InterruptedException ex) {
                        LOGGER.warn("Shutdown the netty server gradefully failed.", ex);
                    }
                }

                LOGGER.info("The TCP server has finished.");
            };

            this.serverWorkers.submit(serverThread);

        }
        catch (Exception ex) {
            LOGGER.warn("Start the server failed.", ex);
        }

        try {
            startLatch.await(5, TimeUnit.SECONDS);
        }
        catch (Exception ex) {
            LOGGER.warn("Wait for startup of server failed.", ex);
        }

    }

    /**
     * Stop the server.
     */
    public void stop() {

        LOGGER.info("Stop the NetBidibServer instance.");

        if (this.netBidibServerHandler != null) {
            // notify the server handler that the connection was closed
            this.netBidibServerHandler.cleanupHandlerContext(null);
        }

        if (this.channelFuture != null) {
            LOGGER.info("Close the channel of the server.");

            try {
                this.channelFuture.channel().disconnect();
            }
            catch (Exception ex) {
                LOGGER.warn("Disconnect the channel failed.", ex);
            }

            this.channelFuture = null;
        }
        else {
            LOGGER.info("No channelFuture available.");
        }

        if (group != null) {
            Future<?> terminate = group.shutdownGracefully(0, 5, TimeUnit.SECONDS);

            LOGGER.info("Wait for termination of server.");
            try {
                terminate.await(5000);
            }
            catch (InterruptedException ex) {
                LOGGER.warn("Wait for termination of server was interrupted.");
            }
            group = null;
        }

        if (workerGroup != null) {
            Future<?> terminate = workerGroup.shutdownGracefully(0, 5, TimeUnit.SECONDS);

            LOGGER.info("Wait for termination of workerGroup.");
            try {

                terminate.await(5000);
            }
            catch (InterruptedException ex) {
                LOGGER.warn("Wait for termination of workerGroup was interrupted.");
            }
            workerGroup = null;
        }

        LOGGER.info("Shutdown the serverWorkers.");
        if (this.serverWorkers != null) {
            try {
                this.serverWorkers.shutdown();

                this.serverWorkers.awaitTermination(2000, TimeUnit.MILLISECONDS);
                this.serverWorkers = null;
            }
            catch (InterruptedException ex) {
                LOGGER.warn("Wait for termination of serverWorker failed.", ex);
            }
        }

        synchronized (shutdownLock) {
            LOGGER.info("Notify the shutdownLock.");
            shutdownLock.notifyAll();
        }
    }

    public void signalUserAction(String actionKey, final Context context) {
        LOGGER.info("signalUserAction, actionKey: {}, context: {}", actionKey, context);

        final Long uniqueId = context.get(Context.UNIQUE_ID, Long.class, null);

        if (uniqueId == null) {
            LOGGER.warn("No uniqueId delivered. Skip signal user action with actionKey: {}", actionKey);
            return;
        }

        final PairingStateHandler netBidibPairingStateHandler =
            this.netBidibServerHandler.getPairingStateHandler(uniqueId);

        switch (actionKey) {
            case NetBidibContextKeys.KEY_PAIRING_STATUS:

                final PairingStatus pairingStatus =
                    context.get(Context.PAIRING_STATUS, PairingStatus.class, PairingStatus.UNPAIRED);

                // we must get the netBidibPairingStateHandler from the NetBidibServerHandler to get the
                // remotePartnerLinkData
                final BidibLinkData remotePartnerLinkData = netBidibPairingStateHandler.getRemotePartnerLinkData();

                if (PairingStatus.PAIRING_REQUESTED == remotePartnerLinkData.getPairingStatus()) {
                    LOGGER
                        .info(
                            "The remote partner has sent the pairing request message to the server. Now send the pairing status: {}",
                            pairingStatus);

                    switch (pairingStatus) {
                        case PAIRED:
                            // send the PAIRED
                            ((PairingStateInteractionHandler) netBidibPairingStateHandler)
                                .pairingResult(uniqueId, PairingResult.PAIRED);
                            break;
                        default:
                            // send the UNPAIRED
                            ((PairingStateInteractionHandler) netBidibPairingStateHandler)
                                .pairingResult(uniqueId, PairingResult.UNPAIRED);
                            break;
                    }
                }
                else {
                    LOGGER
                        .info("The pairing status is not sent because the pairing status of the remote partner is: {}",
                            remotePartnerLinkData);
                }
                break;
            case NetBidibContextKeys.KEY_PAIRING_REQUEST:
                if (netBidibPairingStateHandler instanceof PairingStateInteractionHandler) {
                    LOGGER.info("Initiate the pairing.");
                    ((PairingStateInteractionHandler) netBidibPairingStateHandler).initiatePairing();
                }
                else {
                    LOGGER.warn("The netBidibPairingStateHandler is not of expected type. Check configuration.");
                }
                break;

            default:
                LOGGER.warn("Unhandled user action: {}, context: {}", actionKey, context);
                break;

        }

    }

}
