package org.bidib.jbidibc.netbidib.server;

import java.util.Collections;
import java.util.List;
import java.util.Set;

import org.bidib.jbidibc.core.AbstractBidib;
import org.bidib.jbidibc.core.AbstractMessageReceiver;
import org.bidib.jbidibc.core.BidibMessageProcessor;
import org.bidib.jbidibc.core.MessageListener;
import org.bidib.jbidibc.core.NodeListener;
import org.bidib.jbidibc.core.node.NodeRegistry;
import org.bidib.jbidibc.core.node.RootNode;
import org.bidib.jbidibc.core.node.listener.TransferListener;
import org.bidib.jbidibc.messages.ConnectionListener;
import org.bidib.jbidibc.messages.base.RawMessageListener;
import org.bidib.jbidibc.messages.enums.PairingResult;
import org.bidib.jbidibc.messages.exception.NoAnswerException;
import org.bidib.jbidibc.messages.exception.PortNotFoundException;
import org.bidib.jbidibc.messages.exception.PortNotOpenedException;
import org.bidib.jbidibc.messages.exception.ProtocolException;
import org.bidib.jbidibc.messages.helpers.Context;
import org.bidib.jbidibc.messages.message.BidibMessageInterface;
import org.bidib.jbidibc.messages.message.BidibRequestFactory;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NetBidibServerBidib extends AbstractBidib<ServerNetMessageReceiver> {

    private static final Logger LOGGER = LoggerFactory.getLogger(NetBidibServerBidib.class);

    private NetBidibNettyServer<BidibMessageInterface> netBidibNettyServer;

    private String requestedPortName;

    private ConnectionListener connectionListenerDelegate;

    protected NetBidibServerBidib() {
        LOGGER.info("Create instance of NetBidibServerBidib.");
    }

    /**
     * Get a new initialized instance of NetBidibServerBidib.
     *
     * @return the instance of NetBidibServerBidib
     */
    public static NetBidibServerBidib createInstance(final Context context) {

        NetBidibServerBidib instance = new NetBidibServerBidib();
        instance.initialize(context);

        return instance;
    }

    @Override
    protected ServerNetMessageReceiver createMessageReceiver(
        final NodeRegistry nodeRegistry, final RawMessageListener rawMessageListener, final Context context) {

        // the netbidib does not need CRC check
        ServerNetMessageReceiver messageReceiver =
            new ServerNetMessageReceiver(nodeRegistry, nodeRegistry.getRequestFactory(), false);
        messageReceiver.setRawMessageListener(rawMessageListener);
        messageReceiver.init(context);
        return messageReceiver;
    }

    @Override
    protected BidibRequestFactory createRequestFactory() {
        final BidibRequestFactory requestFactory = super.createRequestFactory();
        requestFactory.setEscapeMagic(false);
        return requestFactory;
    }

    @Override
    public void initialize(final Context context) {
        LOGGER.info("Initialize. Current context: {}", context);

        super.initialize(context);
    }

    @Override
    public void open(
        String portName, final ConnectionListener connectionListener, final Set<NodeListener> nodeListeners,
        final Set<MessageListener> messageListeners, final Set<TransferListener> transferListeners,
        final Context context) throws PortNotFoundException, PortNotOpenedException {

        LOGGER.info("Open is called, portName: {}", portName);
        this.requestedPortName = portName;

        setConnectionListener(connectionListener);

        // register the listeners
        registerListeners(nodeListeners, messageListeners, transferListeners, context);

        LOGGER.info("Get the NetBidibNettyServer instance from the context.");

        final NetBidibNettyServer<BidibMessageInterface> netBidibNettyServer =
            context.get(NetBidibNettyServer.class.getSimpleName(), NetBidibNettyServer.class, null);

        this.netBidibNettyServer = netBidibNettyServer;
        this.netBidibNettyServer.setNetMessageReceiver(getMessageReceiver());

        this.netBidibNettyServer.startServer(this.connectionListenerDelegate);

    }

    @Override
    public void close() {
        LOGGER.info("Close the port.");

        if (this.netBidibNettyServer != null) {

            LOGGER.info("Stop the netBidibJettyServer.");
            this.netBidibNettyServer.stop();
            this.netBidibNettyServer = null;
            LOGGER.info("Stop the netBidibJettyServer has finished.");

            super.close();

            if (getConnectionListener() != null) {
                LOGGER.info("Notify that the port was closed: {}", requestedPortName);
                getConnectionListener().closed(requestedPortName);
            }
            else {
                LOGGER
                    .info("No connection listener available to publish the closed report for port: {}",
                        requestedPortName);
            }

            cleanupAfterClose(getMessageReceiver());

            LOGGER.info("Close the port has finished.");
        }

    }

    @Override
    public boolean isOpened() {
        return this.netBidibNettyServer != null;
    }

    @Override
    public List<String> getPortIdentifiers() {
        return Collections.emptyList();
    }

    @Override
    protected void contactInterface() {
        LOGGER.info("Contact the interface node.");

        try {
            sendDisableAndEnableMessageReceiver();

            LOGGER.info("Sent disable to the root node.");
        }
        catch (ProtocolException ex) {
            throw new NoAnswerException("Contact the interface node failed.", ex);
        }
    }

    /**
     * Send disable the root node.
     * 
     * @throws ProtocolException
     */
    private void sendDisableAndEnableMessageReceiver() throws ProtocolException {
        RootNode rootNode = getRootNode();

        LOGGER.info("Send sysDisable to the rootNode.");
        rootNode.sysDisable();

        try {
            LOGGER.info("Wait 300ms before enable the message receiver.");
            Thread.sleep(300);
        }
        catch (InterruptedException ex) {
            LOGGER.warn("Wait before enable the message receiver failed.", ex);
        }

        final BidibMessageProcessor serialMessageReceiver = getBidibMessageProcessor();

        if (serialMessageReceiver instanceof AbstractMessageReceiver) {
            LOGGER.info("Enable the message receiver: {}", serialMessageReceiver);
            // enable the message receiver
            ((AbstractMessageReceiver) serialMessageReceiver).enable();
        }
    }

    @Override
    public void send(byte[] data) {

        // TODO check if the root node address is the target of this message
        getMessageReceiver().processMessageFromHost(data);

    }

    @Override
    public void signalUserAction(String actionKey, final Context context) {
        LOGGER.info("Signal the user action, actionKey: {}, context: {}", actionKey, context);

        this.netBidibNettyServer.signalUserAction(actionKey, context);

    }

    @Override
    public void setConnectionListener(final ConnectionListener connectionListener) {

        this.connectionListenerDelegate = new ConnectionListener() {

            @Override
            public void status(String messageKey, final Context context) {
                LOGGER.info("The status was signalled: {}, context: {}", messageKey, context);
                connectionListener.status(messageKey, context);
            }

            @Override
            public void actionRequired(String messageKey, final Context context) {
                connectionListener.actionRequired(messageKey, context);
            }

            @Override
            public void opened(String port) {
                LOGGER.info("The port of the netBiDiB connection was opened: {}", port);

                connectionListener.opened(port);

                LOGGER.info("The port was opened. Contact the interface.");
                try {
                    contactInterface();
                }
                catch (RuntimeException ex) {
                    LOGGER.warn("Contact interface failed. Abort signal opened.", ex);

                    throw ex;
                }
            }

            @Override
            public void closed(String portName) {
                LOGGER.info("The port was closed: {}", portName);

                connectionListener.closed(portName);
            }

            @Override
            public void stall(boolean stall) {
                connectionListener.stall(stall);
            }

            @Override
            public void pairingFinished(PairingResult pairingResult, long uniqueId) {
                LOGGER
                    .info("The pairing was finished, pairingResult: {}, uniqueId: {}", pairingResult,
                        ByteUtils.formatHexUniqueId(uniqueId));

                connectionListener.pairingFinished(pairingResult, uniqueId);
            }

            @Override
            public void logonReceived(int localNodeAddr, long uniqueId) {
                LOGGER
                    .info("The logon was received, localNodeAddr: {}, uniqueId: {}", localNodeAddr,
                        ByteUtils.formatHexUniqueId(uniqueId));

                // TODO wait for receipt of MSG_LOCAL_LOGON, use timer to detect timeout
                connectionListener.logonReceived(localNodeAddr, uniqueId);
            }

            @Override
            public void logoffReceived(long uniqueId) {
                LOGGER.info("The logoff was received, uniqueId: {}", ByteUtils.formatHexUniqueId(uniqueId));
                connectionListener.logoffReceived(uniqueId);
            }
        };

        super.setConnectionListener(this.connectionListenerDelegate);
    }
}
