package org.bidib.jbidibc.netbidib.server;

import java.util.Arrays;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.StringUtils;
import org.bidib.jbidibc.core.BidibMessageEvaluator;
import org.bidib.jbidibc.core.node.BidibNode;
import org.bidib.jbidibc.core.node.NodeRegistry;
import org.bidib.jbidibc.messages.Node;
import org.bidib.jbidibc.messages.exception.ProtocolException;
import org.bidib.jbidibc.messages.message.BidibMessageInterface;
import org.bidib.jbidibc.messages.message.RequestFactory;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.jbidibc.messages.utils.NodeUtils;
import org.bidib.jbidibc.messages.utils.ThreadFactoryBuilder;
import org.bidib.jbidibc.netbidib.client.NetMessageReceiver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ServerNetMessageReceiver extends NetMessageReceiver {

    private static final Logger LOGGER = LoggerFactory.getLogger(ServerNetMessageReceiver.class);

    private final ScheduledExecutorService rootNodeWorkers;

    public ServerNetMessageReceiver(final NodeRegistry nodeRegistry, final RequestFactory requestFactory,
        boolean checkCRC) {
        super(nodeRegistry, requestFactory, checkCRC);

        LOGGER.info("Create new instance of ServerNetMessageReceiver.");

        final ThreadFactory namedThreadFactory =
            new ThreadFactoryBuilder().setNameFormat("rootNodeWorkers-thread-%d").build();
        rootNodeWorkers = Executors.newScheduledThreadPool(1, namedThreadFactory);
    }

    @Override
    public void cleanup() {
        LOGGER.info("Cleanup the ServerNetMessageReceiver.");
        super.cleanup();

        try {
            this.rootNodeWorkers.shutdownNow();
            this.rootNodeWorkers.awaitTermination(100, TimeUnit.MILLISECONDS);
        }
        catch (Exception ex) {
            LOGGER.warn("Shutdown rootNodeWorkers failed.", ex);
        }

    }

    public void processMessageFromHost(final byte[] data) {
        LOGGER.info("Process the message from the host: {}", ByteUtils.bytesToHex(data));

        if (data.length > 2) {

            this.rootNodeWorkers.submit(() -> {
                LOGGER.info("Create messages from data: {}", ByteUtils.bytesToHex(data));

                try {
                    List<BidibMessageInterface> messages = getRequestFactory().create(data);
                    for (BidibMessageInterface message : messages) {
                        LOGGER.info("Current message: {}", message);

                        final BidibNode bidibNode = getNodeRegistry().findNode(message.getAddr());

                        getBidibMessageEvaluator().processBidibMessageFromHost(message, bidibNode, response -> {
                            logRX(message, message.getContent());
                            processMessageToHost(response, "");
                        });
                    }
                }
                catch (Exception ex) {
                    LOGGER.warn("Create request for root node failed.", ex);
                }
            });
        }
        else {
            LOGGER.warn("Invalid message part detected that is discarded: {}", ByteUtils.bytesToHex(data));
        }

    }

    public void notifyConnectionOpened(final String contextKey) {
        LOGGER.info("The connection was opened, contextKey: {}", contextKey);

        getBidibMessageEvaluator().connectionOpened(null);
    }

    public void notifyConnectionClosed(final String contextKey) {
        LOGGER.info("The connection was closed, contextKey: {}", contextKey);

        if (StringUtils.isBlank(contextKey)) {
            getBidibMessageEvaluator().connectionClosed(null);
        }
        else {
            // Cleanup nodes from this connection
            getBidibMessageEvaluator().connectionClosed(contextKey, message -> {
                logRX(message, message.getContent());
                processMessageToHost(message, contextKey);
            });
        }
    }

    // @Override
    // protected void logRX(final BidibMessageInterface bidibCommand, byte[] messageArray) {
    //
    // if (MSG_RX_NET_LOGGER.isInfoEnabled()) {
    // MSG_RX_NET_LOGGER.info("<<net<< {} : {}", bidibCommand, ByteUtils.bytesToHex(messageArray));
    // }
    //
    // if (MSG_RX_LOGGER.isInfoEnabled()) {
    //
    // StringBuilder sb = new StringBuilder("<<net<< ");
    // sb.append(bidibCommand);
    // sb.append(" : ");
    // sb.append(ByteUtils.bytesToHex(messageArray));
    //
    // MSG_RX_LOGGER.info(sb.toString());
    // }
    // }

    @Override
    protected void doProcessMessage(
        final BidibMessageInterface bidibCommand, String contextKey, final byte[] messageArray)
        throws ProtocolException {
        LOGGER.info("Processing BiDiB node related command: {}", bidibCommand);

        // get the local address of the node by the contextKey and add it to the provided address

        final BidibMessageEvaluator bidibMessageEvaluator = getBidibMessageEvaluator();
        final Integer localNodeAddress = bidibMessageEvaluator.getLocalNodeAddress(contextKey);

        if (localNodeAddress == null) {
            LOGGER.warn("No local node address found for contextKey: {}, bidibCommand: {}", contextKey, bidibCommand);
            throw new ProtocolException("No local node address found for contextKey: " + contextKey);
        }

        byte[] addr = null;
        if (Arrays.equals(bidibCommand.getAddr(), Node.ROOTNODE_ADDR)) {
            addr = new byte[] { ByteUtils.getLowByte(localNodeAddress) };
        }
        else {
            addr = ByteUtils.prepend(ByteUtils.getLowByte(localNodeAddress), bidibCommand.getAddr());
        }
        LOGGER.info("Prepared the new address for the message: {}", NodeUtils.formatAddress(addr));

        bidibCommand.setAddr(addr);

        LOGGER.info("Continue processing BiDiB node related command: {}", bidibCommand);

        super.doProcessMessage(bidibCommand, contextKey, messageArray);
    }

}
