package org.bidib.jbidibc.purejavacomm;

import java.util.List;

import org.bidib.jbidibc.core.BidibInterface;
import org.bidib.jbidibc.core.MessageReceiver;
import org.bidib.jbidibc.core.base.DataTransferStatusListener;
import org.bidib.jbidibc.core.helpers.Context;
import org.bidib.jbidibc.serial.AbstractSerialBidib;
import org.bidib.jbidibc.serial.LineStatusListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This is the default bidib implementation. It creates and initializes the MessageReceiver and the NodeFactory that is
 * used in the system.
 */
public final class PureJavaCommSerialBidib extends AbstractSerialBidib {

    private static final Logger LOGGER = LoggerFactory.getLogger(PureJavaCommSerialBidib.class);

    private PureJavaCommSerialConnector connector;

    private PureJavaCommSerialBidib() {
    }

    /**
     * Get a new initialized instance of PureJavaCommSerialBidib.
     *
     * @return the instance of PureJavaCommSerialBidib
     */
    public static BidibInterface createInstance() {
        LOGGER.info("Create new instance of PureJavaCommSerialBidib.");

        PureJavaCommSerialBidib instance = new PureJavaCommSerialBidib();
        instance.initialize();

        return instance;
    }

    @Override
    public void initialize() {
        LOGGER.info("Initialize. Create the connector.");
        super.initialize();

        connector = new PureJavaCommSerialConnector();

        final MessageReceiver serialMessageReceiver = getMessageReceiver();
        connector.setMessageReceiver(serialMessageReceiver);
        connector.setLineStatusListener(new LineStatusListener() {

            @Override
            public void notifyLineStatusChanged(boolean ready) {
                fireCtsChanged(ready);
            }
        });

        connector.setDataTransferStatusListener(new DataTransferStatusListener() {

            @Override
            public void notifySendStarted() {
                fireSendStarted();
            }

            @Override
            public void notifySendStopped() {
                fireSendStopped();
            }

            @Override
            public void notifyReceiveStarted() {
                fireReceiveStarted();
            }

            @Override
            public void notifyReceiveStopped() {
                fireReceiveStopped();
            }

        });
        connector.initialize();
    }

    @Override
    public List<String> getPortIdentifiers() {
        return connector.getPortIdentifiers();
    }

    @Override
    protected boolean isImplAvaiable() {
        return connector.isImplAvaiable();
    }

    @Override
    protected void internalOpen(String portName, final Context context) throws Exception {

        connector.internalOpen(portName, context);
    }

    @Override
    public void close() {
        LOGGER.info("Close is called.");

        if (connector.close()) {
            cleanupAfterClose(getMessageReceiver());
        }
    }

    @Override
    public boolean isOpened() {
        boolean isOpened = false;
        try {
            isOpened = connector.isOpened();
        }
        catch (Exception ex) {
            LOGGER.warn("The connector is not available.", ex);
        }
        return isOpened;
    }

    @Override
    public void send(byte[] data) {
        connector.send(data);
    }
}
