package org.bidib.jbidibc.serial;

import java.io.DataOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.Set;
import java.util.TooManyListenersException;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;

import org.bidib.jbidibc.core.AbstractBidib;
import org.bidib.jbidibc.core.BidibInterface;
import org.bidib.jbidibc.core.BidibMessageProcessor;
import org.bidib.jbidibc.core.ConnectionListener;
import org.bidib.jbidibc.core.MessageListener;
import org.bidib.jbidibc.core.NodeListener;
import org.bidib.jbidibc.core.exception.InvalidConfigurationException;
import org.bidib.jbidibc.core.exception.NoAnswerException;
import org.bidib.jbidibc.core.exception.PortNotFoundException;
import org.bidib.jbidibc.core.exception.PortNotOpenedException;
import org.bidib.jbidibc.core.exception.ProtocolException;
import org.bidib.jbidibc.core.exception.ProtocolNoAnswerException;
import org.bidib.jbidibc.core.helpers.Context;
import org.bidib.jbidibc.core.node.BidibNode;
import org.bidib.jbidibc.core.node.NodeFactory;
import org.bidib.jbidibc.core.node.listener.TransferListener;
import org.bidib.jbidibc.core.utils.ByteUtils;
import org.bidib.jbidibc.serial.exception.InvalidLibraryException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gnu.io.CommPortIdentifier;
import gnu.io.CommPortOwnershipListener;
import gnu.io.NoSuchPortException;
import gnu.io.PortInUseException;
import gnu.io.SerialPort;
import gnu.io.SerialPortEvent;
import gnu.io.SerialPortEventListener;
import gnu.io.UnsupportedCommOperationException;

/**
 * This is the default bidib implementation. It creates and initializes the MessageReceiver and the NodeFactory that is
 * used in the system.
 * 
 */
public final class Bidib extends AbstractBidib {

    private static final Logger LOGGER = LoggerFactory.getLogger(Bidib.class);

    private static final Logger MSG_RAW_LOGGER = LoggerFactory.getLogger("RAW");

    private static final Logger MSG_OUTPUTSTREAM_LOGGER = LoggerFactory.getLogger("OPS");

    private SerialPort port;

    private Semaphore portSemaphore = new Semaphore(1);

    private Semaphore sendSemaphore = new Semaphore(1);

    private static Bidib instance;

    private String requestedPortName;

    private CommPortOwnershipListener commPortOwnershipListener;

    private CommPortIdentifier commPort;

    static {
        Runtime.getRuntime().addShutdownHook(new Thread() {
            public void run() {
                try {
                    LOGGER.debug("Close the communication ports and perform cleanup.");
                    getInstance().close();
                }
                catch (Exception e) { // NOSONAR
                    // ignore
                }
            }
        });
    }

    private Bidib() {
    }

    @Override
    protected BidibMessageProcessor createMessageReceiver(NodeFactory nodeFactory) {
        return new SerialMessageReceiver(nodeFactory);
    }

    private SerialMessageReceiver getSerialMessageReceiver() {
        return (SerialMessageReceiver) getMessageReceiver();
    }

    public static synchronized BidibInterface getInstance() {
        if (instance == null) {
            instance = new Bidib();
            instance.initialize();
        }
        return instance;
    }

    @Override
    public void close() {
        if (port != null) {
            LOGGER.info("Close the port.");
            long start = System.currentTimeMillis();

            // no longer process received messages
            getSerialMessageReceiver().disable();

            // this makes the close operation faster ...
            try {
                port.removeEventListener();
                // port.enableReceiveTimeout(200);
            }
            catch (Exception e) {
                LOGGER.warn("Remove event listener failed.", e);
            }

            // stop the send queue worker
            stopSendQueueWorker();

            try {
                port.close();
            }
            catch (Exception e) {
                LOGGER.warn("Close port failed.", e);
            }

            try {
                port.getInputStream().close();
            }
            catch (Exception e) {
                LOGGER.warn("Close input stream on port failed.", e);
            }

            try {
                port.getOutputStream().close();
            }
            catch (Exception e) {
                LOGGER.warn("Close output stream on port failed.", e);
            }

            long end = System.currentTimeMillis();
            LOGGER.info("Closed the port. duration: {}", end - start);

            if (commPortOwnershipListener != null) {
                LOGGER.info("Remove the PortOwnershipListener from commPort: {}", commPort);
                try {
                    commPort.removePortOwnershipListener(commPortOwnershipListener);
                }
                catch (Exception ex) {
                    LOGGER.warn("Remove port ownership listener failed.", ex);
                }
                commPortOwnershipListener = null;
            }

            port = null;
            commPort = null;

            if (getNodeFactory() != null) {
                // remove all stored nodes from the node factory
                getNodeFactory().reset();
            }

            InvalidConfigurationException ice = null;
            if (getMessageReceiver() != null) {
                getSerialMessageReceiver().clearMessageListeners();
                getSerialMessageReceiver().clearNodeListeners();

                try {
                    getSerialMessageReceiver().purgeOutputStream();
                }
                catch (InvalidConfigurationException ex) {
                    LOGGER.warn("Purge output stream has signaled an error.", ex);

                    if ("debug-interface-active".equals(ex.getReason())) {
                        ice = ex;
                    }
                }
            }

            if (getConnectionListener() != null) {
                getConnectionListener().closed(requestedPortName);
            }

            requestedPortName = null;

            if (ice != null) {
                LOGGER.warn("Signal the invalid configuration exception to the caller.");
                throw ice;
            }
        }
    }

    @Override
    public List<String> getPortIdentifiers() {
        List<String> portIdentifiers = new ArrayList<String>();

        try {
            // get the comm port identifiers
            Enumeration<?> e = CommPortIdentifier.getPortIdentifiers();
            while (e.hasMoreElements()) {
                CommPortIdentifier id = (CommPortIdentifier) e.nextElement();
                LOGGER.debug("Process current CommPortIdentifier, name: {}, portType: {}", id.getName(),
                    id.getPortType());

                if (id.getPortType() == CommPortIdentifier.PORT_SERIAL) {
                    portIdentifiers.add(id.getName());
                }
                else {
                    LOGGER.debug("Skip port because no serial port, name: {}, portType: {}", id.getName(),
                        id.getPortType());
                }
            }
        }
        catch (UnsatisfiedLinkError ule) {
            LOGGER.warn("Get comm port identifiers failed.", ule);
            throw new InvalidLibraryException(ule.getMessage(), ule.getCause());
        }
        catch (Error error) {
            LOGGER.warn("Get comm port identifiers failed.", error);
            throw new RuntimeException(error.getMessage(), error.getCause());
        }
        return portIdentifiers;
    }

    private void clearInputStream(SerialPort serialPort) {

        // get and clear stream
        final byte[] inputBuffer = new byte[1024];

        LOGGER.info("Clear the input stream of port: {}", serialPort);

        try {
            InputStream serialStream = serialPort.getInputStream();
            // purge contents, if any
            int count = serialStream.available();
            LOGGER.info("The input stream shows {} bytes available.", count);

            while (count > 0) {
                count = serialStream.read(inputBuffer);
                if (count > 0) {
                    LOGGER.info("Cleared data from input stream: {}", ByteUtils.bytesToHex(inputBuffer, count));
                }
            }
            LOGGER.info("The input stream shows {} bytes available after purge.", count);
        }
        catch (Exception e) {
            LOGGER.warn("Clear input stream failed.", e);
        }

    }

    private static final int DEFAULT_INPUT_BUFFER_SIZE = 8192;

    private static final int DEFAULT_OUTPUT_BUFFER_SIZE = 4096;

    private SerialPort internalOpen(CommPortIdentifier commPort, int baudRate, Context context)
        throws PortInUseException, UnsupportedCommOperationException, TooManyListenersException {

        if (commPortOwnershipListener == null) {
            commPortOwnershipListener = new CommPortOwnershipListener() {

                @Override
                public void ownershipChange(int type) {
                    LOGGER.info("Ownership changed, type: {}", type);
                }
            };
        }
        commPort.addPortOwnershipListener(commPortOwnershipListener);

        startSendQueueWorker();

        // open the port
        final SerialPort serialPort = (SerialPort) commPort.open(Bidib.class.getName(), 2000);

        LOGGER.info("Set flow control mode to SerialPort.FLOWCONTROL_RTSCTS_IN | SerialPort.FLOWCONTROL_RTSCTS_OUT!");
        serialPort.setFlowControlMode(SerialPort.FLOWCONTROL_RTSCTS_IN | SerialPort.FLOWCONTROL_RTSCTS_OUT);

        serialPort.setInputBufferSize(DEFAULT_INPUT_BUFFER_SIZE);
        serialPort.setOutputBufferSize(DEFAULT_OUTPUT_BUFFER_SIZE);

        serialPort.setSerialPortParams(baudRate, SerialPort.DATABITS_8, SerialPort.STOPBITS_1, SerialPort.PARITY_NONE);

        clearInputStream(serialPort);

        if (context != null) {
            Boolean ignoreWrongMessageNumber =
                context.get("ignoreWrongReceiveMessageNumber", Boolean.class, Boolean.FALSE);
            getSerialMessageReceiver().setIgnoreWrongMessageNumber(ignoreWrongMessageNumber);
        }

        // Activate DTR
        try {
            LOGGER.info("Activate DTR.");

            serialPort.setDTR(true); // pin 1 in DIN8; on main connector, this is DTR
        }
        catch (Exception e) {
            LOGGER.warn("Set DTR true failed.", e);
        }

        // enable the message receiver before the event listener is added
        getSerialMessageReceiver().enable();

        SerialPortEventListener eventListener = new SerialPortEventListener() {

            @Override
            public void serialEvent(SerialPortEvent event) {
                // this callback is called every time data is available
                LOGGER.trace("serialEvent received: {}", event);
                switch (event.getEventType()) {
                    case SerialPortEvent.DATA_AVAILABLE:
                        MSG_OUTPUTSTREAM_LOGGER.info("<<<<");
                        try {
                            ((SerialMessageReceiver) getMessageReceiver()).receive(serialPort.getInputStream());
                        }
                        catch (Exception ex) {
                            LOGGER.error("Message receiver has terminated with an exception!", ex);
                        }
                        break;
                    case SerialPortEvent.OUTPUT_BUFFER_EMPTY:
                        LOGGER.trace("The output buffer is empty.");

                        // only the sendQueueWorker must release the semaphore
                        if (sendQueueWorkerThreadId.get() == Thread.currentThread().getId()) {
                            MSG_OUTPUTSTREAM_LOGGER.info(">>>> OBE");
                            sendSemaphore.release();
                        }
                        break;
                    case SerialPortEvent.CD:
                        LOGGER.warn("CD is signalled.");
                        break;
                    case SerialPortEvent.CTS:
                        LOGGER.warn("The CTS value has changed, old value: {}, new value: {}",
                            new Object[] { event.getOldValue(), event.getNewValue() });

                        if (event.getNewValue() == false) {
                            LOGGER.warn("Close the port.");
                            Thread worker = new Thread(new Runnable() {
                                public void run() {
                                    LOGGER.info("Start close port because error was detected.");
                                    try {
                                        // the listeners are notified in close()
                                        close();
                                    }
                                    catch (Exception ex) {
                                        LOGGER.warn("Close after error failed.", ex);
                                    }
                                    LOGGER.warn("The port was closed.");
                                }
                            });
                            worker.start();
                        }
                        break;
                    case SerialPortEvent.OE:
                        LOGGER.warn("OE (overrun error) is signalled.");
                        break;
                    case SerialPortEvent.DSR:
                        LOGGER.warn("DSR is signalled.");
                        break;
                    default:
                        LOGGER.warn("SerialPortEvent was triggered, type: {}, old value: {}, new value: {}",
                            new Object[] { event.getEventType(), event.getOldValue(), event.getNewValue() });
                        break;
                }
            }
        };
        serialPort.addEventListener(eventListener);

        LOGGER.info("Let the serial port notify data available.");
        serialPort.notifyOnDataAvailable(true);

        // react on port removed ...
        serialPort.notifyOnCTS(true);
        serialPort.notifyOnCarrierDetect(true);
        serialPort.notifyOnBreakInterrupt(true);
        serialPort.notifyOnDSR(true);
        serialPort.notifyOnOverrunError(true);
        serialPort.notifyOnOutputEmpty(true);

        serialPort.setRTS(true);

        return serialPort;
    }

    @Override
    public void open(
        String portName, ConnectionListener connectionListener, Set<NodeListener> nodeListeners,
        Set<MessageListener> messageListeners, Set<TransferListener> transferListeners, Context context)
        throws PortNotFoundException, PortNotOpenedException {

        setConnectionListener(connectionListener);

        // register the listeners
        registerListeners(nodeListeners, messageListeners, transferListeners);

        if (port == null) {
            if (portName == null || portName.trim().isEmpty()) {
                throw new PortNotFoundException("");
            }

            LOGGER.info("Open port with name: {}", portName);

            File file = new File(portName);

            if (file.exists()) {
                try {
                    portName = file.getCanonicalPath();
                    LOGGER.info("Changed port name to: {}", portName);
                }
                catch (IOException ex) {
                    throw new PortNotFoundException(portName);
                }
            }

            try {
                commPort = CommPortIdentifier.getPortIdentifier(portName);
            }
            catch (NoSuchPortException ex) {
                LOGGER.warn("Requested port is not available: {}", portName, ex);
                throw new PortNotFoundException(portName);
            }

            requestedPortName = portName;

            try {
                portSemaphore.acquire();

                try {
                    // 115200 Baud
                    close();

                    // open the commPort
                    port = internalOpen(commPort, 115200, context);

                    LOGGER.info("The port was opened internally, get the magic.");
                    int magic = sendResetAndMagic();
                    LOGGER.info("The root node returned the magic: {}", ByteUtils.magicToHex(magic));
                }
                catch (PortInUseException ex) {
                    LOGGER.warn("Open communication failed  because port is in use.", ex);
                    try {
                        close();
                    }
                    catch (Exception e4) { // NOSONAR
                        // ignore
                    }
                    throw new PortNotOpenedException(portName, PortNotOpenedException.PORT_IN_USE);
                }
                catch (NoAnswerException naex) {
                    LOGGER.warn("Open communication failed.", naex);
                    try {
                        close();
                    }
                    catch (Exception e4) { // NOSONAR
                        // ignore
                    }
                    throw naex;
                }
                catch (ProtocolNoAnswerException naex) {
                    LOGGER.warn("Open communication failed.", naex);

                    try {
                        close();
                    }
                    catch (InvalidConfigurationException e4) { // NOSONAR
                        LOGGER.info("Rethrow the InvalidConfigurationException.", e4);
                        throw e4;
                    }
                    catch (Exception e4) { // NOSONAR
                        // ignore
                    }

                    throw new NoAnswerException(naex.getMessage());
                }
                catch (Exception e2) {
                    LOGGER.info("Open port failed. Close port and throw exception.", e2);
                    try {
                        close();
                    }
                    catch (Exception e3) { // NOSONAR
                        LOGGER.warn("Close port failed.", e3);
                    }
                    throw new PortNotOpenedException(portName, PortNotOpenedException.UNKNOWN);
                }
            }
            catch (InterruptedException ex) {
                LOGGER.warn("Wait for portSemaphore was interrupted.", ex);
                throw new PortNotOpenedException(portName, PortNotOpenedException.UNKNOWN);
            }
            finally {
                portSemaphore.release();
            }
        }
        else {
            LOGGER.warn("Port is already opened.");
        }
    }

    @Override
    public boolean isOpened() {
        boolean isOpened = false;
        try {
            portSemaphore.acquire();

            LOGGER.debug("Check if port is opened: {}", port);
            isOpened = (port != null && port.getOutputStream() != null);
        }
        catch (InterruptedException ex) {
            LOGGER.warn("Wait for portSemaphore was interrupted.", ex);
        }
        catch (IOException ex) {
            LOGGER.warn("OutputStream is not available.", ex);
        }
        finally {
            portSemaphore.release();
        }
        return isOpened;
    }

    private BlockingQueue<byte[]> sendQueue = new LinkedBlockingQueue<>();

    private Thread sendQueueWorker;

    private AtomicBoolean running = new AtomicBoolean();

    private AtomicLong sendQueueWorkerThreadId = new AtomicLong();

    /**
     * Send the bytes to the outputstream.
     * 
     * @param bytes
     *            the bytes to send
     */
    @Override
    public void send(final byte[] bytes) {

        boolean added = sendQueue.offer(bytes);
        if (!added) {
            LOGGER.error("The message was not added to the send queue: {}", ByteUtils.bytesToHex(bytes));
        }
    }

    private void startSendQueueWorker() {
        running.set(true);

        LOGGER.info("Start the sendQueueWorker. Current sendQueueWorker: {}", sendQueueWorker);
        sendQueueWorker = new Thread(new Runnable() {

            @Override
            public void run() {
                try {
                    processSendQueue();
                }
                catch (Exception ex) {
                    LOGGER.warn("The processing of the send queue was terminated with an exception!", ex);

                    running.set(false);
                }

                LOGGER.info("Process send queue has finished.");
            }
        }, "sendQueueWorker");

        try {
            sendQueueWorkerThreadId.set(sendQueueWorker.getId());
            sendQueueWorker.start();
        }
        catch (Exception ex) {
            LOGGER.error("Start the sendQueueWorker failed.", ex);
        }
    }

    private void stopSendQueueWorker() {
        LOGGER.info("Stop the send queue worker.");
        running.set(false);

        try {
            sendQueueWorker.interrupt();

            sendQueueWorker.join(1000);

            LOGGER.info("sendQueueWorker has finished.");
        }
        catch (Exception ex) {
            LOGGER.warn("Interrupt sendQueueWorker failed.", ex);
        }
        sendQueueWorker = null;

        int availablePermits = sendSemaphore.availablePermits();
        LOGGER.info("The number of availablePermits after stop send queue worker: {}", availablePermits);
        if (availablePermits < 1) {
            sendSemaphore.release();
        }
    }

    private long totalMessagesSentCounter;

    private void processSendQueue() {
        byte[] bytes = null;
        LOGGER.info("The sendQueueWorker is ready for processing.");
        while (running.get()) {
            try {

                bytes = sendQueue.take();
            }
            catch (InterruptedException ex) {
                LOGGER.warn("Get message from sendQueue failed because thread was interrupted.");
            }
            catch (Exception ex) {
                LOGGER.warn("Get message from sendQueue failed.", ex);
            }

            if (port != null && bytes != null) {
                try {
                    // keep the sendSemaphore until the output buffer is empty
                    sendSemaphore.acquire();

                    if (MSG_RAW_LOGGER.isInfoEnabled()) {
                        MSG_RAW_LOGGER.info(">> [{}] - {}", bytes.length, ByteUtils.bytesToHex(bytes));
                    }

                    totalMessagesSentCounter++;
                    MSG_OUTPUTSTREAM_LOGGER.info(">>>> {}", totalMessagesSentCounter);

                    DataOutputStream output = new DataOutputStream(port.getOutputStream());

                    output.write(bytes);
                    output.flush();
                }
                catch (Exception ex) {
                    LOGGER.warn("Send message to output stream failed: [{}] - {}", bytes.length,
                        ByteUtils.bytesToHex(bytes));

                    sendSemaphore.release();

                    throw new RuntimeException("Send message to output stream failed: " + ByteUtils.bytesToHex(bytes),
                        ex);
                }
                finally {
                    // sendSemaphore.release();
                }
            }
        }
        LOGGER.info("The sendQueueWorker has finished processing.");
        sendQueueWorkerThreadId.set(0);
    }

    /**
     * Get the magic from the root node
     * 
     * @return the magic provided by the root node
     * @throws ProtocolException
     */
    private int sendResetAndMagic() throws ProtocolException {
        BidibNode rootNode = getRootNode();

        LOGGER.info("Send reset to the rootNode.");
        rootNode.reset();

        try {
            LOGGER.info("Wait 500ms before send the magic request.");
            Thread.sleep(500);
        }
        catch (InterruptedException ex) {
            LOGGER.warn("Wait before send the magic request failed.", ex);
        }

        int magic = rootNode.getMagic();

        LOGGER.debug("The node returned magic: {}", magic);
        return magic;
    }

    /**
     * Set the response timeout for the port.
     * 
     * @param timeout
     *            the receive timeout to set
     */
    @Override
    public void setResponseTimeout(int timeout) {
        // if (port != null) {
        // LOGGER.info("Set the response timeout for the serial port: {}", timeout);
        // try {
        // port.enableReceiveTimeout(timeout);
        // }
        // catch (Exception ex) {
        // LOGGER.warn("Set the receive timeout on port failed.", ex);
        // throw new RuntimeException(ex);
        // }
        // }
    }
}
