package org.bidib.jbidibc.serial.scm;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;

import org.bidib.jbidibc.core.AbstractBidib;
import org.bidib.jbidibc.core.BidibInterface;
import org.bidib.jbidibc.core.BidibMessageProcessor;
import org.bidib.jbidibc.core.ConnectionListener;
import org.bidib.jbidibc.core.MessageListener;
import org.bidib.jbidibc.core.NodeListener;
import org.bidib.jbidibc.core.exception.NoAnswerException;
import org.bidib.jbidibc.core.exception.PortNotFoundException;
import org.bidib.jbidibc.core.exception.PortNotOpenedException;
import org.bidib.jbidibc.core.exception.ProtocolException;
import org.bidib.jbidibc.core.exception.ProtocolNoAnswerException;
import org.bidib.jbidibc.core.helpers.Context;
import org.bidib.jbidibc.core.node.BidibNode;
import org.bidib.jbidibc.core.node.NodeFactory;
import org.bidib.jbidibc.core.node.listener.TransferListener;
import org.bidib.jbidibc.core.utils.ByteUtils;
import org.bidib.jbidibc.serial.SerialMessageReceiver;
import org.bidib.jbidibc.serial.exception.InvalidLibraryException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.embeddedunveiled.serial.ISerialComDataListener;
import com.embeddedunveiled.serial.ISerialComEventListener;
import com.embeddedunveiled.serial.SerialComDataEvent;
import com.embeddedunveiled.serial.SerialComException;
import com.embeddedunveiled.serial.SerialComLineEvent;
import com.embeddedunveiled.serial.SerialComManager;
import com.embeddedunveiled.serial.SerialComManager.BAUDRATE;
import com.embeddedunveiled.serial.SerialComManager.DATABITS;
import com.embeddedunveiled.serial.SerialComManager.FLOWCONTROL;
import com.embeddedunveiled.serial.SerialComManager.PARITY;
import com.embeddedunveiled.serial.SerialComManager.STOPBITS;

//import gnu.io.CommPortIdentifier;
//import gnu.io.SerialPort;

/**
 * This is the default bidib implementation. It creates and initializes the MessageReceiver and the NodeFactory that is
 * used in the system.
 * 
 */
public final class ScmSerialBidib extends AbstractBidib {

    private static final Logger LOGGER = LoggerFactory.getLogger(ScmSerialBidib.class);

    private static final Logger MSG_RAW_LOGGER = LoggerFactory.getLogger("RAW");

    // private static final Logger MSG_OUTPUTSTREAM_LOGGER = LoggerFactory.getLogger("OPS");

    private SerialComManager scm;

    private long handle = -1;

    private ISerialComDataListener dataListener;

    private Semaphore portSemaphore = new Semaphore(1);

    private static ScmSerialBidib instance;

    private String requestedPortName;

    private ISerialComEventListener eventListener;

    private boolean addEventListener = false;

    static {
        Runtime.getRuntime().addShutdownHook(new Thread() {
            public void run() {
                try {
                    LOGGER.debug("Close the communication ports and perform cleanup.");
                    getInstance().close();
                }
                catch (Exception e) { // NOSONAR
                    // ignore
                }
            }
        });
    }

    private ScmSerialBidib() {
    }

    @Override
    protected BidibMessageProcessor createMessageReceiver(NodeFactory nodeFactory) {
        return new SerialMessageReceiver(nodeFactory);
    }

    private SerialMessageReceiver getSerialMessageReceiver() {
        return (SerialMessageReceiver) getMessageReceiver();
    }

    public static synchronized BidibInterface getInstance() {
        if (instance == null) {
            instance = new ScmSerialBidib();
            instance.initialize();
        }
        return instance;
    }

    @Override
    public void open(
        String portName, ConnectionListener connectionListener, Set<NodeListener> nodeListeners,
        Set<MessageListener> messageListeners, Set<TransferListener> transferListeners, Context context)
        throws PortNotFoundException, PortNotOpenedException {

        setConnectionListener(connectionListener);

        // register the listeners
        registerListeners(nodeListeners, messageListeners, transferListeners);

        if (scm == null) {
            if (portName == null || portName.trim().isEmpty()) {
                throw new PortNotFoundException("");
            }

            LOGGER.info("Open port with name: {}", portName);

            File file = new File(portName);

            if (file.exists()) {
                try {
                    portName = file.getCanonicalPath();
                    LOGGER.info("Changed port name to: {}", portName);
                }
                catch (IOException ex) {
                    throw new PortNotFoundException(portName);
                }
            }

            // CommPortIdentifier commPort = null;
            // try {
            // commPort = CommPortIdentifier.getPortIdentifier(portName);
            // }
            // catch (NoSuchPortException ex) {
            // LOGGER.warn("Requested port is not available: {}", portName, ex);
            // throw new PortNotFoundException(portName);
            // }

            requestedPortName = portName;

            try {
                portSemaphore.acquire();

                try {
                    // 115200 Baud
                    close();

                    scm = new SerialComManager();

                    // open the commPort
                    internalOpen(portName, BAUDRATE.B115200, context);

                    LOGGER.info("The port was opened internally, get the magic.");
                    int magic = sendResetAndMagic();
                    LOGGER.info("The root node returned the magic: {}", ByteUtils.magicToHex(magic));
                }
                catch (NoAnswerException naex) {
                    LOGGER.warn("Open communication failed.", naex);
                    try {
                        close();
                    }
                    catch (Exception e4) { // NOSONAR
                        // ignore
                    }
                    throw naex;
                }
                catch (ProtocolNoAnswerException naex) {
                    LOGGER.warn("Open communication failed.", naex);
                    try {
                        close();
                    }
                    catch (Exception e4) { // NOSONAR
                        // ignore
                    }
                    throw new NoAnswerException(naex.getMessage());
                }
                catch (Exception e2) {
                    LOGGER.info("Open port failed. Close port and throw exception.", e2);
                    try {
                        close();
                    }
                    catch (Exception e3) { // NOSONAR
                        LOGGER.warn("Close port failed.", e3);
                    }
                    throw new PortNotOpenedException(portName, PortNotOpenedException.UNKNOWN);
                }
                catch (UnsatisfiedLinkError err) {
                    LOGGER.info("Open port failed. Close port and throw exception.", err);

                    throw new PortNotOpenedException(portName, PortNotOpenedException.UNKNOWN);
                }
            }
            catch (InterruptedException ex) {
                LOGGER.warn("Wait for portSemaphore was interrupted.", ex);
                throw new PortNotOpenedException(portName, PortNotOpenedException.UNKNOWN);
            }
            finally {
                portSemaphore.release();
            }
        }
        else {
            LOGGER.warn("Port is already opened.");
        }
    }

    @Override
    public void close() {
        if (scm != null && handle > 0) {
            LOGGER.info("Close the port.");
            long start = System.currentTimeMillis();

            // unregister data listener
            LOGGER.info("Unregister data listener: {}", dataListener);

            if (dataListener != null) {
                try {
                    scm.unregisterDataListener(dataListener);
                }
                catch (SerialComException ex) {
                    LOGGER.warn("Unregister dataListener failed.", ex);
                }

                try {
                    Thread.sleep(200);
                }
                catch (InterruptedException ex) {
                    LOGGER.warn("Sleep after unregister data listener failed.", ex);
                }
                dataListener = null;
            }

            // unregister line event listener
            if (eventListener != null) {
                LOGGER.info("Unregister line event listener.");
                try {
                    scm.unregisterLineEventListener(eventListener);
                }
                catch (SerialComException ex) {
                    LOGGER.warn("Unregister lineEventListener failed.", ex);
                }

                try {
                    Thread.sleep(200);
                }
                catch (InterruptedException ex) {
                    LOGGER.warn("Sleep after unregister line event listener failed.", ex);
                }
            }

            eventListener = null;

            // no longer process received messages
            getSerialMessageReceiver().disable();

            // stop the send queue worker
            stopSendQueueWorker();

            try {
                LOGGER.info("Close the COM port: {}", handle);
                // port.close();
                scm.closeComPort(handle);
            }
            catch (Exception e) {
                LOGGER.warn("Close port failed.", e);
            }

            long end = System.currentTimeMillis();
            LOGGER.info("Closed the port. duration: {}", end - start);

            scm = null;
            handle = -1;

            if (getNodeFactory() != null) {
                // remove all stored nodes from the node factory
                getNodeFactory().reset();
            }

            if (getMessageReceiver() != null) {
                getSerialMessageReceiver().clearMessageListeners();
                getSerialMessageReceiver().clearNodeListeners();

                getSerialMessageReceiver().purgeOutputStream();
            }

            if (getConnectionListener() != null) {
                getConnectionListener().closed(requestedPortName);
            }

            requestedPortName = null;
        }
        else {
            LOGGER.info("No port to close available.");
        }
    }

    @Override
    public List<String> getPortIdentifiers() {
        List<String> portIdentifiers = new ArrayList<String>();

        try {
            SerialComManager scm = new SerialComManager();
            String[] ports = scm.listAvailableComPorts();
            for (String portIdentifier : ports) {
                portIdentifiers.add(portIdentifier);
            }
        }
        catch (Error error) {
            LOGGER.warn("Get comm port identifiers failed.", error);
            throw new RuntimeException(error.getMessage(), error.getCause());
        }
        catch (SerialComException ex) {
            LOGGER.warn("Get comm port identifiers failed.", ex);
            throw new InvalidLibraryException(ex.getMessage(), ex.getCause());
        }
        return portIdentifiers;
    }

    private long internalOpen(String portName, BAUDRATE baudRate, Context context) throws SerialComException {

        startSendQueueWorker();

        // open the port
        handle = scm.openComPort(portName, true, true, true);
        LOGGER.info("Opened serial port, handle: {}", handle);

        // LOGGER.info("Set flow control mode to SerialPort.FLOWCONTROL_RTSCTS_IN |
        // SerialPort.FLOWCONTROL_RTSCTS_OUT!");
        // serialPort.setFlowControlMode(SerialPort.FLOWCONTROL_RTSCTS_IN | SerialPort.FLOWCONTROL_RTSCTS_OUT);

        // LOGGER.info("Current input buffer size: {}", serialPort.getInputBufferSize());
        //
        // serialPort.setInputBufferSize(DEFAULT_INPUT_BUFFER_SIZE);
        // serialPort.enableReceiveThreshold(DEFAULT_RECEIVE_TIMEOUT);

        // serialPort.setSerialPortParams(baudRate, SerialPort.DATABITS_8, SerialPort.STOPBITS_1,
        // SerialPort.PARITY_NONE);

        scm.configureComPortControl(handle, FLOWCONTROL.NONE, 'x', 'x', false, true);

        scm.configureComPortData(handle, DATABITS.DB_8, STOPBITS.SB_1, PARITY.P_NONE, baudRate, 0);

        // clearInputStream(serialPort);

        if (context != null) {
            Boolean ignoreWrongMessageNumber =
                context.get("ignoreWrongReceiveMessageNumber", Boolean.class, Boolean.FALSE);
            getSerialMessageReceiver().setIgnoreWrongMessageNumber(ignoreWrongMessageNumber);
        }

        // TODO does throw an exception in 1.0.3
        if (addEventListener) {
            eventListener = new ISerialComEventListener() {

                @Override
                public void onNewSerialEvent(SerialComLineEvent lineEvent) {
                    // TODO Auto-generated method stub
                    LOGGER.error("eventCTS : {}, eventDSR : {}", lineEvent.getCTS(), lineEvent.getDSR());
                }
            };
            scm.registerLineEventListener(handle, eventListener);
        }

        // scm.registerPortMonitorListener(handle, portMonitor)

        // enable the message receiver before the event listener is added
        getSerialMessageReceiver().enable();

        dataListener = new ISerialComDataListener() {

            @Override
            public void onNewSerialDataAvailable(SerialComDataEvent dataEvent) {

                // LOGGER.info("Serial data available: {}", ByteUtils.bytesToHex(dataEvent.getDataBytes()));

                ByteArrayInputStream bis = null;
                try {
                    bis = new ByteArrayInputStream(dataEvent.getDataBytes());

                    ((SerialMessageReceiver) getMessageReceiver()).receive(bis);
                }
                catch (Exception ex) {
                    LOGGER.warn("Process received bytes failed.", ex);
                }
                finally {
                    if (bis != null) {
                        try {
                            bis.close();
                        }
                        catch (IOException ex) {
                            LOGGER.warn("Close bis failed.", ex);
                        }
                    }
                }
            }

            @Override
            public void onDataListenerError(int errorNum) {
                // TODO Auto-generated method stub

                LOGGER.error("Data listener notified an error: {}", errorNum);

                Thread t1 = new Thread(new Runnable() {

                    @Override
                    public void run() {
                        // TODO Auto-generated method stub
                        LOGGER.info("Error detected. Close the port.");

                        try {
                            scm.unregisterDataListener(dataListener);
                            dataListener = null;
                        }
                        catch (Exception ex) {
                            LOGGER.warn("Unregister data listener after error detection failed.", ex);
                        }
                        close();
                    }
                });
                t1.start();

            }
        };

        LOGGER.info("Registering data listener fro handle: {}.", handle);
        // register data listener for this port
        scm.registerDataListener(handle, dataListener);

        LOGGER.info("Registered data listener.");

        // Activate DTR
        try {
            LOGGER.info("Activate DTR.");

            scm.setDTR(handle, true); // pin 1 in DIN8; on main connector, this is DTR
        }
        catch (Exception e) {
            LOGGER.warn("Set DTR true failed.", e);
        }

        try {
            LOGGER.info("Activate RTS.");
            scm.setRTS(handle, true);
        }
        catch (Exception e) {
            LOGGER.warn("Set RTS true failed.", e);
        }

        return handle;
    }

    @Override
    public boolean isOpened() {
        boolean isOpened = (handle > 0);
        // try {
        // portSemaphore.acquire();
        //
        //
        //
        // LOGGER.debug("Check if port is opened: {}", port);
        // isOpened = (port != null && port.getOutputStream() != null);
        // }
        // catch (InterruptedException ex) {
        // LOGGER.warn("Wait for portSemaphore was interrupted.", ex);
        // }
        // catch (IOException ex) {
        // LOGGER.warn("OutputStream is not available.", ex);
        // }
        // finally {
        // portSemaphore.release();
        // }
        return isOpened;
    }

    private BlockingQueue<byte[]> sendQueue = new LinkedBlockingQueue<>();

    private Thread sendQueueWorker;

    private AtomicBoolean running = new AtomicBoolean();

    private AtomicLong sendQueueWorkerThreadId = new AtomicLong();

    /**
     * Send the bytes to the send queue.
     * 
     * @param bytes
     *            the bytes to send
     */
    @Override
    public void send(final byte[] bytes) {
        boolean added = sendQueue.offer(bytes);
        if (!added) {
            LOGGER.error("The message was not added to the send queue: {}", ByteUtils.bytesToHex(bytes));
        }
    }

    private void startSendQueueWorker() {
        running.set(true);

        LOGGER.info("Start the sendQueueWorker. Current sendQueueWorker: {}", sendQueueWorker);
        sendQueueWorker = new Thread(new Runnable() {

            @Override
            public void run() {
                try {
                    processSendQueue();
                }
                catch (Exception ex) {
                    LOGGER.warn("The processing of the sendQueue was terminated.", ex);
                }

                LOGGER.info("Process send queue has finished.");
            }
        }, "sendQueueWorker");

        try {
            sendQueueWorkerThreadId.set(sendQueueWorker.getId());
            sendQueueWorker.start();
        }
        catch (Exception ex) {
            LOGGER.error("Start the sendQueueWorker failed.", ex);
        }
    }

    private void stopSendQueueWorker() {
        LOGGER.info("Stop the send queue worker.");
        running.set(false);

        try {
            sendQueueWorker.interrupt();

            sendQueueWorker.join(1000);

            LOGGER.info("sendQueueWorker has finished.");
        }
        catch (Exception ex) {
            LOGGER.warn("Interrupt sendQueueWorker failed.", ex);
        }
        sendQueueWorker = null;
    }

    private void processSendQueue() {
        byte[] bytes = null;
        LOGGER.info("The sendQueueWorker is ready for processing.");
        while (running.get()) {
            try {

                bytes = sendQueue.take();
            }
            catch (Exception ex) {
                LOGGER.warn("Get message from sendQueue failed.", ex);
            }

            if (handle > 0 && bytes != null) {
                try {
                    if (MSG_RAW_LOGGER.isInfoEnabled()) {
                        MSG_RAW_LOGGER.info(">> [{}] - {}", bytes.length, ByteUtils.bytesToHex(bytes));
                    }

                    // MSG_OUTPUTSTREAM_LOGGER.info(">>>>");

                    boolean sent = scm.writeBytes(handle, bytes);

                    if (!sent) {
                        MSG_RAW_LOGGER.warn(">> sent: {}", sent);
                        LOGGER.error("The message has not been sent to handle: {}, message: {}", handle,
                            ByteUtils.bytesToHex(bytes));

                        throw new RuntimeException("Write message to output failed: " + ByteUtils.bytesToHex(bytes));
                    }
                    else {
                        MSG_RAW_LOGGER.info(">> sent: {}", sent);
                    }
                }
                catch (Exception ex) {
                    LOGGER.warn("Send message to output stream failed: [{}] - {}", bytes.length,
                        ByteUtils.bytesToHex(bytes), ex);

                    throw new RuntimeException("Send message to output stream failed: " + ByteUtils.bytesToHex(bytes),
                        ex);
                }
            }
        }
        LOGGER.info("The sendQueueWorker has finished processing.");
        sendQueueWorkerThreadId.set(0);
    }

    /**
     * Get the magic from the root node
     * 
     * @return the magic provided by the root node
     * @throws ProtocolException
     */
    private int sendResetAndMagic() throws ProtocolException {
        BidibNode rootNode = getRootNode();

        LOGGER.info("Send reset to the rootNode.");
        rootNode.reset();

        try {
            LOGGER.info("Wait 500ms before send the magic request.");
            Thread.sleep(500);
        }
        catch (InterruptedException ex) {
            LOGGER.warn("Wait before send the magic request failed.", ex);
        }

        // LOGGER.info("Clear the output buffer before get the magic from the rootNode.");
        // clearInputStream(port);
        // getSerialMessageReceiver().purgeOutputStream();

        int magic = rootNode.getMagic();

        LOGGER.debug("The node returned magic: {}", magic);
        return magic;
    }

    /**
     * Set the response timeout for the port.
     * 
     * @param timeout
     *            the receive timeout to set
     */
    @Override
    public void setResponseTimeout(int timeout) {
        // if (port != null) {
        // LOGGER.info("Set the response timeout for the serial port: {}", timeout);
        // try {
        // port.enableReceiveTimeout(timeout);
        // }
        // catch (Exception ex) {
        // LOGGER.warn("Set the receive timeout on port failed.", ex);
        // throw new RuntimeException(ex);
        // }
        // }
    }
}
