package org.bidib.jbidibc.serial;

import java.io.File;
import java.io.IOException;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;

import org.bidib.jbidibc.core.AbstractBidib;
import org.bidib.jbidibc.core.BidibMessageProcessor;
import org.bidib.jbidibc.core.ConnectionListener;
import org.bidib.jbidibc.core.MessageListener;
import org.bidib.jbidibc.core.NodeListener;
import org.bidib.jbidibc.core.exception.InvalidConfigurationException;
import org.bidib.jbidibc.core.exception.NoAnswerException;
import org.bidib.jbidibc.core.exception.PortNotFoundException;
import org.bidib.jbidibc.core.exception.PortNotOpenedException;
import org.bidib.jbidibc.core.exception.ProtocolException;
import org.bidib.jbidibc.core.exception.ProtocolNoAnswerException;
import org.bidib.jbidibc.core.helpers.Context;
import org.bidib.jbidibc.core.node.NodeFactory;
import org.bidib.jbidibc.core.node.RootNode;
import org.bidib.jbidibc.core.node.listener.TransferListener;
import org.bidib.jbidibc.core.utils.ByteUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This is the abstract serial bidib implementation. It creates and initializes the MessageReceiver and the NodeFactory
 * that is used in the system.
 */
public abstract class AbstractSerialBidib extends AbstractBidib {

    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractSerialBidib.class);

    // private static final Logger MSG_RAW_LOGGER = LoggerFactory.getLogger("RAW");

    // private static final Logger MSG_OUTPUTSTREAM_LOGGER = LoggerFactory.getLogger("OPS");

    protected Semaphore portSemaphore = new Semaphore(1);

    private static AbstractSerialBidib instance;

    protected String requestedPortName;

    private BlockingQueue<byte[]> sendQueue = new LinkedBlockingQueue<>();

    private Thread sendQueueWorker;

    private AtomicBoolean running = new AtomicBoolean();

    private AtomicLong sendQueueWorkerThreadId = new AtomicLong();

    private BlockingQueue<byte[]> receiveQueue = new LinkedBlockingQueue<>();

    private Thread receiveQueueWorker;

    private AtomicBoolean receiverRunning = new AtomicBoolean();

    private AtomicLong receiveQueueWorkerThreadId = new AtomicLong();

    @Override
    protected BidibMessageProcessor createMessageReceiver(NodeFactory nodeFactory) {
        return new SerialMessageReceiver(nodeFactory);
    }

    protected SerialMessageReceiver getSerialMessageReceiver() {
        SerialMessageReceiver serialMessageReceiver = (SerialMessageReceiver) getMessageReceiver();
        LOGGER.debug("Get the serial message receiver: {}", serialMessageReceiver);
        return serialMessageReceiver;
    }

    /**
     * Returns if an instance of ScmSerialBidib is available. The instance will not be created if not available.
     * 
     * @return an instance of ScmSerialBidib is available
     */
    public static synchronized boolean isInstanceAvailable() {
        return (instance != null);
    }

    protected void stopReceiverAndQueues(final SerialMessageReceiver serialMessageReceiver) {
        if (serialMessageReceiver != null) {
            // no longer process received messages
            serialMessageReceiver.disable();
        }
        else {
            LOGGER.warn("No message receiver to disable available.");
        }

        // stop the send queue worker
        stopSendQueueWorker();
        stopReceiveQueueWorker();
    }

    protected void cleanupAfterClose(final SerialMessageReceiver serialMessageReceiver) {
        releaseRootNode();

        InvalidConfigurationException ice = null;
        if (serialMessageReceiver != null) {
            serialMessageReceiver.clearMessageListeners();
            serialMessageReceiver.clearNodeListeners();

            LOGGER.info("Purge the received data in the message buffer.");

            try {
                serialMessageReceiver.purgeReceivedDataInBuffer();
            }
            catch (InvalidConfigurationException ex) {
                LOGGER.warn("Purge output stream has signaled an error.", ex);

                if ("debug-interface-active".equals(ex.getReason())) {
                    ice = ex;
                }
            }
        }
        else {
            LOGGER.warn("No message receiver to purge received data buffer available.");
        }

        if (getConnectionListener() != null) {
            getConnectionListener().closed(requestedPortName);
        }

        requestedPortName = null;

        if (ice != null) {
            LOGGER.warn("Signal the invalid configuration exception to the caller.");
            throw ice;
        }
    }

    protected void startReceiverAndQueues(final SerialMessageReceiver serialMessageReceiver, final Context context) {

        startSendQueueWorker();
        startReceiveQueueWorker();

        if (serialMessageReceiver != null && context != null) {
            Boolean ignoreWrongMessageNumber =
                context.get("ignoreWrongReceiveMessageNumber", Boolean.class, Boolean.FALSE);
            serialMessageReceiver.setIgnoreWrongMessageNumber(ignoreWrongMessageNumber);
        }
        else {
            LOGGER.warn("No message receiver available.");
        }

        // TODO remove this????
        if (serialMessageReceiver != null) {
            // enable the message receiver before the event listener is added
            serialMessageReceiver.enable();
        }
        else {
            LOGGER.warn("No message receiver to enable available.");
        }

    }

    protected abstract boolean isImplAvaiable();

    protected abstract void internalOpen(String portName, int baudRate, final Context context) throws Exception;

    @Override
    public void open(
        String portName, ConnectionListener connectionListener, Set<NodeListener> nodeListeners,
        Set<MessageListener> messageListeners, Set<TransferListener> transferListeners, Context context)
        throws PortNotFoundException, PortNotOpenedException {

        setConnectionListener(connectionListener);

        // register the listeners
        registerListeners(nodeListeners, messageListeners, transferListeners);

        if (!isImplAvaiable()) {
            if (portName == null || portName.trim().isEmpty()) {
                throw new PortNotFoundException("");
            }

            LOGGER.info("Open port with name: {}", portName);

            File file = new File(portName);

            if (file.exists()) {
                try {
                    portName = file.getCanonicalPath();
                    LOGGER.info("Changed port name to: {}", portName);
                }
                catch (IOException ex) {
                    throw new PortNotFoundException(portName);
                }
            }

            requestedPortName = portName;

            try {
                portSemaphore.acquire();

                try {
                    // 115200 Baud
                    close();

                    // open the commPort
                    internalOpen(portName, 115200, context);

                    LOGGER.info("The port was opened internally, get the magic.");
                    int magic = sendResetAndMagic();

                    LOGGER.info("The root node returned the magic: {}", ByteUtils.magicToHex(magic));
                }
                catch (NoAnswerException naex) {
                    LOGGER.warn("Open communication failed.", naex);
                    try {
                        close();
                    }
                    catch (Exception e4) { // NOSONAR
                        // ignore
                    }
                    throw naex;
                }
                catch (ProtocolNoAnswerException naex) {
                    LOGGER.warn("Open communication failed.", naex);
                    try {
                        close();
                    }
                    catch (Exception e4) { // NOSONAR
                        // ignore
                    }
                    throw new NoAnswerException(naex.getMessage());
                }
                catch (PortNotFoundException pnfEx) {
                    LOGGER.info("Open port failed. Close port and throw exception.", pnfEx);

                    // close port to cleanup and stop the send queue worker

                    try {
                        close();
                    }
                    catch (Exception e3) { // NOSONAR
                        LOGGER.warn("Close port failed.", e3);
                    }
                    throw new PortNotOpenedException(portName, PortNotOpenedException.PORT_NOT_FOUND);
                }
                catch (Exception e2) {
                    LOGGER.info("Open port failed. Close port and throw exception.", e2);

                    // close port to cleanup and stop the send queue worker

                    try {
                        close();
                    }
                    catch (Exception e3) { // NOSONAR
                        LOGGER.warn("Close port failed.", e3);
                    }
                    throw new PortNotOpenedException(portName, PortNotOpenedException.UNKNOWN);
                }
                catch (UnsatisfiedLinkError err) {
                    LOGGER.info("Open port failed. Close port and throw exception.", err);

                    throw new PortNotOpenedException(portName, PortNotOpenedException.UNKNOWN);
                }
            }
            catch (InterruptedException ex) {
                LOGGER.warn("Wait for portSemaphore was interrupted.", ex);
                throw new PortNotOpenedException(portName, PortNotOpenedException.UNKNOWN);
            }
            finally {
                portSemaphore.release();
            }
        }
        else {
            LOGGER.warn("Port is already opened.");
        }
    }

    private AtomicBoolean isConnected = new AtomicBoolean();

    protected boolean isConnected() {
        return isConnected.get();
    }

    protected void setConnected(boolean connected) {
        isConnected.set(connected);
    }

    @Override
    public abstract boolean isOpened();

    /**
     * Send the bytes to the send queue.
     * 
     * @param bytes
     *            the bytes to send
     */
    @Override
    public void send(final byte[] bytes) {
        boolean added = sendQueue.offer(bytes);
        if (!added) {
            LOGGER.error("The message was not added to the send queue: {}", ByteUtils.bytesToHex(bytes));
        }
    }

    private void startSendQueueWorker() {
        running.set(true);

        LOGGER.info("Start the sendQueueWorker. Current sendQueueWorker: {}", sendQueueWorker);
        sendQueueWorker = new Thread(new Runnable() {

            @Override
            public void run() {
                try {
                    processSendQueue();
                }
                catch (Exception ex) {
                    LOGGER.warn("The processing of the send queue was terminated with an exception!", ex);

                    running.set(false);
                }

                LOGGER.info("Process send queue has finished.");
            }
        }, "sendQueueWorker");

        try {
            sendQueueWorkerThreadId.set(sendQueueWorker.getId());
            sendQueueWorker.start();
        }
        catch (Exception ex) {
            LOGGER.error("Start the sendQueueWorker failed.", ex);
        }
    }

    private void stopSendQueueWorker() {
        LOGGER.info("Stop the send queue worker.");
        running.set(false);

        try {
            sendQueueWorker.interrupt();

            sendQueueWorker.join(1000);

            LOGGER.info("sendQueueWorker has finished.");
        }
        catch (Exception ex) {
            LOGGER.warn("Interrupt sendQueueWorker failed.", ex);
        }
        sendQueueWorker = null;
    }

    protected abstract void sendData(byte[] data);

    private void processSendQueue() {
        byte[] bytes = null;
        LOGGER.info("The sendQueueWorker is ready for processing.");
        while (running.get()) {

            try {
                // get the message to process
                bytes = sendQueue.take();
            }
            catch (InterruptedException ex) {
                LOGGER.warn("Get message from sendQueue failed because thread was interrupted.");
            }
            catch (Exception ex) {
                LOGGER.warn("Get message from sendQueue failed.", ex);
                bytes = null;
            }

            sendData(bytes);

            // release the bytes
            bytes = null;
        }

        LOGGER.info("The sendQueueWorker has finished processing.");
        sendQueueWorkerThreadId.set(0);
    }

    /**
     * Get the magic from the root node
     * 
     * @return the magic provided by the root node
     * @throws ProtocolException
     */
    private int sendResetAndMagic() throws ProtocolException {
        RootNode rootNode = getRootNode();

        LOGGER.info("Send sysDisable to the rootNode.");
        rootNode.sysDisable();
        // LOGGER.info("Send reset to the rootNode.");
        // rootNode.reset();

        try {
            LOGGER.info("Wait 300ms before send the magic request.");
            Thread.sleep(300);
        }
        catch (InterruptedException ex) {
            LOGGER.warn("Wait before send the magic request failed.", ex);
        }

        final SerialMessageReceiver serialMessageReceiver = getSerialMessageReceiver();
        LOGGER.info("Enable the message receiver before send magic: {}", serialMessageReceiver);
        // enable the message receiver
        serialMessageReceiver.enable();

        LOGGER.info("Send the magic request.");
        int magic = rootNode.getMagic(1500);

        LOGGER.debug("The node returned magic: {}", magic);
        return magic;
    }

    protected void receive(final byte[] data) {

        // byte[] buffer = new byte[bytes.length];
        // System.arraycopy(bytes, 0, buffer, 0, bytes.length);
        //
        // boolean added = receiveQueue.offer(buffer);
        // if (!added) {
        // LOGGER.error("The message was not added to the receive queue: {}", ByteUtils.bytesToHex(buffer));
        // }

        receive(data, data.length);
    }

    protected void receive(final byte[] data, int len) {

        byte[] buffer = new byte[len];
        System.arraycopy(data, 0, buffer, 0, len);

        boolean added = receiveQueue.offer(buffer);
        if (!added) {
            LOGGER.error("The message was not added to the receive queue: {}", ByteUtils.bytesToHex(buffer));
        }
    }

    private void startReceiveQueueWorker() {
        receiverRunning.set(true);

        LOGGER.info("Start the receiveQueueWorker. Current receiveQueueWorker: {}", receiveQueueWorker);
        receiveQueueWorker = new Thread(new Runnable() {

            @Override
            public void run() {
                try {
                    processReceiveQueue();
                }
                catch (Exception ex) {
                    LOGGER.warn("The processing of the receive queue was terminated with an exception!", ex);
                }

                LOGGER.info("Process receive queue has finished.");
            }
        }, "receiveQueueWorker");

        try {
            receiveQueueWorkerThreadId.set(receiveQueueWorker.getId());
            receiveQueueWorker.start();
        }
        catch (Exception ex) {
            LOGGER.error("Start the receiveQueueWorker failed.", ex);
        }
    }

    private void stopReceiveQueueWorker() {
        LOGGER.info("Stop the receive queue worker.");
        receiverRunning.set(false);

        try {
            receiveQueueWorker.interrupt();

            receiveQueueWorker.join(1000);

            LOGGER.info("receiveQueueWorker has finished.");
        }
        catch (Exception ex) {
            LOGGER.warn("Interrupt receiveQueueWorker failed.", ex);
        }
        receiveQueueWorker = null;
    }

    private void processReceiveQueue() {
        byte[] bytes = null;
        LOGGER.info("The receiveQueueWorker is ready for processing.");

        final SerialMessageReceiver serialMessageReceiver = getSerialMessageReceiver();
        while (receiverRunning.get()) {
            try {
                // get the message to process
                bytes = receiveQueue.take();

                if (bytes != null) {
                    // process
                    try {
                        serialMessageReceiver.receive(bytes, bytes.length);
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Process received bytes failed.", ex);
                    }

                }
            }
            catch (InterruptedException ex) {
                LOGGER.warn("Get message from receiveQueue failed because thread was interrupted.");
            }
            catch (Exception ex) {
                LOGGER.warn("Get message from receiveQueue failed.", ex);
                bytes = null;
            }

        }

        LOGGER.info("The receiveQueueWorker has finished processing.");
        receiveQueueWorkerThreadId.set(0);
    }
}
