package org.bidib.jbidibc.serial;

import java.io.File;
import java.io.IOException;
import java.util.Set;
import java.util.concurrent.Semaphore;

import org.apache.commons.lang3.StringUtils;
import org.bidib.jbidibc.core.AbstractBidib;
import org.bidib.jbidibc.core.BidibMessageProcessor;
import org.bidib.jbidibc.core.MessageListener;
import org.bidib.jbidibc.core.NodeListener;
import org.bidib.jbidibc.core.node.NodeRegistry;
import org.bidib.jbidibc.core.node.RootNode;
import org.bidib.jbidibc.core.node.listener.TransferListener;
import org.bidib.jbidibc.messages.ConnectionListener;
import org.bidib.jbidibc.messages.MessageReceiver;
import org.bidib.jbidibc.messages.base.AbstractBaseBidib;
import org.bidib.jbidibc.messages.base.RawMessageListener;
import org.bidib.jbidibc.messages.exception.InvalidConfigurationException;
import org.bidib.jbidibc.messages.exception.NoAnswerException;
import org.bidib.jbidibc.messages.exception.PortNotFoundException;
import org.bidib.jbidibc.messages.exception.PortNotOpenedException;
import org.bidib.jbidibc.messages.exception.ProtocolException;
import org.bidib.jbidibc.messages.exception.ProtocolNoAnswerException;
import org.bidib.jbidibc.messages.helpers.Context;
import org.bidib.jbidibc.messages.helpers.DefaultContext;
import org.bidib.jbidibc.messages.message.BidibResponseFactory;
import org.bidib.jbidibc.messages.message.ResponseFactory;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This is the abstract serial bidib implementation. It creates and initializes the MessageReceiver and the NodeFactory
 * that is used in the system.
 */
public abstract class AbstractSerialBidib extends AbstractBidib<MessageReceiver> {

    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractSerialBidib.class);

    protected Semaphore portSemaphore = new Semaphore(1);

    protected String requestedPortName;

    protected Object receiveLock = new Object();

    @Override
    public void setDelayAfterSend(long delay) {
        getConnector().setDelayAfterSend(delay);
    }

    protected abstract AbstractBaseBidib<MessageReceiver> getConnector();

    @Override
    protected SerialMessageReceiver createMessageReceiver(
        final NodeRegistry nodeRegistry, final RawMessageListener rawMessageListener, final Context context) {

        final ResponseFactory responseFactory = new BidibResponseFactory();
        responseFactory.initialize();

        SerialMessageReceiver messageReceiver = new SerialMessageReceiver(nodeRegistry, responseFactory);
        messageReceiver.setRawMessageListener(rawMessageListener);
        messageReceiver.init(context);
        return messageReceiver;
    }

    @Override
    protected void cleanupAfterClose(final MessageReceiver bidibMessageProcessor) {
        releaseRootNode();

        InvalidConfigurationException ice = null;
        if (bidibMessageProcessor instanceof SerialMessageReceiver) {
            SerialMessageReceiver serialMessageReceiver = (SerialMessageReceiver) bidibMessageProcessor;

            if (!hasMoreRetryAvailable()) {
                LOGGER.info("Clear message and node listeners from serial message receiver.");

                serialMessageReceiver.clearMessageListeners();
                serialMessageReceiver.clearNodeListeners();
            }
            else {
                LOGGER
                    .info(
                        "Do not clear message and node listeners from serial message receiver in cleanupAfterClose because more retry is available.");
            }

            LOGGER.info("Purge the received data in the message buffer.");

            try {
                serialMessageReceiver.purgeReceivedDataInBuffer();
            }
            catch (InvalidConfigurationException ex) {
                LOGGER.warn("Purge output stream has signaled an error.", ex);

                if ("debug-interface-active".equals(ex.getReason())) {
                    ice = ex;
                }
            }
        }
        else {
            LOGGER.warn("No message receiver to purge received data buffer available.");
        }

        if (ice == null && hasMoreRetryAvailable()) {
            LOGGER
                .info("Do not signal the connection was closed in cleanupAfterClose because more retry is available.");
        }
        else {
            LOGGER.info("No more retry available.");
            if (getConnectionListener() != null) {
                LOGGER.info("Notify that the port was closed: {}", requestedPortName);
                getConnectionListener().closed(requestedPortName);
            }
            else {
                LOGGER
                    .info("No connection listener available to publish the closed report for port: {}",
                        requestedPortName);
            }

            requestedPortName = null;

            super.cleanupAfterClose(bidibMessageProcessor);
        }

        if (ice != null) {
            LOGGER.warn("Signal the invalid configuration exception to the caller.");
            throw ice;
        }
    }

    protected abstract boolean isImplAvaiable();

    protected abstract void internalOpen(String portName, final Context context) throws Exception;

    /**
     * @param messageKey
     *            the message key to notify
     */
    protected void notifyStatusKey(String messageKey, final Context context) {

        if (getConnectionListener() != null) {
            getConnectionListener().status(messageKey, context);
        }
        else {
            LOGGER.info("No connection listener available.");
        }
    }

    private int[] baudRates = { /* 1000000, */ 115200, 19200 };

    private int baudRateIndex;

    @Override
    public void open(
        String portName, ConnectionListener connectionListener, Set<NodeListener> nodeListeners,
        Set<MessageListener> messageListeners, Set<TransferListener> transferListeners, Context context)
        throws PortNotFoundException, PortNotOpenedException {

        setConnectionListener(connectionListener);

        // register the listeners
        registerListeners(nodeListeners, messageListeners, transferListeners);

        if (!isImplAvaiable()) {
            if (StringUtils.isBlank(portName)) {
                throw new PortNotFoundException("No portName to open provided!");
            }

            LOGGER.info("Open port with name: {}", portName);

            File file = new File(portName);

            if (file.exists()) {

                Boolean symlink = context.get("symlink", Boolean.class, Boolean.FALSE);
                if (!symlink) {
                    try {
                        portName = file.getCanonicalPath();
                        LOGGER.info("Changed port name to: {}", portName);
                    }
                    catch (IOException ex) {
                        throw new PortNotFoundException(portName);
                    }
                }
                else {
                    LOGGER.info("Do not replace symlink name.");
                }
            }

            requestedPortName = portName;

            boolean openPassed = false;

            try {
                portSemaphore.acquire();

                while (baudRateIndex < baudRates.length) {

                    Integer baudRate = baudRates[baudRateIndex];
                    try {
                        // make sure the port is closed
                        close();

                        // open the commPort

                        context.register("serial.baudrate", baudRate);

                        LOGGER.info("Try to connect with baudRate: {}", context.get("serial.baudrate"));

                        final Context statusContext = new DefaultContext();
                        statusContext.register("baudrate", Integer.valueOf(baudRates[baudRateIndex]));

                        notifyStatusKey("bidib-connect-with-baudrate", statusContext);

                        internalOpen(portName, context);

                        openPassed = true;

                        break;
                    }
                    catch (ProtocolNoAnswerException naex) {
                        LOGGER.warn("Open communication failed because the interface did not send an answer.", naex);
                        try {
                            close();
                        }
                        catch (Exception e4) { // NOSONAR
                            // ignore
                        }

                        // check if we could retry with another baudrate
                        if (baudRateIndex < (baudRates.length - 1)) {
                            baudRateIndex++;
                            LOGGER.info("Try to connect with the next baudRate: {}", baudRates[baudRateIndex]);

                            final Context statusContext = new DefaultContext();
                            statusContext.register("baudrate", Integer.valueOf(baudRates[baudRateIndex]));

                            notifyStatusKey("bidib-reconnect-with-next-baudrate", statusContext);

                            try {
                                Thread.sleep(150);
                            }
                            catch (InterruptedException ie) {
                                LOGGER.warn("Wait to display status message was interrupted.", ie);
                            }
                            continue;
                        }

                        throw new NoAnswerException(naex.getMessage());
                    }
                    catch (NoAnswerException naex) {
                        LOGGER
                            .warn(
                                "Open communication failed because no answer from interface was received. Current baudRate: {}",
                                baudRate, naex);

                        try {
                            close();
                        }
                        catch (Exception e4) { // NOSONAR
                            // ignore
                        }

                        // check if we could retry with another baudrate
                        if (baudRateIndex < (baudRates.length - 1)) {
                            baudRateIndex++;
                            LOGGER.info("Try to connect with the next baudRate: {}", baudRates[baudRateIndex]);

                            final Context statusContext = new DefaultContext();
                            statusContext.register("baudrate", Integer.valueOf(baudRates[baudRateIndex]));

                            notifyStatusKey("bidib-reconnect-with-next-baudrate", statusContext);

                            try {
                                Thread.sleep(150);
                            }
                            catch (InterruptedException ie) {
                                LOGGER.warn("Wait to display status message was interrupted.", ie);
                            }
                            continue;
                        }

                        throw new NoAnswerException(naex.getMessage());
                    }
                    catch (PortNotFoundException pnfEx) {
                        LOGGER.info("Open port failed. Close port and throw exception.", pnfEx);

                        // close port to cleanup and stop the send queue worker

                        try {
                            close();
                        }
                        catch (Exception e3) { // NOSONAR
                            LOGGER.warn("Close port failed.", e3);
                        }
                        throw new PortNotOpenedException(portName, PortNotOpenedException.PORT_NOT_FOUND);
                    }
                    catch (PortNotOpenedException pnoEx) {
                        LOGGER.info("Open port failed. Close port and throw exception.", pnoEx);

                        String reason = pnoEx.getReason();
                        // close port to cleanup and stop the send queue worker

                        try {
                            close();
                        }
                        catch (Exception e3) { // NOSONAR
                            LOGGER.warn("Close port failed.", e3);
                        }
                        throw new PortNotOpenedException(portName,
                            (StringUtils.isNotBlank(reason) ? reason : PortNotOpenedException.UNKNOWN));
                    }
                    catch (IOException e2) {
                        LOGGER.info("Open port failed with IOException. Close port and throw exception.", e2);

                        String reason = e2.getMessage();

                        // close port to cleanup and stop the send queue worker

                        try {
                            close();
                        }
                        catch (Exception e3) { // NOSONAR
                            LOGGER.warn("Close port failed but this is ignored.", e3);
                        }
                        throw new PortNotOpenedException(portName,
                            (StringUtils.isNotBlank(reason) ? reason : PortNotOpenedException.UNKNOWN));
                    }
                    catch (Exception e2) {
                        LOGGER.info("Open port failed. Close port and throw exception.", e2);

                        // close port to cleanup and stop the send queue worker

                        try {
                            close();
                        }
                        catch (Exception e3) { // NOSONAR
                            LOGGER.warn("Close port failed.", e3);
                        }
                        throw new PortNotOpenedException(portName, PortNotOpenedException.UNKNOWN);
                    }
                    catch (UnsatisfiedLinkError err) {
                        LOGGER.info("Open port failed. Close port and throw exception.", err);

                        throw new PortNotOpenedException(portName, PortNotOpenedException.UNKNOWN);
                    }
                    catch (NoSuchMethodError err) {
                        LOGGER.info("Open port failed. Close port and throw exception.", err);

                        throw new PortNotOpenedException(portName, PortNotOpenedException.UNKNOWN);
                    }
                }
            }
            catch (InterruptedException ex) {
                LOGGER.warn("Wait for portSemaphore was interrupted.", ex);
                throw new PortNotOpenedException(portName, PortNotOpenedException.UNKNOWN);
            }
            finally {
                portSemaphore.release();
            }

            if (!openPassed) {
                LOGGER.warn("The connection was not established successfully. Configured portName: {}", portName);
                throw new NoAnswerException("The connection was not established successfully.");
            }
        }
        else {
            LOGGER.warn("Port is already opened.");
        }
    }

    @Override
    protected boolean hasMoreRetryAvailable() {
        boolean hasMoreRetryAvailable = baudRateIndex < (baudRates.length - 1);
        LOGGER.info("Current hasMoreRetryAvailable: {}", hasMoreRetryAvailable);
        return hasMoreRetryAvailable;
    }

    @Override
    public abstract boolean isOpened();

    /**
     * Contact the interface and get the magic
     * 
     * @return the magic
     */
    @Override
    protected int contactInterface() {
        LOGGER.info("Contact the interface node.");

        try {
            int magic = sendDisableAndMagic();

            LOGGER.info("The root node returned the magic: {}", ByteUtils.magicToHex(magic));

            // if the magic was received we have to skip the connect retry with different baudrate
            baudRateIndex = baudRates.length + 1;

            return magic;
        }
        catch (ProtocolException ex) {
            throw new NoAnswerException("Contact the interface node failed.", ex);
        }
    }

    /**
     * Get the magic from the root node
     * 
     * @return the magic provided by the root node
     * @throws ProtocolException
     */
    private int sendDisableAndMagic() throws ProtocolException {
        RootNode rootNode = getRootNode();

        LOGGER.info("Send sysDisable to the rootNode.");
        rootNode.sysDisable();
        // LOGGER.info("Send reset to the rootNode.");
        // rootNode.reset();

        try {
            LOGGER.info("Wait 300ms before send the magic request.");
            Thread.sleep(300);
        }
        catch (InterruptedException ex) {
            LOGGER.warn("Wait before send the magic request failed.", ex);
        }

        final BidibMessageProcessor serialMessageReceiver = getBidibMessageProcessor();

        if (serialMessageReceiver instanceof SerialMessageReceiver) {
            LOGGER.info("Enable the message receiver before send magic: {}", serialMessageReceiver);
            // enable the message receiver
            ((SerialMessageReceiver) serialMessageReceiver).enable();
        }

        LOGGER.info("Send the magic request.");
        int magic = rootNode.getMagic(1500);

        LOGGER.debug("The node returned magic: {}", magic);
        return magic;
    }
}
