package org.bidib.jbidibc.serial;

import java.io.ByteArrayOutputStream;
import java.nio.charset.StandardCharsets;

import org.bidib.jbidibc.core.AbstractMessageReceiver;
import org.bidib.jbidibc.core.MessageParser;
import org.bidib.jbidibc.core.node.NodeRegistry;
import org.bidib.jbidibc.messages.exception.InvalidConfigurationException;
import org.bidib.jbidibc.messages.exception.ProtocolException;
import org.bidib.jbidibc.messages.helpers.Context;
import org.bidib.jbidibc.messages.message.RequestFactory;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SerialMessageReceiver extends AbstractMessageReceiver {
    private static final Logger LOGGER = LoggerFactory.getLogger(SerialMessageReceiver.class);

    private ByteArrayOutputStream receiveBuffer = new ByteArrayOutputStream(2048);

    private SerialMessageParser serialMessageParser;

    public SerialMessageReceiver(NodeRegistry nodeFactory, final RequestFactory requestFactory) {
        super(nodeFactory, requestFactory, true);

    }

    @Override
    public void init(final Context context) {
        super.init(context);

        this.serialMessageParser = (SerialMessageParser) getMessageParser();

        LOGGER.info("Keep the messageParser as serialMessageParser: {}", serialMessageParser);
    }

    @Override
    protected MessageParser createMessageParser() {
        LOGGER.info("Create the serial message parser.");
        MessageParser messageParser = new SerialMessageParser();
        return messageParser;
    }

    @Override
    public void enable() {
        LOGGER.info("enable is called.");

        serialMessageParser.setEscapeHot(false);

        MSG_RAW_LOGGER.info("++++ Enable the serial message receiver.");

        try {
            receiveBuffer.reset();
        }
        catch (Exception ex) {
            LOGGER.warn("Reset buffered received data failed.", ex);
        }

        try {
            serialMessageParser.reset();
        }
        catch (Exception ex) {
            LOGGER.warn("Reset buffered received data failed.", ex);
        }

        super.enable();
    }

    @Override
    public void disable() {
        LOGGER.info("Disable is called.");
        super.disable();

        MSG_RAW_LOGGER.info("++++ Disable the serial message receiver.");

        serialMessageParser.setEscapeHot(false);
    }

    @Override
    public String getErrorInformation() {
        if (receiveBuffer.size() > 0) {

            byte[] remaining = receiveBuffer.toByteArray();
            return new String(remaining);
        }

        return null;
    }

    public void purgeReceivedDataInBuffer() {
        LOGGER.info("Purge the received data in the message buffer before get the lock.");

        LOGGER
            .info("Purge the received data in the message buffer. Current escapeHot: {}",
                serialMessageParser.isEscapeHot());

        serialMessageParser.setEscapeHot(false);

        if (receiveBuffer.size() > 0) {

            try {
                byte[] remaining = receiveBuffer.toByteArray();

                String remainingValue = new String(remaining, StandardCharsets.UTF_8);

                LOGGER
                    .info("Cleared remaining data from message output buffer: {}, text: {}",
                        ByteUtils.bytesToHex(remaining), remainingValue);

                if (remainingValue.indexOf("I=") > -1) {
                    // This is the detection of the debug interface active
                    InvalidConfigurationException ice =
                        new InvalidConfigurationException("An active debug interface configuration was detected.");
                    ice.setReason("debug-interface-active");
                    throw ice;
                }
            }
            finally {
                receiveBuffer.reset();
            }

            LOGGER.info("Finished purge output stream.");
        }
        else {
            LOGGER.info("The message buffer is empty.");
        }
    }

    /**
     * Receive messages from the configured port
     * 
     * @param data
     *            the received data
     */
    @Override
    public void receive(final ByteArrayOutputStream data) {

        if (!isEnabled()) {
            LOGGER.info("The receiver is not running. Skip processing of messages.");
            try {
                byte[] rawdata = data.toByteArray();
                LOGGER
                    .info("Receiver is stopped, number of bytes read: {}, buffer: {}", rawdata.length,
                        ByteUtils.bytesToHex(rawdata));
            }
            catch (Exception ex) {
                LOGGER.warn("Read data from input stream to buffer failed.", ex);
            }
            return;
        }

        MSG_RAW_LOGGER.info("<<<< start parse input");

        try {
            byte[] rawdata = data.toByteArray();
            parseInput(rawdata, rawdata.length);
        }
        catch (Exception e) {
            LOGGER.warn("Exception detected in message receiver!", e);

            // reset the escapeHot flag
            if (serialMessageParser.isEscapeHot()) {
                LOGGER.warn("Reset the escapeHot to false.");

                serialMessageParser.setEscapeHot(false);
            }
            throw new RuntimeException(e);
        }
        finally {
            MSG_RAW_LOGGER.info("<<<< finished parse input");
        }
    }

    /**
     * Parse the received data to process the received bidib packets.
     * 
     * @param receivedData
     *            the received data
     * @param len
     *            the len of the recevided data packet
     * @throws ProtocolException
     */
    @Override
    protected void parseInput(final byte[] receivedData, int len) throws ProtocolException {

        serialMessageParser.parseInput(this, receivedData, len);
    }
}
