package org.bidib.jbidibc.serial.raw;

import java.io.ByteArrayOutputStream;
import java.nio.charset.StandardCharsets;

import org.bidib.jbidibc.core.AbstractRawMessageReceiver;
import org.bidib.jbidibc.messages.exception.InvalidConfigurationException;
import org.bidib.jbidibc.messages.exception.ProtocolException;
import org.bidib.jbidibc.messages.helpers.Context;
import org.bidib.jbidibc.messages.message.BidibMessageInterface;
import org.bidib.jbidibc.messages.message.BidibRequestFactory;
import org.bidib.jbidibc.messages.message.RequestFactory;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.jbidibc.serial.SerialMessageParser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SerialRawMessageReceiver extends AbstractRawMessageReceiver {

    private static final Logger LOGGER = LoggerFactory.getLogger(SerialRawMessageReceiver.class);

    protected static final Logger MSG_RAW_LOGGER = LoggerFactory.getLogger("RAW");

    private ByteArrayOutputStream receiveBuffer = new ByteArrayOutputStream(2048);

    private SerialMessageParser messageParser;

    private RequestFactory requestFactory;

    private final MessagePublisher toHostPublisher;

    public SerialRawMessageReceiver(boolean checkCRC, final MessagePublisher toHostPublisher) {
        super(checkCRC);

        this.toHostPublisher = toHostPublisher;
    }

    @Override
    public void init(final Context context) {
        createMessageParser();

        requestFactory = new BidibRequestFactory();
        requestFactory.initialize();

        super.init(context);
    }

    protected void createMessageParser() {
        LOGGER.info("Create the serial message parser.");
        messageParser = new SerialMessageParser();
    }

    @Override
    public void enable() {
        LOGGER.info("enable is called.");

        messageParser.setEscapeHot(false);

        MSG_RAW_LOGGER.info("++++ Enable the message receiver.");

        try {
            receiveBuffer.reset();
        }
        catch (Exception ex) {
            LOGGER.warn("Reset buffered received data failed.", ex);
        }

        try {
            messageParser.reset();
        }
        catch (Exception ex) {
            LOGGER.warn("Reset buffered received data failed.", ex);
        }

        super.enable();
    }

    @Override
    public void disable() {
        LOGGER.info("Disable is called.");
        super.disable();

        MSG_RAW_LOGGER.info("++++ Disable the message receiver.");

        messageParser.setEscapeHot(false);
    }

    public void purgeReceivedDataInBuffer() {
        LOGGER.info("Purge the received data in the message buffer before get the lock.");

        LOGGER
            .info("Purge the received data in the message buffer. Current escapeHot: {}", messageParser.isEscapeHot());

        messageParser.setEscapeHot(false);

        if (receiveBuffer.size() > 0) {

            try {
                byte[] remaining = receiveBuffer.toByteArray();

                String remainingValue = new String(remaining, StandardCharsets.UTF_8);

                LOGGER
                    .info("Cleared remaining data from message output buffer: {}, text: {}",
                        ByteUtils.bytesToHex(remaining), remainingValue);

                if (remainingValue.indexOf("I=") > -1) {
                    // This is the detection of the debug interface active
                    InvalidConfigurationException ice =
                        new InvalidConfigurationException("An active debug interface configuration was detected.");
                    ice.setReason("debug-interface-active");
                    throw ice;
                }
            }
            finally {
                receiveBuffer.reset();
            }

            LOGGER.info("Finished purge output stream.");
        }
        else {
            LOGGER.info("The message buffer is empty.");
        }
    }

    @Override
    public void receive(final ByteArrayOutputStream data) {
        if (!isEnabled()) {
            LOGGER.info("The receiver is not running. Skip processing of messages.");
            try {
                byte[] rawdata = data.toByteArray();
                LOGGER
                    .info("Receiver is stopped, number of bytes read: {}, buffer: {}", rawdata.length,
                        ByteUtils.bytesToHex(rawdata));
            }
            catch (Exception ex) {
                LOGGER.warn("Read data from input stream to buffer failed.", ex);
            }
            return;
        }

        MSG_RAW_LOGGER.info("<<<< start parse input");

        try {
            byte[] rawdata = data.toByteArray();
            parseInput(rawdata, rawdata.length);
        }
        catch (Exception e) {
            LOGGER.warn("Exception detected in message receiver!", e);

            // reset the escapeHot flag
            if (messageParser.isEscapeHot()) {
                LOGGER.warn("Reset the escapeHot to false.");

                messageParser.setEscapeHot(false);
            }
            throw new RuntimeException(e);
        }
        finally {
            MSG_RAW_LOGGER.info("<<<< finished parse input");
        }
    }

    /**
     * Parse the received data to process the received bidib packets.
     * 
     * @param receivedData
     *            the received data
     * @param len
     *            the length of the received data packet
     * @throws ProtocolException
     */
    protected void parseInput(final byte[] receivedData, int len) throws ProtocolException {
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("Parse input, len: {}, data: {}", len, ByteUtils.bytesToHex(receivedData));
        }
        messageParser.parseInput(this, receivedData, len);
    }

    @Override
    protected void processMessage(byte[] messageArray, final String contextKey) throws ProtocolException {
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("Publish the message to the host publisher: {}", ByteUtils.bytesToHex(messageArray));
        }

        BidibMessageInterface bidibMessage = requestFactory.createRaw(messageArray);

        toHostPublisher.publishMessage(bidibMessage);
    }
}
