/*
 * Decompiled with CFR 0.152.
 */
package org.biopax.paxtools.normalizer;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Map;
import java.util.TreeSet;
import org.apache.commons.lang3.StringUtils;
import org.biopax.paxtools.controller.ModelUtils;
import org.biopax.paxtools.controller.ShallowCopy;
import org.biopax.paxtools.converter.LevelUpgrader;
import org.biopax.paxtools.io.SimpleIOHandler;
import org.biopax.paxtools.model.BioPAXElement;
import org.biopax.paxtools.model.BioPAXLevel;
import org.biopax.paxtools.model.Model;
import org.biopax.paxtools.model.level3.BioSource;
import org.biopax.paxtools.model.level3.ControlledVocabulary;
import org.biopax.paxtools.model.level3.EntityReference;
import org.biopax.paxtools.model.level3.Named;
import org.biopax.paxtools.model.level3.NucleicAcidReference;
import org.biopax.paxtools.model.level3.ProteinReference;
import org.biopax.paxtools.model.level3.Provenance;
import org.biopax.paxtools.model.level3.PublicationXref;
import org.biopax.paxtools.model.level3.RelationshipTypeVocabulary;
import org.biopax.paxtools.model.level3.RelationshipXref;
import org.biopax.paxtools.model.level3.SimplePhysicalEntity;
import org.biopax.paxtools.model.level3.SmallMoleculeReference;
import org.biopax.paxtools.model.level3.UnificationXref;
import org.biopax.paxtools.model.level3.XReferrable;
import org.biopax.paxtools.model.level3.Xref;
import org.biopax.paxtools.normalizer.MiriamLink;
import org.biopax.paxtools.util.BPCollections;
import org.biopax.paxtools.util.ClassFilterSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class Normalizer {
    private static final Logger log = LoggerFactory.getLogger(Normalizer.class);
    private SimpleIOHandler biopaxReader = new SimpleIOHandler(BioPAXLevel.L3);
    private String description = "";
    private boolean fixDisplayName;
    private String xmlBase;
    public static final String PROPERTY_NORMALIZER_URI_STRATEGY = "biopax.normalizer.uri.strategy";
    public static final String VALUE_NORMALIZER_URI_STRATEGY_SIMPLE = "simple";
    public static final String VALUE_NORMALIZER_URI_STRATEGY_MD5 = "md5";

    public Normalizer() {
        this.biopaxReader.mergeDuplicates(true);
        this.fixDisplayName = true;
        this.xmlBase = "";
    }

    public String normalize(String biopaxOwlData) {
        if (biopaxOwlData == null || biopaxOwlData.length() == 0) {
            throw new IllegalArgumentException("no data. " + this.description);
        }
        biopaxOwlData = biopaxOwlData.replaceAll("taxonXref", "xref");
        Model model = null;
        try {
            model = this.biopaxReader.convertFromOWL(new ByteArrayInputStream(biopaxOwlData.getBytes("UTF-8")));
        }
        catch (UnsupportedEncodingException e) {
            throw new IllegalArgumentException("Failed! " + this.description, e);
        }
        if (model == null) {
            throw new IllegalArgumentException("Failed to create Model! " + this.description);
        }
        if (model.getLevel() != BioPAXLevel.L3) {
            log.info("Converting model to BioPAX Level3...");
            model = new LevelUpgrader().filter(model);
        }
        this.normalize(model);
        return this.convertToOWL(model);
    }

    private void normalizeXrefs(Model model) {
        NormalizerMap map = new NormalizerMap(model);
        String xmlBase = this.getXmlBase(model);
        HashSet<Xref> xrefs = new HashSet<Xref>(model.getObjects(Xref.class));
        for (Xref ref : xrefs) {
            if (ref.getDb() == null || ref.getId() == null) continue;
            ref.setDb(ref.getDb().toLowerCase());
            String idPart = ref.getId();
            if (ref instanceof RelationshipXref) {
                if (!ref.getUri().startsWith("http://identifiers.org/")) continue;
                RelationshipTypeVocabulary cv = ((RelationshipXref)ref).getRelationshipType();
                if (ref.getIdVersion() != null) {
                    idPart = idPart + "_" + ref.getIdVersion();
                }
                if (cv != null && !cv.getTerm().isEmpty()) {
                    idPart = idPart + "_" + StringUtils.join(cv.getTerm(), '_').toLowerCase();
                }
            } else if (ref instanceof UnificationXref) {
                try {
                    ref.setDb(MiriamLink.getName(ref.getDb()).toLowerCase());
                }
                catch (IllegalArgumentException e) {
                    if (ref.getIdVersion() != null) {
                        idPart = idPart + "_" + ref.getIdVersion();
                    }
                    map.put(ref, Normalizer.uri(xmlBase, ref.getDb(), idPart, ref.getModelInterface()));
                    continue;
                }
                if (ref.getDb().startsWith("uniprot")) {
                    if (this.isValidDbId("uniprot isoform", ref.getId()) && ref.getId().contains("-")) {
                        ref.setDb("uniprot isoform");
                    } else if (ref.getDb().equals("uniprot isoform")) {
                        if (ref.getIdVersion() != null && ref.getIdVersion().matches("^\\d+$")) {
                            idPart = ref.getId() + "-" + ref.getIdVersion();
                        }
                        if (this.isValidDbId(ref.getDb(), idPart)) {
                            ref.setId(idPart);
                            ref.setIdVersion(null);
                        } else if (!this.isValidDbId(ref.getDb(), ref.getId())) {
                            ref.setDb("uniprot knowledgebase");
                        }
                        idPart = ref.getId();
                    }
                }
            }
            map.put(ref, Normalizer.uri(xmlBase, ref.getDb(), idPart, ref.getModelInterface()));
        }
        map.doSubs();
    }

    private boolean isValidDbId(String db, String id) {
        return MiriamLink.checkRegExp(id, db);
    }

    public static String uri(String xmlBase, String dbName, String idPart, Class<? extends BioPAXElement> type) {
        if (type == null || dbName == null && idPart == null) {
            throw new IllegalArgumentException("'Either type' is null, or both dbName and idPart are nulls.");
        }
        if (idPart != null) {
            idPart = idPart.trim();
        }
        if (dbName != null) {
            dbName = dbName.trim();
        }
        if (dbName != null) {
            try {
                dbName = MiriamLink.getName(dbName);
                if (type.equals(PublicationXref.class) && "pubmed".equalsIgnoreCase(dbName) || type.equals(RelationshipTypeVocabulary.class) || ProteinReference.class.isAssignableFrom(type) || SmallMoleculeReference.class.isAssignableFrom(type) || type.equals(BioSource.class) && "taxonomy".equalsIgnoreCase(dbName) && idPart != null && idPart.matches("^\\d+$")) {
                    return MiriamLink.getIdentifiersOrgURI(dbName, idPart);
                }
            }
            catch (IllegalArgumentException e) {
                log.info(String.format("uri(for a %s): db:%s, id:%s are not standard; %s)", type.getSimpleName(), dbName, idPart, e.getMessage()));
            }
        }
        StringBuilder sb = new StringBuilder();
        if (dbName != null) {
            sb.append(dbName.toLowerCase());
        }
        if (idPart != null) {
            if (dbName != null) {
                sb.append("_");
            }
            sb.append(idPart);
        }
        String localPart = sb.toString();
        String strategy = System.getProperty(PROPERTY_NORMALIZER_URI_STRATEGY, VALUE_NORMALIZER_URI_STRATEGY_MD5);
        localPart = VALUE_NORMALIZER_URI_STRATEGY_SIMPLE.equals(strategy) || Xref.class.isAssignableFrom(type) ? localPart.replaceAll("[^-\\w]", "_") : ModelUtils.md5hex(localPart);
        return (xmlBase != null ? xmlBase : "") + type.getSimpleName() + "_" + localPart;
    }

    public String getDescription() {
        return this.description;
    }

    public void setDescription(String description) {
        this.description = description;
    }

    private void fixDisplayName(Model model) {
        log.info("Trying to auto-fix 'null' displayName...");
        for (Named e : model.getObjects(Named.class)) {
            if (e.getDisplayName() != null) continue;
            if (e.getStandardName() != null) {
                e.setDisplayName(e.getStandardName());
                log.info(e + " displayName auto-fix: " + e.getDisplayName() + ". " + this.description);
                continue;
            }
            if (e.getName().isEmpty()) continue;
            String dsp = e.getName().iterator().next();
            for (String name : e.getName()) {
                if (name.length() >= dsp.length()) continue;
                dsp = name;
            }
            e.setDisplayName(dsp);
            log.info(e + " displayName auto-fix: " + dsp + ". " + this.description);
        }
        for (EntityReference er : model.getObjects(EntityReference.class)) {
            for (SimplePhysicalEntity spe : er.getEntityReferenceOf()) {
                if (spe.getDisplayName() != null && spe.getDisplayName().trim().length() != 0 || er.getDisplayName() == null || er.getDisplayName().trim().length() <= 0) continue;
                spe.setDisplayName(er.getDisplayName());
            }
        }
    }

    private String convertToOWL(Model model) {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        new SimpleIOHandler(model.getLevel()).convertToOWL(model, out);
        return out.toString();
    }

    private Collection<UnificationXref> getUnificationXrefsSorted(XReferrable r) {
        ArrayList<UnificationXref> urefs = new ArrayList<UnificationXref>();
        for (UnificationXref ux : new ClassFilterSet<Xref, UnificationXref>(r.getXref(), UnificationXref.class)) {
            if (ux.getDb() == null || ux.getId() == null) continue;
            urefs.add(ux);
        }
        Collections.sort(urefs, new Comparator<UnificationXref>(){

            @Override
            public int compare(UnificationXref o1, UnificationXref o2) {
                String s1 = o1.getDb() + o1.getId();
                String s2 = o2.getDb() + o2.getId();
                return s1.compareTo(s2);
            }
        });
        return urefs;
    }

    private UnificationXref findPreferredUnificationXref(XReferrable bpe) {
        UnificationXref toReturn = null;
        Collection<UnificationXref> orderedUrefs = this.getUnificationXrefsSorted(bpe);
        if (bpe instanceof ProteinReference) {
            toReturn = this.findSingleUnificationXref(orderedUrefs, "uniprot");
            if (toReturn == null) {
                toReturn = this.findSingleUnificationXref(orderedUrefs, "refseq");
            }
        } else if (bpe instanceof SmallMoleculeReference) {
            toReturn = this.findSingleUnificationXref(orderedUrefs, "chebi");
            if (toReturn == null) {
                toReturn = this.findSingleUnificationXref(orderedUrefs, "pubchem");
            }
        } else if (bpe instanceof NucleicAcidReference) {
            toReturn = this.findSingleUnificationXref(orderedUrefs, "ncbi gene");
            if (toReturn == null) {
                toReturn = this.findSingleUnificationXref(orderedUrefs, "entrez");
            }
        } else if (orderedUrefs.size() == 1) {
            toReturn = orderedUrefs.iterator().next();
        }
        return toReturn;
    }

    private UnificationXref findSingleUnificationXref(Collection<UnificationXref> uXrefs, String dbStartsWith) {
        Xref ret = null;
        for (Xref xref2 : uXrefs) {
            if (!(xref2 instanceof UnificationXref) || xref2.getId() == null || xref2.getDb() == null || !xref2.getDb().toLowerCase().startsWith(dbStartsWith)) continue;
            if (ret == null) {
                ret = (UnificationXref)xref2;
                continue;
            }
            if (!ret.getDb().equalsIgnoreCase(xref2.getDb()) || ret.getId().equals(xref2.getId())) continue;
            ret = null;
            break;
        }
        return ret;
    }

    public void normalize(Model model) {
        if (model.getLevel() != BioPAXLevel.L3) {
            throw new IllegalArgumentException("Not Level3 model. Consider converting it first (e.g., with the PaxTools).");
        }
        if (this.xmlBase != null && !this.xmlBase.isEmpty()) {
            model.setXmlBase(this.xmlBase);
        }
        log.info("Normalizing xrefs..." + this.description);
        this.normalizeXrefs(model);
        if (this.fixDisplayName) {
            log.info("Normalizing display names..." + this.description);
            this.fixDisplayName(model);
        }
        log.info("Normalizing CVs..." + this.description);
        this.normalizeCVs(model);
        log.info("Normalizing organisms..." + this.description);
        this.normalizeBioSources(model);
        for (SimplePhysicalEntity spe : new HashSet<SimplePhysicalEntity>(model.getObjects(SimplePhysicalEntity.class))) {
            ModelUtils.addMissingEntityReference(model, spe);
        }
        log.info("Normalizing entity references..." + this.description);
        this.normalizeERs(model);
        log.info("Repairing..." + this.description);
        model.repair();
        log.info("Optional tasks (reasoning)..." + this.description);
    }

    private void normalizeCVs(Model model) {
        NormalizerMap map = new NormalizerMap(model);
        for (ControlledVocabulary cv : model.getObjects(ControlledVocabulary.class)) {
            UnificationXref uref = this.findPreferredUnificationXref(cv);
            if (uref != null) {
                map.put(cv, Normalizer.uri(this.xmlBase, uref.getDb(), uref.getId(), cv.getModelInterface()));
                continue;
            }
            if (!cv.getTerm().isEmpty()) {
                map.put(cv, Normalizer.uri(this.xmlBase, null, cv.getTerm().iterator().next(), cv.getModelInterface()));
                continue;
            }
            log.info("Cannot normalize " + cv.getModelInterface().getSimpleName() + " : no unification xrefs nor terms found in " + cv.getUri() + ". " + this.description);
        }
        map.doSubs();
    }

    private void normalizeBioSources(Model model) {
        NormalizerMap map = new NormalizerMap(model);
        for (BioSource bs : model.getObjects(BioSource.class)) {
            UnificationXref uref = this.findPreferredUnificationXref(bs);
            if (uref != null && (uref.getDb().toLowerCase().contains("taxonomy") || uref.getDb().equalsIgnoreCase("newt"))) {
                String idPart = uref.getId();
                if (bs.getTissue() != null && !bs.getTissue().getTerm().isEmpty()) {
                    idPart = idPart + "_" + bs.getTissue().getTerm().iterator().next();
                }
                if (bs.getCellType() != null && !bs.getCellType().getTerm().isEmpty()) {
                    idPart = idPart + "_" + bs.getCellType().getTerm().iterator().next();
                }
                String uri = idPart.equals(uref.getId()) && idPart.matches("^\\d+$") ? Normalizer.uri(this.xmlBase, uref.getDb(), idPart, BioSource.class) : "http://identifiers.org/taxonomy/" + idPart;
                map.put(bs, uri);
                continue;
            }
            log.debug("Won't normalize BioSource : no taxonomy unification xref found in " + bs.getUri() + ". " + this.description);
        }
        map.doSubs();
    }

    private void normalizeERs(Model model) {
        NormalizerMap map = new NormalizerMap(model);
        for (EntityReference bpe : model.getObjects(EntityReference.class)) {
            if (bpe.getUri().startsWith("http://identifiers.org/")) {
                log.info("Skip already normalized: " + bpe.getUri());
                continue;
            }
            UnificationXref uref = this.findPreferredUnificationXref(bpe);
            if (uref != null) {
                String db = uref.getDb();
                String id = uref.getId();
                String uri = null;
                try {
                    uri = MiriamLink.getIdentifiersOrgURI(db, id);
                }
                catch (Exception e) {
                    log.error("Cannot get a Miriam standard ID for " + bpe + " (" + bpe.getModelInterface().getSimpleName() + ") , using " + db + ":" + id + ". " + e.getMessage());
                    return;
                }
                if (uri == null) continue;
                map.put(bpe, uri);
                continue;
            }
            log.info("Cannot normalize EntityReference: no unification xrefs found in " + bpe.getUri() + ". " + this.description);
        }
        map.doSubs();
    }

    public static void autoName(Provenance pro) {
        if (!pro.getUri().startsWith("urn:miriam:") && !pro.getUri().startsWith("http://identifiers.org/") && pro.getName().isEmpty()) {
            log.info("Skipping: cannot normalize Provenance: " + pro.getUri());
        } else {
            TreeSet<String> names = new TreeSet<String>();
            String key = null;
            key = pro.getUri().startsWith("urn:miriam:") || pro.getUri().startsWith("http://identifiers.org/") ? pro.getUri() : (pro.getStandardName() != null ? pro.getStandardName() : pro.getDisplayName());
            if (key != null) {
                try {
                    names.addAll(Arrays.asList(MiriamLink.getNames(key)));
                    pro.setStandardName(MiriamLink.getName(key));
                    String description = MiriamLink.getDataTypeDef(pro.getStandardName());
                    pro.addComment(description);
                }
                catch (IllegalArgumentException illegalArgumentException) {
                    // empty catch block
                }
            }
            if (names.isEmpty()) {
                for (String name : pro.getName()) {
                    try {
                        names.addAll(Arrays.asList(MiriamLink.getNames(name)));
                    }
                    catch (IllegalArgumentException illegalArgumentException) {}
                }
                if (!names.isEmpty()) {
                    pro.setStandardName(MiriamLink.getName((String)names.iterator().next()));
                }
            }
            for (String name : names) {
                pro.addName(name);
            }
            if (pro.getDisplayName() == null) {
                pro.setDisplayName(pro.getStandardName());
            }
        }
    }

    public static String convertToLevel3(String biopaxData) {
        String toReturn = "";
        try {
            ByteArrayOutputStream os = new ByteArrayOutputStream();
            ByteArrayInputStream is = new ByteArrayInputStream(biopaxData.getBytes());
            SimpleIOHandler io = new SimpleIOHandler();
            io.mergeDuplicates(true);
            Model model = io.convertFromOWL(is);
            if (model.getLevel() != BioPAXLevel.L3) {
                log.info("Converting to BioPAX Level3... " + model.getXmlBase());
                model = new LevelUpgrader().filter(model);
                if (model != null) {
                    io.setFactory(model.getLevel().getDefaultFactory());
                    io.convertToOWL(model, os);
                    toReturn = os.toString();
                }
            } else {
                toReturn = biopaxData;
            }
        }
        catch (Exception e) {
            throw new RuntimeException("Cannot convert to BioPAX Level3", e);
        }
        return toReturn;
    }

    private String getXmlBase(Model modelToNormalize) {
        if (this.xmlBase != null) {
            return this.xmlBase;
        }
        return "";
    }

    public boolean isFixDisplayName() {
        return this.fixDisplayName;
    }

    public void setFixDisplayName(boolean fixDisplayName) {
        this.fixDisplayName = fixDisplayName;
    }

    public String getXmlBase() {
        return this.xmlBase;
    }

    public void setXmlBase(String xmlBase) {
        this.xmlBase = xmlBase;
    }

    private static class NormalizerMap {
        final Model model;
        final Map<BioPAXElement, BioPAXElement> subs = BPCollections.I.createMap();
        final Map<String, BioPAXElement> uriToSub = BPCollections.I.createMap();
        final ShallowCopy copier;

        NormalizerMap(Model model) {
            this.model = model;
            this.copier = new ShallowCopy();
        }

        void put(BioPAXElement bpe, String newUri) {
            if (this.model.containsID(newUri)) {
                this.map(bpe, this.model.getByID(newUri));
            } else if (this.uriToSub.containsKey(newUri)) {
                this.map(bpe, this.uriToSub.get(newUri));
            } else {
                BioPAXElement copy = this.copier.copy(bpe, newUri);
                this.map(bpe, copy);
            }
        }

        void doSubs() {
            for (BioPAXElement e : this.subs.keySet()) {
                this.model.remove(e);
            }
            try {
                ModelUtils.replace(this.model, this.subs);
            }
            catch (Exception e) {
                log.error("Failed to replace BioPAX elements.", e);
                return;
            }
            for (BioPAXElement e : this.subs.values()) {
                if (this.model.contains(e)) continue;
                this.model.add(e);
            }
            for (BioPAXElement e : this.model.getObjects()) {
                ModelUtils.fixDanglingInverseProperties(e, this.model);
            }
        }

        private void map(BioPAXElement bpe, BioPAXElement newBpe) {
            this.subs.put(bpe, newBpe);
            this.uriToSub.put(newBpe.getUri(), newBpe);
        }
    }
}

