/*
 * Decompiled with CFR 0.152.
 */
package com.subgraph.orchid.connections;

import com.subgraph.orchid.Cell;
import com.subgraph.orchid.Circuit;
import com.subgraph.orchid.Connection;
import com.subgraph.orchid.ConnectionFailedException;
import com.subgraph.orchid.ConnectionHandshakeException;
import com.subgraph.orchid.ConnectionIOException;
import com.subgraph.orchid.ConnectionTimeoutException;
import com.subgraph.orchid.Router;
import com.subgraph.orchid.Threading;
import com.subgraph.orchid.TorConfig;
import com.subgraph.orchid.TorException;
import com.subgraph.orchid.circuits.TorInitializationTracker;
import com.subgraph.orchid.circuits.cells.CellImpl;
import com.subgraph.orchid.connections.ConnectionHandshake;
import com.subgraph.orchid.crypto.TorRandom;
import com.subgraph.orchid.dashboard.DashboardRenderable;
import com.subgraph.orchid.dashboard.DashboardRenderer;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.net.SocketTimeoutException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.net.ssl.SSLSocket;

public class ConnectionImpl
implements Connection,
DashboardRenderable {
    private static final Logger logger = Logger.getLogger(ConnectionImpl.class.getName());
    private static final int CONNECTION_IDLE_TIMEOUT = 300000;
    private static final int DEFAULT_CONNECT_TIMEOUT = 5000;
    private static final Cell connectionClosedSentinel = CellImpl.createCell(0, 0);
    private final TorConfig config;
    private final SSLSocket socket;
    private InputStream input;
    private OutputStream output;
    private final Router router;
    private final Map<Integer, Circuit> circuitMap;
    private final BlockingQueue<Cell> connectionControlCells;
    private final TorInitializationTracker initializationTracker;
    private final boolean isDirectoryConnection;
    private int currentId = 1;
    private boolean isConnected;
    private volatile boolean isClosed;
    private final Thread readCellsThread;
    private final ReentrantLock connectLock = Threading.lock("connect");
    private final ReentrantLock circuitsLock = Threading.lock("circuits");
    private final ReentrantLock outputLock = Threading.lock("output");
    private final AtomicLong lastActivity = new AtomicLong();

    public ConnectionImpl(TorConfig config, SSLSocket socket, Router router, TorInitializationTracker tracker, boolean isDirectoryConnection) {
        this.config = config;
        this.socket = socket;
        this.router = router;
        this.circuitMap = new HashMap<Integer, Circuit>();
        this.readCellsThread = new Thread(this.createReadCellsRunnable());
        this.readCellsThread.setDaemon(true);
        this.connectionControlCells = new LinkedBlockingQueue<Cell>();
        this.initializationTracker = tracker;
        this.isDirectoryConnection = isDirectoryConnection;
        this.initializeCurrentCircuitId();
    }

    private void initializeCurrentCircuitId() {
        TorRandom random = new TorRandom();
        this.currentId = random.nextInt(65535) + 1;
    }

    @Override
    public Router getRouter() {
        return this.router;
    }

    @Override
    public boolean isClosed() {
        return this.isClosed;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public int bindCircuit(Circuit circuit) {
        this.circuitsLock.lock();
        try {
            while (this.circuitMap.containsKey(this.currentId)) {
                this.incrementNextId();
            }
            int id = this.currentId;
            this.incrementNextId();
            this.circuitMap.put(id, circuit);
            int n = id;
            return n;
        }
        finally {
            this.circuitsLock.unlock();
        }
    }

    private void incrementNextId() {
        ++this.currentId;
        if (this.currentId > 65535) {
            this.currentId = 1;
        }
    }

    void connect() throws ConnectionFailedException, ConnectionTimeoutException, ConnectionHandshakeException {
        this.connectLock.lock();
        try {
            if (this.isConnected) {
                return;
            }
            try {
                this.doConnect();
            }
            catch (SocketTimeoutException e) {
                throw new ConnectionTimeoutException();
            }
            catch (IOException e) {
                throw new ConnectionFailedException(e.getClass().getName() + " : " + e.getMessage());
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new ConnectionHandshakeException("Handshake interrupted");
            }
            catch (ConnectionHandshakeException e) {
                throw e;
            }
            catch (ConnectionIOException e) {
                throw new ConnectionFailedException(e.getMessage());
            }
            this.isConnected = true;
        }
        finally {
            this.connectLock.unlock();
        }
    }

    private void doConnect() throws IOException, InterruptedException, ConnectionIOException {
        this.connectSocket();
        ConnectionHandshake handshake = ConnectionHandshake.createHandshake(this.config, this, this.socket);
        this.input = this.socket.getInputStream();
        this.output = this.socket.getOutputStream();
        this.readCellsThread.start();
        handshake.runHandshake();
        this.updateLastActivity();
    }

    private void connectSocket() throws IOException {
        if (this.initializationTracker != null) {
            if (this.isDirectoryConnection) {
                this.initializationTracker.notifyEvent(5);
            } else {
                this.initializationTracker.notifyEvent(80);
            }
        }
        this.socket.connect(this.routerToSocketAddress(this.router), 5000);
        if (this.initializationTracker != null) {
            if (this.isDirectoryConnection) {
                this.initializationTracker.notifyEvent(10);
            } else {
                this.initializationTracker.notifyEvent(85);
            }
        }
    }

    private SocketAddress routerToSocketAddress(Router router) {
        InetAddress address = router.getAddress().toInetAddress();
        return new InetSocketAddress(address, router.getOnionPort());
    }

    @Override
    public void sendCell(Cell cell) throws ConnectionIOException {
        if (!this.socket.isConnected()) {
            throw new ConnectionIOException("Cannot send cell because connection is not connected");
        }
        this.updateLastActivity();
        this.outputLock.lock();
        try {
            try {
                this.output.write(cell.getCellBytes());
            }
            catch (IOException e) {
                logger.fine("IOException writing cell to connection " + e.getMessage());
                this.closeSocket();
                throw new ConnectionIOException(e.getClass().getName() + " : " + e.getMessage());
            }
        }
        finally {
            this.outputLock.unlock();
        }
    }

    private Cell recvCell() throws ConnectionIOException {
        try {
            return CellImpl.readFromInputStream(this.input);
        }
        catch (EOFException e) {
            this.closeSocket();
            throw new ConnectionIOException();
        }
        catch (IOException e) {
            if (!this.isClosed) {
                logger.fine("IOException reading cell from connection " + this + " : " + e.getMessage());
                this.closeSocket();
            }
            throw new ConnectionIOException(e.getClass().getName() + " " + e.getMessage());
        }
    }

    void closeSocket() {
        try {
            logger.fine("Closing connection to " + this);
            this.isClosed = true;
            this.socket.close();
            this.isConnected = false;
        }
        catch (IOException e) {
            logger.warning("Error closing socket: " + e.getMessage());
        }
    }

    private Runnable createReadCellsRunnable() {
        return new Runnable(){

            @Override
            public void run() {
                try {
                    ConnectionImpl.this.readCellsLoop();
                }
                catch (Exception e) {
                    logger.log(Level.WARNING, "Unhandled exception processing incoming cells on connection " + e, e);
                }
            }
        };
    }

    private void readCellsLoop() {
        while (!Thread.interrupted()) {
            try {
                this.processCell(this.recvCell());
            }
            catch (ConnectionIOException e) {
                this.connectionControlCells.add(connectionClosedSentinel);
                this.notifyCircuitsLinkClosed();
                return;
            }
            catch (TorException e) {
                logger.log(Level.WARNING, "Unhandled Tor exception reading and processing cells: " + e.getMessage(), e);
            }
        }
    }

    private void notifyCircuitsLinkClosed() {
    }

    Cell readConnectionControlCell() throws ConnectionIOException {
        try {
            return this.connectionControlCells.take();
        }
        catch (InterruptedException e) {
            this.closeSocket();
            throw new ConnectionIOException();
        }
    }

    private void processCell(Cell cell) {
        this.updateLastActivity();
        int command = cell.getCommand();
        if (command == 3) {
            this.processRelayCell(cell);
            return;
        }
        switch (command) {
            case 7: 
            case 8: 
            case 129: 
            case 130: {
                this.connectionControlCells.add(cell);
                break;
            }
            case 2: 
            case 4: 
            case 6: {
                this.processControlCell(cell);
                break;
            }
        }
    }

    private void processRelayCell(Cell cell) {
        Circuit circuit;
        this.circuitsLock.lock();
        try {
            circuit = this.circuitMap.get(cell.getCircuitId());
            if (circuit == null) {
                logger.warning("Could not deliver relay cell for circuit id = " + cell.getCircuitId() + " on connection " + this + ". Circuit not found");
                return;
            }
        }
        finally {
            this.circuitsLock.unlock();
        }
        circuit.deliverRelayCell(cell);
    }

    private void processControlCell(Cell cell) {
        Circuit circuit;
        this.circuitsLock.lock();
        try {
            circuit = this.circuitMap.get(cell.getCircuitId());
        }
        finally {
            this.circuitsLock.unlock();
        }
        if (circuit != null) {
            circuit.deliverControlCell(cell);
        }
    }

    void idleCloseCheck() {
        this.circuitsLock.lock();
        try {
            boolean needClose;
            boolean bl = needClose = !this.isClosed && this.circuitMap.isEmpty() && this.getIdleMilliseconds() > 300000L;
            if (needClose) {
                logger.fine("Closing connection to " + this + " on idle timeout");
                this.closeSocket();
            }
        }
        finally {
            this.circuitsLock.unlock();
        }
    }

    private void updateLastActivity() {
        this.lastActivity.set(System.currentTimeMillis());
    }

    private long getIdleMilliseconds() {
        if (this.lastActivity.get() == 0L) {
            return 0L;
        }
        return System.currentTimeMillis() - this.lastActivity.get();
    }

    @Override
    public void removeCircuit(Circuit circuit) {
        this.circuitsLock.lock();
        try {
            this.circuitMap.remove(circuit.getCircuitId());
        }
        finally {
            this.circuitsLock.unlock();
        }
    }

    public String toString() {
        return "!" + this.router.getNickname() + "!";
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void dashboardRender(DashboardRenderer renderer, PrintWriter writer, int flags) throws IOException {
        int circuitCount;
        this.circuitsLock.lock();
        try {
            circuitCount = this.circuitMap.size();
        }
        finally {
            this.circuitsLock.unlock();
        }
        if (circuitCount == 0 && (flags & 2) == 0) {
            return;
        }
        writer.print("  [Connection router=" + this.router.getNickname());
        writer.print(" circuits=" + circuitCount);
        writer.print(" idle=" + this.getIdleMilliseconds() / 1000L + "s");
        writer.println("]");
    }
}

