/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.specification.eoa.application.spring;

import javax.xml.namespace.QName;

/**
 * Models a spring application instance implemented by engine at runtime. Note, for access to
 * the EOA 'middleware' api, <code>MyApplication</code> can be cast to an instance of
 * {@link SpringEngine}.
 * 
 */
public interface MyApplication {

    /**
     * Gets name of 'my' application, i.e. name of runtime engine which implements 'my'
     * application.
     * 
     * @return
     */
    public QName getName();

    /**
     * Returns a partner application, i.e. an object which models an executable partner
     * application instance implemented by named engine.
     * <p>
     * Note that this application is modeled as a 'partner' application where partner name
     * equals {@link #getName()}, i.e. to allow for the invocation of self. This MAY be
     * necessary such that request and corresponding response are processed by components
     * within the EOA binding and application layer, e.g. binding modules such as ws-a
     * and any application modules which may be defined by application.
     * 
     * @param engineName
     *        name of engine which implements partner application
     * 
     * @return PartnerApplication or <code>null</code> if undefined.
     */
    public PartnerApplication getPartner(QName engineName);

    /**
     * Convenience method which retrieves a 'spring' bean from root application context by
     * name. Note that an alternative, but less convenient, approach would be to instead cast
     * <code>MyApplication</code> to {@link SpringEngine} for access to root application
     * context.
     * 
     * @param name
     *        bean name
     * 
     * @return bean or <code>null</code> if undefined
     */
    public Object getBean(String name);

    /**
     * Retrieves an instance of <code>MyRole</code> implemented by this application.
     * 
     * @param roleName
     * 
     * @return MyRole or <code>null</code> if role undefined.
     */
    public MyRole getMyRole(String roleName);

}
