/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.specification.eoa.application.spring;

import javax.xml.namespace.QName;

import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointReference;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceOperation;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.ActionContext;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.FaultContext;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.MessageContext;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.SystemFault;
import org.bluestemsoftware.specification.eoa.ext.binding.http.rt.HTTPBindingRT;
import org.bluestemsoftware.specification.eoa.ext.binding.soap.rt.SOAPHTTPBindingRT;
import org.w3c.dom.Element;

/**
 * Models a 'message exchange' instance the abstract definition of which is defined on an
 * interface implemented by a partner service. The exchange is initiated by 'my' application.
 */
public interface PartnerOperation extends MessageExchange {

    /**
     * Convenience method which returns name of partner operation.
     * 
     * @return
     */
    public QName getName();

    /**
     * Gets metadata which statically describes partner operation.
     * 
     * @return
     */
    public InterfaceOperation getMetaData();

    /**
     * Creates a request action instance which can be used to invoke partner operation i.e. via
     * {@link #sendRequest(ActionContext)}.
     * 
     * @return
     */
    public ActionContext createRequestAction();

    /**
     * Invokes partner operation without blocking for a response. Partner response, if any, is
     * delivered asynchronously to user defined class mapped to partner service reference.
     * <p>
     * Note that the concept of a 'blocking' versus 'non-blocking' invocation has no bearing on
     * how the underlying binding actually transmits request, i.e. whether or not anonymous
     * responses are used. Binding behavior is constrained via ws-policy.
     * <p>
     * Note that when using this method, if partner application upon which operation is defined
     * models 'my' application, i.e. to allow for invocation of self, the MEP must be
     * 'in-only'. If MEP is either 'in-out', or 'robust-in-only', the other send methods which
     * employ a blocking invocation style MUST be used.
     * 
     * @param request
     *        request action created via {@link #createRequestAction()}
     * 
     * @throws SystemFault
     *         If request was not accepted for processing by receiving node, e.g. due to a
     *         transport failure, malformed request, etc ... Note that the rules which define
     *         'accepted for processing' are specific to binding used to transmit request.
     * 
     * @see SOAPHTTPBindingRT#sendAction(EndpointReference, ActionContext)
     * @see HTTPBindingRT#sendAction(EndpointReference, ActionContext)
     */
    public void sendRequest(ActionContext request) throws SystemFault;

    /**
     * Invokes partner operation and blocks for a response, if any.
     * <p>
     * Note that the concept of a 'blocking' versus 'non-blocking' invocation has no bearing on
     * how the underlying binding actually transmits request, i.e. whether or not anonymous
     * responses are used. Binding behavior is constrained via ws-policy.
     * 
     * @param request
     *        request action created via {@link #createRequestAction()}
     * @param long
     *        time in milliseconds to wait for response, or zero to block indefinitely (or
     *        until interrupted, e.g. during a system shutdown).
     * 
     * @return ActionContext or <code>null</code> if MEP is 'in-only' or if MEP is
     *         'robust-in-only' and no fault was returned by partner. If not <code>null</code>,
     *         the response will either be an instance of {@link MessageContext} or an instance
     *         of {@link FaultContext}.
     * @throws SystemFault
     *         If request was not accepted for processing by receiving node, e.g. due to a
     *         transport failure, malformed request, etc ... Note that the rules which define
     *         'accepted for processing' are specific to binding used to transmit request.
     * 
     * @see SOAPHTTPBindingRT#sendAction(EndpointReference, ActionContext)
     * @see HTTPBindingRT#sendAction(EndpointReference, ActionContext)
     */
    public ActionContext sendRequest(ActionContext request, long timeout) throws SystemFault;

    /**
     * An alternative api which obviates the need to reference classes defined within the
     * 'middleware' api, i.e. within the EOA specification. Invokes partner operation and
     * blocks for a response.
     * <p>
     * Note that the concept of a 'blocking' versus 'non-blocking' invocation has no bearing on
     * how the underlying binding actually transmits the request, i.e. whether or not anonymous
     * responses are used. Binding behavior is constrained via ws-policy.
     * 
     * @param request
     *        request payload
     * @param long
     *        time in milliseconds to wait for response, or zero to block indefinitely (or
     *        until interrupted, e.g. during a system shutdown).
     * 
     * @return Element which represents normal response or <code>null</code> if MEP is
     *         in-only.
     * 
     * @throws PartnerServiceFault
     *         Thrown if MEP is 'in-out' or 'robust-in-only' and partner returns a fault
     *         declared on partner operation.
     * 
     * @throws SystemFault
     *         If request was not accepted for processing by receiving node, e.g. due to a
     *         transport failure, malformed request, etc ... or if partner returns an
     *         undeclared fault response, or if MEP 'in-out' or 'robust-in-only' and engine
     *         timed out waiting for response. Note that the rules which define 'accepted for
     *         processing' are specific to binding used to use request.
     * 
     * @see SOAPHTTPBindingRT#sendAction(EndpointReference, ActionContext)
     * @see HTTPBindingRT#sendAction(EndpointReference, ActionContext)
     */
    public Element sendRequest(Element request, long timeout) throws PartnerFault;

}
