/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.specification.eoa.application.spring;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.Iterator;

import javax.xml.namespace.QName;

import org.bluestemsoftware.specification.eoa.application.spring.MyOperation;
import org.bluestemsoftware.specification.eoa.component.engine.Engine;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointActionReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointOperationReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EngineRT;
import org.bluestemsoftware.specification.eoa.component.engine.rt.ServiceReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointActionReference.ResponseActionReference;
import org.bluestemsoftware.specification.eoa.component.intrface.Interface;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceOperation;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.ActionContext;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.FaultContext;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.MessageContext;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.SystemFault;
import org.bluestemsoftware.specification.eoa.ext.binding.http.rt.HTTPBindingRT;
import org.bluestemsoftware.specification.eoa.ext.binding.soap.rt.SOAPHTTPBindingRT;
import org.bluestemsoftware.specification.eoa.system.SystemContext;

/**
 * Models a 'message exhange' instance the abstract definition of which is defined on an
 * interface implemented by 'my' service. Note that this class MAY be externalized and
 * reconsititued as required, e.g. for 'long lived' exchanges.
 */
public class MyOperation implements MessageExchange, Externalizable {

    private static final long serialVersionUID = 1L;

    private String id;
    private String relatesTo;
    private Interface intrface;
    private EndpointOperationReference eor;

    /**
     * Default constructor required for externalization.
     */
    public MyOperation() {
    }

    public MyOperation(EndpointOperationReference eor, String relatesTo) {
        if (eor == null) {
            throw new IllegalArgumentException("eor null");
        }
        if (relatesTo == null || relatesTo.equals("")) {
            throw new IllegalArgumentException("relatesTo null or empty string");
        }
        this.eor = eor;
        this.relatesTo = relatesTo;
        this.intrface = eor.getEndpointOperation().getInterfaceOperation().getParent();
    }

    /**
     * Gets operation name.
     * @return
     */
    public QName getName() {
        return eor.getEndpointOperationName();
    }

    /**
     * Gets 'metadata' which provides an abstract description of this operation.
     * 
     * @return metadata
     */
    public InterfaceOperation getMetaData() {
        return intrface.getOperation(eor.getEndpointOperationName());
    }

    /**
     * Retrieves metadata which describes 'my' role and which can be used to gain access to the
     * 'middleware' api, i.e. to EOA specifacation api.
     * 
     * @return
     */
    public MyRole getMyRole() {
        return (MyRole)eor.getParent().getParent();
    }

    /**
     * Creates action with message direction 'out' for an 'in-out' MEP or a 'robust-in-only'
     * MEP. Action may represent either a normal response or a fault response.
     * 
     * @param action
     *        which uniquely identifies semantics of message within context of interface.
     * 
     * @return
     */
    public ActionContext createAction(String action) {
        return getResponseActionReference(action).createAction(relatesTo);
    }

    /**
     * Convenience method which creates 'normal' response action, i.e. non-fault. Message
     * direction is 'out' and MEP is 'in-out'.
     * 
     * @return
     */
    public MessageContext createResponseAction() {
        EndpointActionReference ear = eor.getResponseAction();
        if (ear == null) {
            throw new IllegalArgumentException("Response action for operation "
                    + eor.getEndpointOperationName()
                    + " is undefined.");
        }
        return (MessageContext)((ResponseActionReference)ear).createAction(relatesTo);
    }

    /**
     * Convenience method which creates a 'business fault' response action. Message direction
     * is 'out' and MEP is 'in-out' or 'robust-in-only'.
     * 
     * @param faultName
     * @return
     */
    public FaultContext createFaultAction(QName faultName) {
        EndpointActionReference ear = eor.getFaultAction(faultName);
        if (ear == null) {
            throw new IllegalArgumentException("Fault ref "
                    + faultName
                    + " is undefined on operation "
                    + eor.getEndpointOperationName()
                    + ".");
        }
        return (FaultContext)((ResponseActionReference)ear).createAction(relatesTo);
    }

    /**
     * Constructs an instance of undeclared action {@link SystemFault#ACTION}.
     * 
     * @param faultReason
     *        fault description
     * @return FaultContext
     */
    public FaultContext createSystemFaultAction(String faultReason) {
        return eor.getParent().createSystemFaultAction(relatesTo, faultReason);
    }

    /**
     * Constructs an instance of undeclared action {@link SystemFault#ACTION}.
     * 
     * @param faultReason
     *        fault description
     * @param faultCode
     *        A specific indication of fault, i.e. a fault sub-type, within context of the
     *        general classification {@link #NAME}.
     * @return FaultContext
     */
    public FaultContext createSystemFaultAction(QName faultCode, String faultReason) {
        return eor.getParent().createSystemFaultAction(relatesTo, faultCode, faultReason);
    }

    /**
     * Constructs an instance of undeclared action {@link SystemFault#ACTION}.
     * 
     * @param faultReason
     *        fault description
     * @param cause
     *        Throwable which caused fault. Note throwable MAY be an instanceof
     *        <code>SystemFault</code>, i.e. a 'nested' fault.
     *        <p>
     *        Note that if <code>Throwable</code> is not an instance of
     *        <code>SystemFault</code>, a stacktrace is serialized to 'FaultDetail' element
     *        of message returned via {@link SystemFault#toMessage()}.
     * @return FaultContext
     */
    public FaultContext createSystemFaultAction(String faultReason, Throwable cause) {
        return eor.getParent().createSystemFaultAction(relatesTo, faultReason, cause);
    }

    /**
     * Constructs an instance of undeclared action {@link SystemFault#ACTION}.
     * 
     * @param cause
     *        Throwable which caused fault. Note throwable MAY be an instanceof
     *        <code>SystemFault</code>.
     *        <p>
     *        Note that if <code>Throwable</code> is not an instance of
     *        <code>SystemFault</code>, a stacktrace is serialized to 'FaultDetail' element
     *        of message returned via {@link SystemFault#toMessage()}.
     * @return FaultContext
     */
    public FaultContext createSystemFaultAction(Throwable cause) {
        return eor.getParent().createSystemFaultAction(relatesTo, cause);
    }

    /**
     * Constructs an instance of undeclared action {@link SystemFault#ACTION}.
     * 
     * @param systemFault
     * 
     * @return FaultContext
     */
    public FaultContext createSystemFaultAction(SystemFault systemFault) {
        return eor.getParent().createSystemFaultAction(relatesTo, systemFault);
    }

    /**
     * Completes the exchange by returning response to client that initiated the message
     * exchange.
     * 
     * @param responseAction
     * @throws SystemFault
     *         if response was 'not accepted for processing' by receiving node, e.g. due to a
     *         transport failure, malformed response, etc ... Note that the rules which define
     *         'accepted for processing' are specific to binding used to transmit response.
     * 
     * @see SOAPHTTPBindingRT#sendAction(EndpointReference, ActionContext)
     * @see HTTPBindingRT#sendAction(EndpointReference, ActionContext)
     */
    public void sendResponse(ActionContext responseAction) throws SystemFault {
        eor.getParent().sendAction(responseAction);
    }

    /**
     * Convenience method used to generate a message exchange id from a provided instance of
     * <code>MyRole</code> and an operation name.
     * 
     * @param myRole
     * @param operation
     * @return
     */
    public static final String toID(MyRole myRole, QName operation) {
        if (myRole == null) {
            throw new IllegalArgumentException("myRole null");
        }
        if (operation == null) {
            throw new IllegalArgumentException("operation null");
        }
        QName partner = ((ServiceReference)myRole).getParent().getReferencedComponentName();
        String role = ((ServiceReference)myRole).getRoleName();
        String action = null; // action is abstract, so any endpoint will do
        Iterator<EndpointReference> itr = ((ServiceReference)myRole).getEndpointReferences();
        while (itr.hasNext()) {
            action = itr.next().getOperationReference(operation).getRequestAction().getAction();
            break;
        }
        if (action == null) {
            throw new IllegalStateException("Service "
                    + ((ServiceReference)myRole).getReferencedComponentName()
                    + " defines no endpoints.");
        }
        return partner + "%" + role + "%" + action;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.application.spring.MessageExchange#getAction()
     */
    public String getAction() {
        return eor.getRequestAction().getAction();
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.application.spring.MessageExchange#getID()
     */
    public String getID() {
        if (id == null) {
            id = getPartner() + "%" + getRole() + "%" + getAction();
        }
        return id;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.application.spring.MessageExchange#getPartner()
     */
    public QName getPartner() {
        return eor.getParent().getParent().getParent().getReferencedComponentName();
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.application.spring.MessageExchange#getRole()
     */
    public String getRole() {
        return eor.getParent().getParent().getRoleName();
    }

    private ResponseActionReference getResponseActionReference(String action) {
        EndpointActionReference ear = eor.getParent().getEndpointActionReference(action);
        if (ear == null) {
            throw new IllegalArgumentException("Action '"
                    + action
                    + "' is undefined on interface "
                    + intrface.getName()
                    + ".");
        }
        if (ear instanceof ResponseActionReference == false) {
            throw new IllegalArgumentException("Action '"
                    + action
                    + "' defined on interface "
                    + intrface.getName()
                    + " maps to action with direction 'in'. Expected direction 'out'");
        }
        return (ResponseActionReference)ear;
    }

    private static final byte NULL = 0;
    private static final byte STRING = 1;

    /*
     * (non-Javadoc)
     * @see java.io.Externalizable#writeExternal(java.io.ObjectOutput)
     */
    public void writeExternal(ObjectOutput out) throws IOException {
        Externalizer.writeString(out, relatesTo);
        Externalizer.writeQName(out, eor.getRootComponent().getName());
        Externalizer.writeQName(out, eor.getParent().getParent().getReferencedComponentName());
        Externalizer.writeString(out, eor.getParent().getEndpointName());
        Externalizer.writeQName(out, eor.getEndpointOperationName());
    }

    /*
     * (non-Javadoc)
     * @see java.io.Externalizable#readExternal(java.io.ObjectInput)
     */
    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException {

        relatesTo = Externalizer.readString(in);

        QName engineName = Externalizer.readQName(in);
        QName serviceName = Externalizer.readQName(in);
        String endpointName = Externalizer.readString(in);
        QName operationName = Externalizer.readQName(in);

        // note that we need to account for the fact that
        // engine definition may be stale or undefined,
        // i.e. changed/deleted since we were
        // externalized

        Engine engine = SystemContext.getContext().getSystem().getEngine(engineName);
        if (engine == null) {
            throw new IOException("Error unmarshalling MyOperation. Engine " + engineName + " is undefined.");
        }

        ServiceReference sr = ((EngineRT)engine.getRuntimeProvider()).getServiceReference(serviceName);
        if (sr == null) {
            throw new IOException("Error unmarshalling MyOperation. ServiceReference "
                    + serviceName
                    + " is undefined on runtime Engine "
                    + engineName
                    + ".");
        }

        EndpointReference er = sr.getEndpointReference(endpointName);
        if (sr == null) {
            throw new IOException("Error unmarshalling MyOperation. EndpointReference '"
                    + endpointName
                    + " is undefined on ServiceReference "
                    + serviceName
                    + " defined on runtime Engine "
                    + engineName
                    + ".");
        }

        eor = er.getOperationReference(operationName);
        if (eor == null) {
            throw new IOException("Error unmarshalling MyOperation. EndpointOperationReference "
                    + operationName
                    + " is undefined on EndpointReference '"
                    + endpointName
                    + " is defined on ServiceReference "
                    + serviceName
                    + " defined on runtime Engine "
                    + engineName
                    + ".");
        }

        intrface = eor.getEndpointOperation().getInterfaceOperation().getParent();

    }

    private static class Externalizer {

        public static final void writeQName(ObjectOutput out, QName qName) throws IOException {
            writeString(out, qName.getNamespaceURI());
            writeString(out, qName.getLocalPart());
        }

        public static final QName readQName(ObjectInput in) throws IOException, ClassNotFoundException {
            String namespaceURI = readString(in);
            String localPart = readString(in);
            return new QName(namespaceURI, localPart);
        }

        public static final void writeString(ObjectOutput out, String string) throws IOException {
            if (string == null) {
                out.writeByte(NULL);
            } else {
                out.writeByte(STRING);
                out.writeObject(string);
            }
        }

        public static final String readString(ObjectInput in) throws IOException, ClassNotFoundException {
            if (in.readByte() == NULL) {
                return null;
            }
            return (String)in.readObject();
        }

    }

}
