/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.engine.spring;

import java.util.Map;
import java.util.Set;

import javax.xml.namespace.QName;

import org.bluestemsoftware.specification.eoa.application.spring.MyApplication;
import org.bluestemsoftware.specification.eoa.application.spring.MyRole;
import org.bluestemsoftware.specification.eoa.application.spring.PartnerApplication;
import org.bluestemsoftware.specification.eoa.application.spring.PartnerOperation;
import org.bluestemsoftware.specification.eoa.component.RootComponent.ComponentName;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EngineRT;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EngineReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.ServiceReference;
import org.springframework.context.ApplicationContext;

/**
 * An implementation that uses the 'Spring Application Framework' to 'wire' together a runtime
 * <code>Application</code> instance, i.e. as modeled by this class. Provides access to root
 * spring <code>ApplicationContext</code> which MAY be shared by child contexts, e.g. created
 * by <code>WebModule</code> instances which MAY be defined within same deployment as
 * application.
 */
public final class SpringEngine extends EngineRT implements MyApplication {

    public static final String TYPE = "http://bluestemsoftware.org/open/eoa/engine/spring/1.0";
    public static final String PROVIDER_SCHEMA_LOCATION = "classpath:///schema/http.bluestemsoftware.org.open.eoa.engine.spring.1.0.xsd";
    
    private Set<String> requiredFeatures;
    
    public interface Provider extends EngineRT.Provider {

        public PartnerApplication spi_getPartner(QName engineName);

        public PartnerOperation spi_createOperation(ServiceReference serviceReference, QName operationName);

        public PartnerOperation spi_createOperation(ServiceReference serviceReference, String endpointName, QName operationName);

        public ApplicationContext spi_getApplicationContext();

    }

    public SpringEngine(SpringEngineFactory factory, Provider provider, ComponentName name,
            Map<QName, EngineReference> engineReferences, Set<String> requiredFeatures) {
        super(factory, provider, name, engineReferences);
        if (requiredFeatures == null) {
            throw new IllegalArgumentException("required features null");
        }
        this.requiredFeatures = requiredFeatures;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.engine.rt.EngineRT#getRequiredApplicationFeatures()
     */
    @Override
    public Set<String> getRequiredApplicationFeatures() {
        return requiredFeatures;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.application.spring10.api.MyApplication#getPartner(javax.xml.namespace.QName)
     */
    public PartnerApplication getPartner(QName engineName) {
        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();
        try {
            thread.setContextClassLoader(factory.getFactoryContext().getClassLoader());
            return ((Provider)provider).spi_getPartner(engineName);
        } finally {
            thread.setContextClassLoader(cl);
        }
    }

    /*
     * invoked by SpringServiceReference
     */
    PartnerOperation createOperation(ServiceReference serviceReference, QName operationName) {
        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();
        try {
            thread.setContextClassLoader(factory.getFactoryContext().getClassLoader());
            return ((Provider)provider).spi_createOperation(serviceReference, operationName);
        } finally {
            thread.setContextClassLoader(cl);
        }
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.engine.rt.EngineRT#createMyEngineReference(java.util.Map)
     */
    @Override
    protected EngineReference createMyEngineReference(Map<QName, ServiceReference> serviceReferences) {
        return new SpringEngineReference(componentName, serviceReferences, true);
    }

    /*
     * invoked by SpringServiceReference
     */
    PartnerOperation createOperation(ServiceReference serviceReference, String endpointName, QName operationName) {
        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();
        try {
            thread.setContextClassLoader(factory.getFactoryContext().getClassLoader());
            return ((Provider)provider).spi_createOperation(serviceReference, endpointName, operationName);
        } finally {
            thread.setContextClassLoader(cl);
        }
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.application.spring.MyApplication#getBean(java.lang.String)
     */
    public Object getBean(String name) {
        return getApplicationContext().getBean(name);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.application.spring.MyApplication#getMyRole(java.lang.String)
     */
    public MyRole getMyRole(String roleName) {
        return (MyRole)getEngineReference(getName()).getServiceReference(roleName);
    }

    /**
     * Returns Spring <code>ApplicationContext</code> which represents 'parent' context of
     * all child contexts which MAY exist within scope of application deployment (the
     * application implemented by this engine), e.g. those associated with
     * <code>WebModule</code> instances, etc ... which MAY be defined within associated
     * deployment.
     * @return
     */
    public ApplicationContext getApplicationContext() {
        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();
        try {
            thread.setContextClassLoader(factory.getFactoryContext().getClassLoader());
            return ((Provider)provider).spi_getApplicationContext();
        } finally {
            thread.setContextClassLoader(cl);
        }
    }

}
