/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.engine.spring;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.xml.XMLConstants;
import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;

import org.bluestemsoftware.open.eoa.engine.spring.SpringEngineConfiguration.ModuleInfo;
import org.bluestemsoftware.open.eoa.engine.spring.SpringEngineConfiguration.WebModuleInfo;
import org.bluestemsoftware.specification.eoa.DeploymentException;
import org.bluestemsoftware.specification.eoa.Resource;
import org.bluestemsoftware.specification.eoa.component.ComponentContext;
import org.bluestemsoftware.specification.eoa.component.ComponentDeployment;
import org.bluestemsoftware.specification.eoa.component.RootComponent.ComponentName;
import org.bluestemsoftware.specification.eoa.component.RootComponent.ComponentType;
import org.bluestemsoftware.specification.eoa.component.engine.Engine;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointActionReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointOperationReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EngineRT;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EngineReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.ServiceReference;
import org.bluestemsoftware.specification.eoa.component.policy.rt.ElementPolicy;
import org.bluestemsoftware.specification.eoa.ext.engine.EngineFactory;
import org.bluestemsoftware.specification.eoa.ext.policy.PolicyFactory;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

/**
 * A type specific <code>Engine</code> factory that creates instances of
 * <code>SpringEngine</code>.
 */
public final class SpringEngineFactory extends EngineFactory {

    public static final String SPRING_TNS = "http://www.springframework.org/schema/beans";
    public static final String SPRING_SCHEMA_LOC = "http://www.springframework.org/schema/beans/spring-beans-2.5.xsd";

    public interface Provider extends EngineFactory.Provider {
        public SpringEngine.Provider spi_createEngine(SpringEngineConfiguration configuration);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.ExtensionFactory#getExtensionType()
     */
    public String getExtensionType() {
        return SpringEngine.TYPE;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.rt.ProviderReader#readProviders(org.bluestemsoftware.specification.eoa.component.ComponentContext,
     *      javax.xml.transform.dom.DOMSource)
     */
    public Set<EngineRT> readProviders(ComponentContext componentContext, DOMSource source) throws DeploymentException {

        if (componentContext == null) {
            throw new IllegalArgumentException("componentContext null");
        }
        if (source == null) {
            throw new IllegalArgumentException("source null");
        }

        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();
        try {

            thread.setContextClassLoader(factoryContext.getClassLoader());

            Set<EngineRT> providers = new HashSet<EngineRT>();

            validateDefinition(source);

            Element definition = (Element)source.getNode();
            ComponentName cname = ComponentName.valueOf(definition, definition.getAttribute("name"));
            String tns = definition.getNamespaceURI();

            // parse engine configuration

            QName childName = new QName(tns, "configuration");
            Element configuration = getChildElement(definition, childName);

            Set<String> requiredFeatures = new HashSet<String>();
            childName = new QName(tns, "features");
            Element featuresElement = getChildElement(configuration, childName);
            if (featuresElement != null) {
                childName = new QName(tns, "feature");
                List<Element> featureElements = getChildElements(featuresElement, childName);
                for (Element featureElement : featureElements) {
                    requiredFeatures.add(getText(featureElement));
                }
            }

            Set<ModuleInfo> moduleInfos = new HashSet<ModuleInfo>();
            childName = new QName(tns, "modules");
            Element modulesElement = getChildElement(configuration, childName);
            if (modulesElement != null) {
                childName = new QName(tns, "web");
                List<Element> moduleElements = getChildElements(modulesElement, childName);
                for (Element moduleElement : moduleElements) {
                    String ref = moduleElement.getAttribute("ref");
                    childName = new QName(tns, "rootContext");
                    Element rootContextElement = getChildElement(moduleElement, childName);
                    String rootContext = getText(rootContextElement);
                    moduleInfos.add(new WebModuleInfo(ref, rootContext));
                }
            }

            DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
            dbf.setNamespaceAware(true);
            Document beanDefs;
            try {
                beanDefs = dbf.newDocumentBuilder().newDocument();
            } catch (ParserConfigurationException pe) {
                throw new DeploymentException("Error parsing provider definition. " + pe.getMessage());
            }
            Element child = getChildElement(configuration, new QName(SPRING_TNS, "beans"));
            beanDefs.appendChild(beanDefs.importNode(child, true));
            child = beanDefs.getDocumentElement();

            // spring should load its configuration grammar onto validator by
            // default, but it doesn't ... so to prevent user from having to
            // define schema location on every beans element we add attrib here

            String xsi = "http://www.w3.org/2001/XMLSchema-instance";
            String xsiPrefix = child.lookupPrefix(xsi);
            if (xsiPrefix == null) {
                xsiPrefix = "xsi";
                child.setAttributeNS("http://www.w3.org/2000/xmlns/", "xmlns:xsi", xsiPrefix);
            }

            // note that spring's entity resolver resolves spring schema loc
            // from classpath, i.e. even though location uses http scheme,
            // which means internet connectivity is not required

            String schemaLocation = child.getAttributeNS(xsi, "schemaLocation");
            if (schemaLocation == null) {
                child.setAttributeNS(xsi, xsiPrefix + ":schemaLocation", SPRING_TNS + " " + SPRING_SCHEMA_LOC);
            } else {
                if (!schemaLocation.contains(SPRING_TNS)) {
                    schemaLocation = schemaLocation + " " + SPRING_TNS + " " + SPRING_SCHEMA_LOC;
                    child.setAttributeNS(xsi, xsiPrefix + ":schemaLocation", schemaLocation);
                }
            }

            SpringEngineConfiguration spe = new SpringEngineConfiguration(moduleInfos, beanDefs);

            // parse engine references
            Map<QName, EngineReference> engineReferences = new HashMap<QName, EngineReference>();
            childName = new QName(tns, "partners");
            Element partnersElement = getChildElement(definition, childName);

            // if no partners are defined, engine will create an implied reference to
            // self when deployed

            if (partnersElement != null) {

                childName = new QName(tns, "engineReference");
                List<Element> engineRefElements = getChildElements(partnersElement, childName);

                for (Element engineRefElement : engineRefElements) {

                    String prefixedName = engineRefElement.getAttribute("engineName");
                    ComponentName engineName = ComponentName.valueOf(engineRefElement, prefixedName);

                    // if name matches name of parent engine element, then we
                    // are referencing self

                    boolean isMyEngine = engineName.equals(cname);

                    Map<QName, ServiceReference> serviceReferences = new HashMap<QName, ServiceReference>();

                    childName = new QName(tns, "serviceReference");
                    List<Element> serviceRefElements = getChildElements(engineRefElement, childName);

                    for (Element serviceRefElement : serviceRefElements) {

                        prefixedName = serviceRefElement.getAttribute("serviceName");
                        ComponentName serviceName = ComponentName.valueOf(serviceRefElement, prefixedName);

                        // process private service policy attached to service ref element,
                        // if any

                        ElementPolicy serviceEP = getElementPolicy(componentContext, serviceRefElement);

                        // process endpoint references which if empty, will be defaulted by
                        // abstract parent component

                        LinkedHashMap<String, EndpointReference> eprs = new LinkedHashMap<String, EndpointReference>();

                        childName = new QName(tns, "endpointReference");
                        List<Element> epElements = getChildElements(serviceRefElement, childName);

                        for (Element epElement : epElements) {

                            String endpointName = epElement.getAttribute("endpointName");

                            // parse private endpoint policy attached to endpoint element,
                            // if any

                            ElementPolicy endpointEP = getElementPolicy(componentContext, epElement);

                            // parse operation references, each of which references an
                            // operation on
                            // referenced endpoint and allows user to attach private policy

                            Map<QName, EndpointOperationReference> eors = new HashMap<QName, EndpointOperationReference>();

                            childName = new QName(tns, "operationReference");
                            List<Element> operationElements = getChildElements(epElement, childName);

                            for (Element operationElement : operationElements) {

                                QName operationName = ComponentName.valueOf(operationElement, operationElement
                                        .getAttribute("operationName"));

                                // parse private operation policy attached to operation
                                // element, if any

                                ElementPolicy operationEP = getElementPolicy(componentContext, operationElement);

                                // parse action references, each of which references an action
                                // on referenced operation and allows user to attach private
                                // policy

                                Map<String, EndpointActionReference> ears = new HashMap<String, EndpointActionReference>();

                                childName = new QName(tns, "actionReference");
                                List<Element> actionElements = getChildElements(operationElement, childName);

                                for (Element actionElement : actionElements) {

                                    String action = actionElement.getAttribute("action");

                                    // parse private action policy attached to action element,
                                    // which should exist, i.e. this is the only reason to
                                    // define an action reference element

                                    ElementPolicy messageEP = getElementPolicy(componentContext, actionElement);

                                    EndpointActionReference ear = new EndpointActionReference(action, messageEP);
                                    if (ears.containsKey(action)) {
                                        throw new DeploymentException(
                                                "Failed to parse EndpointActionReference component '"
                                                        + action
                                                        + "' not unique within context of EndpointOperationReference '"
                                                        + operationName
                                                        + "'.");
                                    }
                                    ears.put(action, ear);

                                }

                                EndpointOperationReference eor = new EndpointOperationReference(operationName,
                                        operationEP, ears);
                                if (eors.containsKey(operationName)) {
                                    throw new DeploymentException(
                                            "Failed to parse EndpointOperationReference component. Name '"
                                                    + operationName
                                                    + "' not unique within context of EndpointReference '"
                                                    + endpointName
                                                    + "'.");
                                }
                                eors.put(operationName, eor);

                            }

                            EndpointReference epr = new EndpointReference(endpointName, endpointEP, eors);
                            if (eprs.containsKey(endpointName)) {
                                throw new DeploymentException(
                                        "Failed to parse EndpointReference component. Name '"
                                                + endpointName
                                                + "' not unique within context of ServiceReference "
                                                + serviceName
                                                + "'.");
                            }
                            eprs.put(endpointName, epr);

                        }

                        // if service ref is defined within engine ref which references
                        // parent component, then we are referencing services which are
                        // hosted by parent component, i.e. isMyService is true

                        boolean isMyService = isMyEngine;

                        ServiceReference serviceRef = new SpringServiceReference(serviceName, isMyService,
                                serviceEP, eprs);

                        serviceReferences.put(serviceName, serviceRef);

                    }

                    EngineReference engineReference = new SpringEngineReference(engineName, serviceReferences,
                            isMyEngine);
                    engineReferences.put(engineName, engineReference);

                }
            }
            ComponentDeployment deployment = componentContext.getDeployment();
            Engine engine = (Engine)deployment.getComponent(ComponentType.ENGINE, cname);
            if (engine == null) {
                throw new DeploymentException("Error loading runtime provider definition "
                        + cname
                        + ". Definition for corresponding RootComponent is undefined."
                        + " Definition for instances of SingletonRT must be defined within"
                        + " same deployment as corresponding RootComponent.");
            }
            File engineEtcDir = createEngineEtcDir(engine.getFragmentIdentifier());
            File engineVarDir = createEngineVarDir(engine.getFragmentIdentifier());
            SpringEngine.Provider p = ((Provider)provider).spi_createEngine(spe);
            providers.add(new SpringEngine(this, p, cname, engineReferences, requiredFeatures, engineEtcDir, engineVarDir));

            return providers;

        } finally {
            thread.setContextClassLoader(cl);
        }

    }

    private ElementPolicy getElementPolicy(ComponentContext componentContext, Element parent) throws DeploymentException {

        Set<PolicyFactory> factories = null;
        factories = SystemContext.getContext().getSystem().getPolicyFactories();
        ElementPolicy result = null;
        for (PolicyFactory factory : factories) {
            ElementPolicy temp = null;
            try {
                DOMSource source = new DOMSource(parent, parent.getOwnerDocument().getDocumentURI());
                temp = factory.readElementPolicy(componentContext, source);
            } catch (DeploymentException de) {
                throw new DeploymentException("Error reading element policy. " + de.getMessage());
            }
            if (result == null) {
                result = temp;
            } else {
                throw new DeploymentException("Attached policy expressions MUST be identically typed.");
            }
        }

        return result;

    }

    private Element getChildElement(Element parent, QName childName) {
        NodeList nodeList = parent.getChildNodes();
        for (int i = 0; i < nodeList.getLength(); i++) {
            if (nodeList.item(i).getNodeType() != Node.ELEMENT_NODE) {
                continue;
            }
            Element element = (Element)nodeList.item(i);
            if (childName.getNamespaceURI().equals(XMLConstants.NULL_NS_URI)) {
                if (element.getNamespaceURI() != null) {
                    continue;
                }
            } else {
                if (element.getNamespaceURI() == null) {
                    continue;
                }
                if (!childName.getNamespaceURI().equals("*")
                        && !element.getNamespaceURI().equals(childName.getNamespaceURI())) {
                    continue;
                }
            }
            if (!childName.getLocalPart().equals("*") && !element.getLocalName().equals(childName.getLocalPart())) {
                continue;
            }
            return element;
        }
        return null;
    }

    private List<Element> getChildElements(Element parent, QName childName) {
        List<Element> answer = new ArrayList<Element>();
        NodeList nodeList = parent.getChildNodes();
        for (int i = 0; i < nodeList.getLength(); i++) {
            if (nodeList.item(i).getNodeType() != Node.ELEMENT_NODE) {
                continue;
            }
            Element element = (Element)nodeList.item(i);
            if (childName.getNamespaceURI().equals(XMLConstants.NULL_NS_URI)) {
                if (element.getNamespaceURI() != null) {
                    continue;
                }
            } else {
                if (element.getNamespaceURI() == null) {
                    continue;
                }
                if (!childName.getNamespaceURI().equals("*")
                        && !element.getNamespaceURI().equals(childName.getNamespaceURI())) {
                    continue;
                }
            }
            if (!childName.getLocalPart().equals("*") && !element.getLocalName().equals(childName.getLocalPart())) {
                continue;
            }
            answer.add(element);
        }
        return answer;
    }

    private String getText(Element element) {
        if (element == null || !element.hasChildNodes()) {
            return null;
        }
        try {
            Text textNode = (Text)element.getFirstChild();
            return textNode.getData();
        } catch (ClassCastException ce) {
        }
        return null;
    }

    private void validateDefinition(DOMSource source) throws DeploymentException {

        Resource resource = null;
        try {
            resource = factoryContext.getResource(SpringEngine.PROVIDER_SCHEMA_LOCATION);
        } catch (DeploymentException de) {
            throw new DeploymentException("Error validating definition. " + de);
        }

        SchemaFactory schemaFactory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);

        javax.xml.validation.Schema schema = null;
        try {
            schema = schemaFactory.newSchema(new StreamSource(resource.getInputStream()));
        } catch (Exception ex) {
            throw new DeploymentException("Error parsing schema. " + ex);
        }
        DOMValidator domValidator = new DOMValidator(schema);
        String validationError;
        try {
            validationError = domValidator.validate(source);
        } catch (Exception ex) {
            throw new DeploymentException("Error validating definition. " + ex);
        }
        if (validationError != null) {
            throw new DeploymentException("Engine definition invalid. " + validationError);
        }

    }

    static final class DOMValidator {

        private Schema schema;

        public DOMValidator(Schema schema) {
            this.schema = schema;
        }

        public String validate(DOMSource source) throws SAXException, IOException {
            Validator validator = schema.newValidator();
            validator.setErrorHandler(new ErrorHandlerImpl());
            validator.validate(source);
            return ((ErrorHandlerImpl)validator.getErrorHandler()).getErrors();
        }

        static class ErrorHandlerImpl implements ErrorHandler {

            private ArrayList<String> errorMessages = new ArrayList<String>();

            public void error(SAXParseException spe) throws SAXException {
                errorMessages.add(spe.getMessage());
            }

            public void warning(SAXParseException spe) throws SAXException {
                errorMessages.add(spe.getMessage());
            }

            public void fatalError(SAXParseException spe) throws SAXException {
                errorMessages.add(spe.getMessage());
            }

            public String getErrors() {
                if (errorMessages.size() > 0) {
                    Iterator<String> messageIterator = errorMessages.iterator();
                    StringBuilder errorMessage = new StringBuilder();
                    while (messageIterator.hasNext()) {
                        errorMessage.append(messageIterator.next() + "  ");
                    }
                    return errorMessage.toString();
                }
                return null;
            }

        }

    }

}
