/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.feature.jms.server.activemq.mgmt;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintStream;
import java.io.StringWriter;
import java.io.Writer;
import java.net.URI;
import java.util.Iterator;
import java.util.Properties;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.activemq.broker.BrokerService;
import org.apache.activemq.broker.TransportConnector;
import org.apache.xbean.spring.context.impl.XBeanXmlBeanFactory;
import org.bluestemsoftware.open.eoa.ext.feature.jms.server.activemq.util.Constants;
import org.bluestemsoftware.open.eoa.ext.feature.jms.server.activemq.util.DOMSerializer;
import org.bluestemsoftware.open.eoa.ext.feature.jms.server.activemq.xbean.EmbeddedBroker;
import org.bluestemsoftware.specification.eoa.Resource;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactoryContext;
import org.bluestemsoftware.specification.eoa.ext.feature.jms.server.JMSServerFeature;
import org.bluestemsoftware.specification.eoa.ext.feature.jms.server.JMSServerFeatureException;
import org.bluestemsoftware.specification.eoa.ext.management.jmx.JMXServerContext;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;
import org.bluestemsoftware.specification.eoa.system.server.Feature;
import org.bluestemsoftware.specification.eoa.system.server.Server;
import org.springframework.beans.factory.config.PropertyPlaceholderConfigurer;
import org.springframework.core.io.InputStreamResource;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.InputSource;

/**
 * Encapsulates feature management which includes configuration management and implementation
 * of administrative operations which MAY be invoked at runtime.
 * <p>
 * Note that xbean objects represent a read-only model of our configuration, i.e all changes
 * are manifested within configuration element which is persisted on {@link Server}. Changes
 * to configuration will not take effect until feature is re-loaded, at which point, xbean
 * object model will reflect the new configuration.
 */

public class JMSServer {

    private static final Log log = SystemContext.getContext().getSystem().getLog(JMSServer.class);

    private EmbeddedBroker embeddedBroker;
    private int port = -1;

    public EmbeddedBroker getEmbeddedBroker() {
        return embeddedBroker;
    }

    public int getPort() {
        return port;
    }
    
    // TODO: expose configuration setters via a JMSServer portlet. when a managed
    // attribute is changed, we will update the configuration element and issue
    // an update on server

    public void shutdown() {
        if (embeddedBroker != null) {
            try {
                embeddedBroker.stop();
            } catch (Exception ex) {
                log.error("error stopping embedded broker " + ex);
            }
        }
    }

    public void start(ExtensionFactoryContext factoryContext, JMSServerFeature consumer, JMXServerContext jmxServerContext) throws JMSServerFeatureException {

        Element configuration = null;
        File configurationFile = new File(consumer.getExtensionEtcDir(), "config.xml");
        if (configurationFile.exists()) {
            try {
                DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
                factory.setNamespaceAware(true);
                DocumentBuilder documentBuilder = factory.newDocumentBuilder();
                Document configurationDoc = documentBuilder.parse(configurationFile);
                configuration = configurationDoc.getDocumentElement();
            } catch (Exception ex) {
                throw new JMSServerFeatureException("Error loading configuration. " + ex);
            }
        }

        // if configuration is null, retrieve default configuration contained within
        // deployment and persist it

        if (configuration == null) {
            Writer writer = null;
            try {
                Resource resource = factoryContext.getResource(Constants.DEFAULT_CONFIGURATION_LOC);
                InputSource inputSource = new InputSource(resource.getInputStream());
                inputSource.setSystemId(Constants.DEFAULT_CONFIGURATION_LOC);
                DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
                factory.setNamespaceAware(true);
                DocumentBuilder documentBuilder = factory.newDocumentBuilder();
                documentBuilder.setEntityResolver(factoryContext);
                configuration = documentBuilder.parse(inputSource).getDocumentElement();
                writer = new FileWriter(configurationFile);
                DOMSerializer.serializeNode(configuration, writer, "UTF-8", true);
            } catch (Exception ex) {
                throw new JMSServerFeatureException("Error retrieving default configuration. " + ex);
            } finally {
                if (writer != null) {
                    try {
                        writer.close();
                    } catch (IOException ignore) {
                    }
                }
            }
        }

        // set a system property which should be used within feature configuration to
        // set data directory

        Properties sysProps = System.getProperties();
        sysProps.put("activemq.system.home", ((Feature)consumer).getExtensionVarDir().getAbsolutePath());
        log.debug("activemq.system.home " + sysProps.getProperty("activemq.system.home"));

        // note that i attempted to create an xbean parser feature with xbean and spring
        // classes scoped to feature deployment, but xbean classes defined within activemq
        // packages directly reference xbean and spring classes ... bummer

        ClassLoader currentClassLoader = Thread.currentThread().getContextClassLoader();

        try {

            // unfortunately spring has no dom input resource, so we must convert
            // to string and then into a byte inputstream

            ByteArrayInputStream bais = null;
            try {
                StringWriter stringWriter = new StringWriter();
                DOMSerializer.serializeNode(configuration, stringWriter, "UTF-8", true);
                bais = new ByteArrayInputStream(stringWriter.toString().getBytes("UTF-8"));
            } catch (Exception ex) {
                throw new JMSServerFeatureException("Error parsing bean definitions. ", ex);
            }

            XBeanXmlBeanFactory beanFactory = null;
            Thread.currentThread().setContextClassLoader(factoryContext.getClassLoader());
            try {
                beanFactory = new XBeanXmlBeanFactory(new InputStreamResource(bais));
            } catch (Exception ex) {
                throw new JMSServerFeatureException(ex.toString(), ex);
            }

            PropertyPlaceholderConfigurer pphc = new PropertyPlaceholderConfigurer();
            pphc.postProcessBeanFactory(beanFactory);

            String[] names = beanFactory.getBeanNamesForType(BrokerService.class);
            for (int i = 0; i < names.length; i++) {
                String name = names[i];
                embeddedBroker = (EmbeddedBroker)beanFactory.getBean(name);
                if (embeddedBroker != null) {
                    break;
                }
            }
            
            if (embeddedBroker == null) {
                throw new JMSServerFeatureException("No 'broker' xbean found within supplied policy");
            } else {
                try {
                    embeddedBroker.setJmsServerFeature(consumer);
                    embeddedBroker.setJmxServerContext(jmxServerContext);
                    embeddedBroker.setUseShutdownHook(false);
                    embeddedBroker.start();
                } catch (Throwable th) {
                    StringBuffer sb = new StringBuffer();
                    try {
                        ByteArrayOutputStream baos = new ByteArrayOutputStream();
                        th.printStackTrace(new PrintStream(baos, true, "UTF-8"));
                        sb.append(System.getProperty("line.separator"));
                        sb.append(new String(baos.toByteArray(), "UTF-8"));
                    } catch (Exception exception) {
                        sb.append(th.toString());
                    }
                    throw new JMSServerFeatureException(sb.toString());
                }
            }

            try {
                Iterator<?> itr = embeddedBroker.getTransportConnectors().iterator();
                while (itr.hasNext()) {
                    TransportConnector connector = (TransportConnector)itr.next();
                    URI connectorURI = connector.getConnectUri();
                    if (connectorURI.getPort() > 0) {
                        port = connectorURI.getPort();
                        break;
                    }
                }
            } catch (Exception ex) {
                throw new JMSServerFeatureException("Error discerning port. " + ex);
            }

        } finally {
            Thread.currentThread().setContextClassLoader(currentClassLoader);
        }

    }

}
