/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.feature.jms.server.activemq.util;

import java.io.PrintWriter;
import java.lang.reflect.Method;
import java.sql.Connection;
import java.sql.SQLException;

import javax.sql.DataSource;

import org.bluestemsoftware.specification.eoa.ext.connector.db.DataSourceConnector;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.container.Container;

public class ActiveMQDatasource implements DataSource {

    private DataSource datasource;
    private Method unwrapMethod;
    private Method isWrapperForMethod;

    public ActiveMQDatasource() {
        this(null);
    }

    public ActiveMQDatasource(String connectorName) {
        Container container = SystemContext.getContext().getSystem().getContainer();
        if (connectorName == null) {
            datasource = (DataSource)container.getSystemDataSource();
            if (datasource == null) {
                throw new IllegalArgumentException("Connector name undefined on dataSource xbean"
                        + " and no system datasource defined on container.");

            }
        } else {
            try {
                datasource = (DataSource)container.getConnector(connectorName);
            } catch (ClassCastException ce) {
                throw new IllegalArgumentException("Connector '"
                        + connectorName
                        + "' MUST be an instanceof "
                        + DataSourceConnector.class.getName()
                        + ".");
            }
            if (datasource == null) {
                throw new IllegalStateException("Connector '" + connectorName + "' not defined on Container.");
            }
        }
        try {
            Class<?> clazz = DataSource.class;
            Class<?>[] signature = { Class.class };
            unwrapMethod = clazz.getDeclaredMethod("unwrap", signature);
        } catch (NoSuchMethodException ignore) {
        }
        try {
            Class<?> clazz = DataSource.class;
            Class<?>[] signature = { Class.class };
            isWrapperForMethod = clazz.getDeclaredMethod("isWrapperFor", signature);
        } catch (NoSuchMethodException ignore) {
        }

    }

    /** *************************** DataSource Impl ************************** * */

    /*
     * (non-Javadoc)
     * @see javax.sql.DataSource#getConnection()
     */
    public Connection getConnection() throws SQLException {
        return datasource.getConnection();
    }

    /*
     * (non-Javadoc)
     * @see javax.sql.DataSource#getConnection(java.lang.String, java.lang.String)
     */
    public Connection getConnection(String username, String password) throws SQLException {
        return datasource.getConnection(username, password);
    }

    public PrintWriter getLogWriter() throws SQLException {
        return datasource.getLogWriter();
    }

    public int getLoginTimeout() throws SQLException {
        return datasource.getLoginTimeout();
    }

    public void setLogWriter(PrintWriter out) throws SQLException {
        datasource.setLogWriter(out);
    }

    public void setLoginTimeout(int seconds) throws SQLException {
        datasource.setLoginTimeout(seconds);
    }

    /*
     * (non-Javadoc)
     * 
     * JDK 1.6 DataSource method. Implemented using reflection API which allows
     * specification-eoa-ext to be compiled using JDK 1.5 in cases where underlying
     * datasource(s) unavailable for JDK 1.6.
     * 
     * @see java.sql.Wrapper#isWrapperFor(java.lang.Class)
     */
    public boolean isWrapperFor(Class<?> iface) throws SQLException {
        if (isWrapperForMethod != null) {
            Boolean answer;
            try {
                answer = (Boolean)isWrapperForMethod.invoke(datasource, new Object[] { iface });
            } catch (Exception ex) {
                throw new SQLException(ex.toString());
            }
            return answer.booleanValue();
        } else {
            return false;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * JDK 1.6 DataSource method. Implemented using reflection API which allows
     * specification-eoa-ext to be compiled using JDK 1.5 in cases where underlying
     * datasource(s) unavailable for JDK 1.6.
     * 
     * @see java.sql.Wrapper#unwrap(java.lang.Class)
     */
    public <T> T unwrap(Class<T> iface) throws SQLException {
        if (unwrapMethod != null) {
            try {
                return iface.cast(unwrapMethod.invoke(datasource, new Object[] { iface }));
            } catch (Exception ex) {
                throw new SQLException(ex.toString());
            }
        }
        return null;
    }

}
