/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.message.axiom;

import java.io.File;
import java.io.FileWriter;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.xml.XMLConstants;
import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.stream.XMLStreamReader;

import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.OMFactory;
import org.apache.axiom.om.OMNode;
import org.apache.axiom.om.OMXMLParserWrapper;
import org.apache.axiom.om.impl.builder.StAXOMBuilder;
import org.apache.axiom.om.impl.dom.DOMMessageFormatter;
import org.apache.axiom.om.impl.dom.DocumentImpl;
import org.bluestemsoftware.open.eoa.aspect.axiom.DocumentElement;
import org.bluestemsoftware.open.eoa.aspect.axiom.PartAccessor;
import org.bluestemsoftware.open.eoa.aspect.axiom.util.STAXUtils;
import org.bluestemsoftware.open.eoa.commons.util.DOMHandler;
import org.bluestemsoftware.open.eoa.ext.message.axiom.util.MessageDefinition;
import org.bluestemsoftware.specification.eoa.DeploymentException;
import org.bluestemsoftware.specification.eoa.component.ComponentContext;
import org.bluestemsoftware.specification.eoa.component.ComponentReference;
import org.bluestemsoftware.specification.eoa.component.FragmentIdentifier;
import org.bluestemsoftware.specification.eoa.component.ComponentDeployment.CurrentReference;
import org.bluestemsoftware.specification.eoa.component.FragmentIdentifier.Scheme;
import org.bluestemsoftware.specification.eoa.component.RootComponent.ComponentName;
import org.bluestemsoftware.specification.eoa.component.RootComponent.ComponentType;
import org.bluestemsoftware.specification.eoa.component.intrface.rt.SystemFault;
import org.bluestemsoftware.specification.eoa.component.message.InterfaceMessage;
import org.bluestemsoftware.specification.eoa.component.message.MessageFragment;
import org.bluestemsoftware.specification.eoa.component.message.MessagePart;
import org.bluestemsoftware.specification.eoa.component.message.InterfaceMessage.AnyMessage;
import org.bluestemsoftware.specification.eoa.component.message.rt.Content;
import org.bluestemsoftware.specification.eoa.component.message.rt.Message;
import org.bluestemsoftware.specification.eoa.component.message.rt.MessageValidationException;
import org.bluestemsoftware.specification.eoa.component.types.Schema;
import org.bluestemsoftware.specification.eoa.component.types.SchemaFragment;
import org.bluestemsoftware.specification.eoa.component.types.SchemaComponent.Type;
import org.bluestemsoftware.specification.eoa.component.types.rt.ExpressionRT;
import org.bluestemsoftware.specification.eoa.component.types.rt.SchemaDocument;
import org.bluestemsoftware.specification.eoa.component.types.rt.SchemaException;
import org.bluestemsoftware.specification.eoa.component.types.rt.SchemaValidationException;
import org.bluestemsoftware.specification.eoa.ext.Extension;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactory;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactoryContext;
import org.bluestemsoftware.specification.eoa.ext.expression.EvaluationException;
import org.bluestemsoftware.specification.eoa.ext.message.default10.DefaultMessage;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.bootstrap.DOMImplementationRegistry;
import org.w3c.dom.ls.DOMImplementationLS;
import org.w3c.dom.ls.LSOutput;
import org.w3c.dom.ls.LSSerializer;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

/**
 * Document element has namespace uri DefaultMessage.TYPE and local name 'message' and name
 * attribute whose qualified name matches name of abstract message.
 * <p>
 * If part is typed as ELEMENT, part content, if any, is contained within a 'part accessor'
 * element with no namespace and local name equal to part name. Part accessor element for parts
 * typed as ELEMENT are created when message is instantiated and should never be detached from
 * message.
 * <p>
 * If part is typed as COMPLEX or as SIMPLE, and part has content the part accessor element IS
 * THE CONTENT, i.e. part accessor element is returned to user. If part has no content, no part
 * accessor exists. The part accessor element has no namespace and local name is equal to part
 * name
 */
public abstract class AbstractMessage extends DocumentImpl implements Message.Provider {

    protected FragmentIdentifier fragmentIdentifier;
    protected ComponentContext componentContext;
    protected ComponentName componentName;
    protected MessageDefinition messageDefinition;
    protected Boolean isDeployed = Boolean.FALSE;

    public AbstractMessage(DocumentImpl ownerDocument, OMFactory factory) {
        super(ownerDocument, factory);
    }

    public AbstractMessage(OMFactory factory) {
        super(factory);
    }

    public AbstractMessage(OMXMLParserWrapper parserWrapper, OMFactory factory) {
        super(parserWrapper, factory);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.rt.RootComponentRT#getName()
     */
    public ComponentName getName() {
        return componentName;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.rt.RootComponentRT#getComponentType()
     */
    public ComponentType getComponentType() {
        return ComponentType.MESSAGE;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.rt.RootComponentRT#getComponentContext()
     */
    public ComponentContext getComponentContext() {

        // if we were created from a deployed provider definition, context
        // was set when we were deployed. if we were generated as a default
        // provider definition, we return the abstract message context

        if (componentContext == null) {
            componentContext = getMetadata().getComponentContext();
        }
        return componentContext;

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.rt.RootComponentRT#getRuntimeProvidable()
     */
    public InterfaceMessage getRuntimeProvidable() {
        return getMetadata();
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.Extension#getExtensionProvider()
     */
    public org.bluestemsoftware.specification.eoa.ext.Extension.Provider getExtensionProvider() {
        return this;
    }

    /**
     * Introduced for testing. As of version 1.2.5, axiom's serialization of message was
     * unpredictable, i.e. part accessors with no namespace caused a problem, and could not
     * verify that definition was correctly generated.
     * @return
     */
    public File getMessageDefinition() {
        return messageDefinition;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.rt.ComponentRT#getFragmentIdentifier()
     */
    public FragmentIdentifier getFragmentIdentifier() {
        if (fragmentIdentifier == null) {
            fragmentIdentifier = new FragmentIdentifier(componentName.getNamespaceURI(), Scheme.EOA, "messageRT",
                    componentName.getLocalPart());
        }
        return fragmentIdentifier;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.open.eoa.ext.message.axiom.AbstractMessage#spi_setConsumer(org.bluestemsoftware.specification.eoa.ext.Extension)
     */
    public void spi_setConsumer(Extension consumer) {
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.message.rt.Message#validate()
     */
    public void validate() throws MessageValidationException {

        if (!isDeployed()) {
            throw new IllegalStateException("message not deployed");
        }

        InterfaceMessage interfaceMessage = getMetadata();

        if (interfaceMessage == null) {
            throw new IllegalStateException("Abstract message " + componentName + " is undefined");
        }

        if (componentName.equals(AnyMessage.NAME)) {
            return;
        }

        Element documentElement = getDocumentElement();

        if (!documentElement.getLocalName().equals("message")) {
            throw new MessageValidationException("Expected root element with localName 'message'.");
        }

        for (MessagePart part : interfaceMessage.getParts()) {

            String partName = part.getName();
            OMElement partAccessor = getPartAccessor(partName);
            if (partAccessor == null) {
                throw new MessageValidationException("Concrete instance of abstract message part "
                        + part.getName()
                        + " is undefined");
            }

            Content content = null;

            if (part.getSchemaComponentType() == Type.ELEMENT_DECLARATION) {

                content = (Content)partAccessor.getFirstElement();

                if (content == null) {
                    continue;
                }

                SchemaDocument schema = part.getReferencedComponent().getRuntimeProvider();

                try {
                    schema.validate(content);
                } catch (SchemaException se) {
                    throw new MessageValidationException("Error validating message. " + se.getMessage());
                } catch (SchemaValidationException sve) {
                    throw new MessageValidationException(sve.getMessage());
                }

            } else {

                content = (Content)partAccessor;

                // just validate that part accessor is appropriately typed, i.e. we
                // cannot validate its content, i.e. part accessor elements would
                // have to be declared within referenced schema

                // TODO: we could define a declarePartAccessor method on schema
                // which could build-up the declaration for us. i developed this
                // and discarded a while back. check old code for an example, ie.
                // the castor-schema-factory. we would probably have to include the
                // schema because the element form default would have to change, i
                // think

                if (!content.getNamespaceURI().equals("")) {
                    throw new MessageValidationException("Part accessor for message part "
                            + part.getName()
                            + " is not valid. Expected type "
                            + new QName("", part.getName()));
                }
                if (!content.getLocalName().equals(part.getName())) {
                    throw new MessageValidationException("Part accessor for message part "
                            + part.getName()
                            + " is not valid. Expected type "
                            + new QName("", part.getName()));
                }

            }

        }

    }

    private void validateDefinition() throws MessageValidationException {

        if (messageDefinition == null) {
            throw new IllegalStateException("message has no definition");
        }

        InterfaceMessage interfaceMessage = getMetadata();

        if (interfaceMessage == null) {
            throw new MessageValidationException("Abstract message " + componentName + " is undefined");
        }

        Element documentElement = getDocumentElement();

        if (!documentElement.getLocalName().equals("message")) {
            throw new MessageValidationException("Expected root element with localName 'message'.");
        }

        // iterate over parts defined within runtime definition and
        // verify part accessor matches name of abstract part

        NodeList nodeList = documentElement.getChildNodes();
        for (int i = 0; i < nodeList.getLength(); i++) {
            if (nodeList.item(i).getNodeType() != Node.ELEMENT_NODE) {
                continue;
            }
            Element child = (Element)nodeList.item(i);
            String ns = child.getNamespaceURI() == null ? XMLConstants.NULL_NS_URI : child.getNamespaceURI();
            if (!ns.equals(XMLConstants.NULL_NS_URI)) {
                throw new MessageValidationException("Element "
                        + new QName(child.getNamespaceURI(), child.getLocalName())
                        + " is is invalid. Part accessor element must have no namespace.");
            }
            if (interfaceMessage.getPart(child.getLocalName()) == null) {
                throw new MessageValidationException("Part accessor defined for part '"
                        + child.getLocalName()
                        + "' fails to match a part defined on abstract message");
            }
        }

        ClassLoader current = Thread.currentThread().getContextClassLoader();

        try {

            ExtensionFactoryContext efc = getExtensionFactory().getFactoryContext();
            Thread.currentThread().setContextClassLoader(efc.getClassLoader());

            // create a new document which will be used to overwrite def file,
            // which may contain unnecessary formatting, i.e. definition was
            // taken directly from descriptor file, which is typically formatted.
            // this will also add default values etc, i.e. psvi to instance

            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            factory.setNamespaceAware(true);
            DocumentBuilder documentBuilder = null;
            try {
                documentBuilder = factory.newDocumentBuilder();
            } catch (ParserConfigurationException pe) {
                throw new MessageValidationException("Error validating message. " + pe);
            }
            Document newSourceDocument = documentBuilder.newDocument();
            String defaultMessageNS = DefaultMessage.TYPE;
            Element newDocumentElement = newSourceDocument.createElementNS(defaultMessageNS, "message");
            String ns = componentName.getNamespaceURI();
            newDocumentElement.setAttributeNS("http://www.w3.org/2000/xmlns/", "xmlns:ns", ns);
            newDocumentElement.setAttribute("name", "ns:" + componentName.getLocalPart());
            newSourceDocument.appendChild(newDocumentElement);

            for (MessagePart part : interfaceMessage.getParts()) {

                String partName = part.getName();
                OMElement partAccessor = getPartAccessor(partName);
                if (partAccessor == null) {
                    throw new MessageValidationException("Concrete instance of abstract message part "
                            + part.getName()
                            + " is undefined");
                }

                Content content = null;

                if (part.getSchemaComponentType() == Type.ELEMENT_DECLARATION) {

                    content = (Content)partAccessor.getFirstElement();

                    // if content is null, then this runtime instance of abstract
                    // part is empty, i.e. just add part accessor element to our
                    // new document and continue

                    if (content == null) {
                        Node temp = newSourceDocument.importNode((Node)partAccessor, true);
                        ((Element)temp).setAttributeNS("http://www.w3.org/2000/xmlns/", "xmlns", "");
                        newDocumentElement.appendChild(temp);
                        continue;
                    }

                    DOMHandler domHandler;
                    try {
                        domHandler = new DOMHandler(true, true);
                    } catch (ParserConfigurationException pe) {
                        throw new MessageValidationException(pe);
                    }

                    // force component deployment to deploy schema provider, i.e.
                    // it must be deployed before we can use it

                    ComponentName schemaName = part.getReferencedComponent().getName();
                    ComponentReference<Schema> cr = new SchemaReferenceInternal(schemaName);
                    try {
                        componentContext.findReferencedComponent(cr);
                    } catch (DeploymentException de) {
                        throw new MessageValidationException(de);
                    }

                    SchemaDocument schema = part.getReferencedComponent().getRuntimeProvider();

                    final List<String> errorMessages = new ArrayList<String>();

                    try {
                        schema.validate(content, domHandler, domHandler, new ErrorHandler() {

                            public void error(SAXParseException spe) throws SAXException {
                                errorMessages.add(spe.getMessage());
                            }

                            public void warning(SAXParseException spe) throws SAXException {
                                errorMessages.add(spe.getMessage());
                            }

                            public void fatalError(SAXParseException spe) throws SAXException {
                                errorMessages.add(spe.getMessage());
                            }

                        });
                    } catch (SchemaException se) {
                        throw new MessageValidationException("Error validating message. " + se.getMessage());
                    }

                    if (errorMessages.size() > 0) {
                        Iterator<String> messageIterator = errorMessages.iterator();
                        StringBuilder errorMessage = new StringBuilder();
                        while (messageIterator.hasNext()) {
                            errorMessage.append(messageIterator.next() + "  ");
                        }
                        throw new MessageValidationException(errorMessage.toString());
                    } else {
                        Node result = domHandler.getDocument();
                        Element newPartAccessorElement = newSourceDocument.createElementNS(null, partName);
                        newDocumentElement.appendChild(newPartAccessorElement);
                        if (result != null) {
                            Element temp = ((Document)result).getDocumentElement();
                            newSourceDocument.adoptNode(temp);
                            newPartAccessorElement.appendChild(temp);
                        }
                    }

                } else {

                    content = (Content)partAccessor;

                    // just validate that part accessor is appropriately typed, i.e. we
                    // cannot validate its content, i.e. part accessor elements would
                    // have to be declared within referenced schema

                    // TODO: we could define a declarePartAccessor method on schema
                    // which could build-up the declaration for us. i developed this
                    // and discarded a while back. check old code for an example, ie.
                    // the castor-schema-factory. we would probably have to include the
                    // schema because the element form default would have to change, i
                    // think

                    if (!content.getNamespaceURI().equals("")) {
                        throw new MessageValidationException("Part accessor for message part "
                                + part.getName()
                                + " is not valid. Expected type "
                                + new QName("", part.getName()));
                    }
                    if (!content.getLocalName().equals(part.getName())) {
                        throw new MessageValidationException("Part accessor for message part "
                                + part.getName()
                                + " is not valid. Expected type "
                                + new QName("", part.getName()));
                    }

                    Node temp = newSourceDocument.importNode((Node)partAccessor, true);
                    ((Element)temp).setAttributeNS("http://www.w3.org/2000/xmlns/", "xmlns", "");
                    newDocumentElement.appendChild(temp);

                }

            }

            // serialize the new definition document to file and re-prime the
            // stax builder with the new stream

            FileWriter writer = null;
            try {
                writer = new FileWriter(messageDefinition, false);
                DOMImplementationRegistry registry = DOMImplementationRegistry.newInstance();
                DOMImplementationLS impl = (DOMImplementationLS)registry.getDOMImplementation("LS");
                LSSerializer serializer = impl.createLSSerializer();
                serializer.getDomConfig().setParameter("xml-declaration", false);
                LSOutput destination = impl.createLSOutput();
                destination.setCharacterStream(writer);
                destination.setEncoding("UTF-8");
                newSourceDocument.normalize();
                serializer.write(newSourceDocument, destination);
            } finally {
                if (writer == null) {
                    writer.flush();
                    writer.close();
                }
            }

            if (this.builder != null) {
                ((StAXOMBuilder)this.builder).close();
            }
            ((DocumentElement)this.documentElement).detachFromDocument();
            InputStream in = messageDefinition.newInputStream();
            XMLStreamReader reader = STAXUtils.createXMLStreamReader(in, "UTF-8");
            this.builder = new StAXOMBuilder(getOMFactory(), reader);
            this.done = false;

        } catch (Exception ex) {
            throw new MessageValidationException("Error validating message. " + ex);
        } finally {
            if (current != null) {
                Thread.currentThread().setContextClassLoader(current);
            }
        }

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.ComponentReference#getReferencedComponentType()
     */
    public ComponentType getReferencedComponentType() {
        return ComponentType.MESSAGE;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.rt.RootComponentRT#deploy(org.bluestemsoftware.specification.eoa.component.ComponentContext)
     */
    public synchronized void deploy(ComponentContext componentContext) throws DeploymentException {
        if (isDeployed == null) {
            throw new DeploymentException("Runtime component "
                    + getFragmentIdentifier()
                    + " is circularly referenced by "
                    + CurrentReference.getCurrentReference());
        }
        if (isDeployed) {
            return;
        }
        isDeployed = null;
        this.componentContext = componentContext;
        try {
            validateDefinition();
        } catch (MessageValidationException se) {
            throw new DeploymentException("Runtime message instance "
                    + componentName
                    + " is not valid. "
                    + se.getMessage());
        }
        isDeployed = Boolean.TRUE;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.rt.RootComponentRT#isDeployed()
     */
    public synchronized boolean isDeployed() {
        if (isDeployed == null) {
            return false;
        }
        return isDeployed;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.message.rt.Message$Provider#getContent()
     */
    public synchronized Content getContent() {
        if (!getMetadata().isWSDL20Style()) {
            throw new UnsupportedOperationException("Message is not a WSDL 2.0 style message");
        }
        OMElement content = getOMDocumentElement().getFirstElement().getFirstElement();
        // as of axiom version 1.2.6, if user appends a text node(s) to the content
        // as opposed to an element, and we then attempt to import the content
        // from one document to another within binding, the switched reader returns
        // this element rather than the text node(s) as expected, so rather than:
        // <myContent>my text</myContent>
        // you get:
        // <myContent><myContent/></myContent>
        // if we advance the parser one step here, that resolves the problem ...
        if (content != null) {
            content.buildNext();
        }
        return (Content)content;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.message.rt.Message$Provider#getContent(java.lang.String)
     */
    public synchronized Content getContent(String partName) {
        if (getMetadata().isWSDL20Style()) {
            return getContent();
        }
        MessagePart abstractPart = getMetadata().getPart(partName);
        if (abstractPart == null) {
            throw new IllegalArgumentException("Part '" + partName + "' is undefined.");
        }
        OMElement partAccessor = getPartAccessor(partName);
        if (abstractPart.getSchemaComponentType() == Type.ELEMENT_DECLARATION) {
            OMElement content = partAccessor.getFirstElement();
            // as of axiom version 1.2.6, if user appends a text node(s) to the content
            // as opposed to an element, and we then attempt to import the content
            // from one document to another within binding, the switched reader returns
            // this element rather than the text node(s) as expected, so rather than:
            // <myContent>my text</myContent>
            // you get:
            // <myContent><myContent/></myContent>
            // if we advance the parser one step here, that resolves the problem ...
            if (content != null) {
                content.buildNext();
            }
            return (Content)content;
        } else {
            return (Content)partAccessor;
        }
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.message.rt.Message$Provider#setContent(org.bluestemsoftware.specification.eoa.component.message.rt.Message.Content)
     */
    public synchronized void setContent(Content content) {

        // system fault may be needed to generate messages during a
        // shutdown, i.e. we can't rely on metadata being available

        String partName = null;
        if (componentName.equals(SystemFault.MESSAGE_NAME)) {
            partName = MessagePart.PAYLOAD;
        } else {
            if (!getMetadata().isWSDL20Style()) {
                throw new UnsupportedOperationException("Message is not a WSDL 2.0 style message");
            }
            if (getMetadata().getParts().size() == 0) {
                throw new IllegalArgumentException("Abstract message defines no parts.");
            }
            MessagePart abstractPart = getMetadata().getParts().iterator().next();
            partName = abstractPart.getName();
        }

        PartAccessor partAccessor = getPartAccessor(partName);
        if (content != null) {
            if (!content.getOwnerDocument().equals(this)) {
                throw new DOMException(DOMException.WRONG_DOCUMENT_ERR, DOMMessageFormatter.formatMessage(
                        DOMMessageFormatter.DOM_DOMAIN, "WRONG_DOCUMENT_ERR", null));
            }
            partAccessor.discardContent();
            partAccessor.addChild((OMElement)content);
        } else {
            partAccessor.discardContent();
        }

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.message.rt.Message$Provider#setContent(java.lang.String,
     *      org.bluestemsoftware.specification.eoa.component.message.rt.Message.Content)
     */
    public synchronized void setContent(String partName, Content content) {

        if (componentName.equals(SystemFault.MESSAGE_NAME)) {
            setContent(content);
            return;
        }

        MessagePart abstractPart = getMetadata().getPart(partName);

        if (abstractPart == null) {
            throw new IllegalArgumentException("Message part '" + partName + "' is undefined.");
        }

        if (getMetadata().isWSDL20Style()) {
            setContent(content);
            return;
        }

        PartAccessor partAccessor = getPartAccessor(partName);
        if (abstractPart.getSchemaComponentType() == Type.ELEMENT_DECLARATION) {
            if (content != null) {
                if (!content.getOwnerDocument().equals(this)) {
                    throw new DOMException(DOMException.WRONG_DOCUMENT_ERR, DOMMessageFormatter.formatMessage(
                            DOMMessageFormatter.DOM_DOMAIN, "WRONG_DOCUMENT_ERR", null));
                }
                partAccessor.discardContent();
                partAccessor.addChild((OMElement)content);
            } else {
                partAccessor.discardContent();
            }
        } else {
            if (content != null) {
                if (!content.getOwnerDocument().equals(this)) {
                    throw new DOMException(DOMException.WRONG_DOCUMENT_ERR, DOMMessageFormatter.formatMessage(
                            DOMMessageFormatter.DOM_DOMAIN, "WRONG_DOCUMENT_ERR", null));
                }
                QName expectedType = new QName(XMLConstants.NULL_NS_URI, partName);
                QName actualType = new QName(content.getNamespaceURI(), content.getLocalName());
                if (!expectedType.equals(actualType)) {
                    throw new IllegalArgumentException("Content incorrectly typed. Found "
                            + actualType
                            + ". Expected part accessor element "
                            + expectedType
                            + ".");
                }
                partAccessor.discardContent();
                getOMDocumentElement().addChild((OMElement)content);
            } else {
                partAccessor.discardContent();
            }
        }
    }

    public synchronized String evaluateString(QName fragmentName) throws EvaluationException {

        MessageFragment fragment = getMetadata().getFragment(fragmentName);
        if (fragment == null) {
            throw new IllegalArgumentException("Message fragment " + fragmentName + ", is undefined.");
        }

        ContextInfo contextInfo = null;
        Content partContent = null;
        try {
            partContent = getContent(fragment.getPartName());
            if (partContent == null) {
                throw new IllegalArgumentException("Message part '"
                        + fragment.getPartName()
                        + "' upon which fragment "
                        + fragmentName
                        + " is defined, has no content.");
            }
            contextInfo = adjustContext(partContent);
            SchemaFragment schemaFragment = fragment.getSchemaFragment();
            ExpressionRT expression = schemaFragment.getFragmentExpression().getRuntimeExpression();
            return expression.evaluateString(this);
        } finally {
            if (contextInfo != null) {
                restoreContext(contextInfo, partContent);
            }
        }

    }

    public synchronized Boolean evaluateBoolean(QName fragmentName) throws EvaluationException {

        MessageFragment fragment = getMetadata().getFragment(fragmentName);
        if (fragment == null) {
            throw new IllegalArgumentException("Message fragment " + fragmentName + ", is undefined.");
        }

        ContextInfo contextInfo = null;
        Content partContent = null;
        try {
            partContent = getContent(fragment.getPartName());
            if (partContent == null) {
                throw new IllegalArgumentException("Message part '"
                        + fragment.getPartName()
                        + "' upon which fragment "
                        + fragmentName
                        + " is defined, has no content.");
            }
            contextInfo = adjustContext(partContent);
            SchemaFragment schemaFragment = fragment.getSchemaFragment();
            ExpressionRT expression = schemaFragment.getFragmentExpression().getRuntimeExpression();
            return expression.evaluateBoolean(this);
        } finally {
            if (contextInfo != null) {
                restoreContext(contextInfo, partContent);
            }
        }

    }

    public synchronized Number evaluateNumber(QName fragmentName) throws EvaluationException {

        MessageFragment fragment = getMetadata().getFragment(fragmentName);
        if (fragment == null) {
            throw new IllegalArgumentException("Message fragment " + fragmentName + ", is undefined.");
        }

        ContextInfo contextInfo = null;
        Content partContent = null;
        try {
            partContent = getContent(fragment.getPartName());
            if (partContent == null) {
                throw new IllegalArgumentException("Message part '"
                        + fragment.getPartName()
                        + "' upon which fragment "
                        + fragmentName
                        + " is defined, has no content.");
            }
            contextInfo = adjustContext(partContent);
            SchemaFragment schemaFragment = fragment.getSchemaFragment();
            ExpressionRT expression = schemaFragment.getFragmentExpression().getRuntimeExpression();
            return expression.evaluateNumber(this);
        } finally {
            if (contextInfo != null) {
                restoreContext(contextInfo, partContent);
            }
        }

    }

    public synchronized Node evaluateNode(QName fragmentName) throws EvaluationException {

        MessageFragment fragment = getMetadata().getFragment(fragmentName);
        if (fragment == null) {
            throw new IllegalArgumentException("Message fragment " + fragmentName + ", is undefined.");
        }

        ContextInfo contextInfo = null;
        Content partContent = null;
        try {
            partContent = getContent(fragment.getPartName());
            if (partContent == null) {
                throw new IllegalArgumentException("Message part '"
                        + fragment.getPartName()
                        + "' upon which fragment "
                        + fragmentName
                        + " is defined, has no content.");
            }
            contextInfo = adjustContext(partContent);
            SchemaFragment schemaFragment = fragment.getSchemaFragment();
            ExpressionRT expression = schemaFragment.getFragmentExpression().getRuntimeExpression();
            return expression.evaluateNode(this);
        } finally {
            if (contextInfo != null) {
                restoreContext(contextInfo, partContent);
            }
        }

    }

    public synchronized List<Node> evaluateNodes(QName fragmentName) throws EvaluationException {

        MessageFragment fragment = getMetadata().getFragment(fragmentName);
        if (fragment == null) {
            throw new IllegalArgumentException("Message fragment " + fragmentName + ", is undefined.");
        }

        ContextInfo contextInfo = null;
        Content partContent = null;
        try {
            partContent = getContent(fragment.getPartName());
            if (partContent == null) {
                throw new IllegalArgumentException("Message part '"
                        + fragment.getPartName()
                        + "' upon which fragment "
                        + fragmentName
                        + " is defined, has no content.");
            }
            contextInfo = adjustContext(partContent);
            SchemaFragment schemaFragment = fragment.getSchemaFragment();
            ExpressionRT expression = schemaFragment.getFragmentExpression().getRuntimeExpression();
            return expression.evaluateNodes(this);
        } finally {
            if (contextInfo != null) {
                restoreContext(contextInfo, partContent);
            }
        }

    }

    /*
     * set by message factory impl
     */
    public void setComponentName(ComponentName componentName) {
        this.componentName = componentName;
    }

    /*
     * set by message factory impl
     */
    public void setMessageDefinition(MessageDefinition messageDefinition) {
        this.messageDefinition = messageDefinition;
    }

    public InterfaceMessage getMetadata() {
        return SystemContext.getContext().getSystem().getMessage(componentName);
    }

    public abstract ExtensionFactory getExtensionFactory();

    /*
     * (non-Javadoc)
     * @see java.lang.Object#toString()
     * 
     * NOTE THIS SHOULD ONLY BE USED FOR DEBUGGING. BUILDS ENTIRE MESSAGE TO MEMORY.
     */
    @Override
    public String toString() {
        return getDocumentElement().toString();
    }

    /*
     * The context Node from which fragment expression is evaluated is a synthetic document
     * information item which contains a single element information item representing root
     * element of message part ...
     */
    private ContextInfo adjustContext(Content content) {

        // retrieve part accessor and force a build. note this
        // MUST be done to keep the builder in synch with our
        // content model
        Node pa = content.getParentNode();
        ((OMNode)pa).build();

        // retrieve a reference to next sibling, which could
        // be null, so that we can re-insert content in the
        // appropriate position
        Node ns = content.getNextSibling();

        // create object which will be used to restore context
        ContextInfo contextInfo = new ContextInfo(this.documentElement, pa, ns);

        // detach document element. not that we do not use dom
        // detach method, which would build the entire message
        ((DocumentElement)this.documentElement).detachFromDocument();

        // now detach the content from part acccessor and
        // set it as the document node
        ((OMElement)content).detach();
        ((DocumentElement)content).attachToDocument();

        return contextInfo;

    }

    private void restoreContext(ContextInfo contextInfo, Content content) {
        ((DocumentElement)content).detachFromDocument();
        ((DocumentElement)contextInfo.documentElement).attachToDocument();
        contextInfo.partAccessor.insertBefore(content, contextInfo.nextSibling);
    }

    /*
     * used instead of getFirstChildWithName() which forces a build of element, i.e. because
     * ParentNode uses OMChildrenQNameIterator which seeds next element by calling
     * getNextOMSibling (as of version 1.2.5 snapshot).
     */
    private PartAccessor getPartAccessor(String partName) {

        OMElement messageElement = getOMDocumentElement();
        OMElement firstChild = messageElement.getFirstElement();

        OMElement answer = null;

        if (firstChild.getLocalName().equals(partName)) {
            answer = firstChild;
        } else {
            OMNode sibling = firstChild.getNextOMSibling();
            while (sibling != null) {
                if (sibling.getType() == OMNode.ELEMENT_NODE) {
                    OMElement temp = (OMElement)sibling;
                    if (temp.getLocalName().equals(partName)) {
                        answer = temp;
                        break;
                    }
                }
                sibling = sibling.getNextOMSibling();
            }
        }

        return (PartAccessor)answer;
    }

    private static class ContextInfo {
        Element documentElement;
        Node partAccessor;
        Node nextSibling;

        public ContextInfo(Element documentElement, Node partAccessor, Node nextSibling) {
            this.documentElement = documentElement;
            this.partAccessor = partAccessor;
            this.nextSibling = nextSibling;
        }
    }

    private static class SchemaReferenceInternal implements ComponentReference<Schema> {

        private ComponentName schemaName;

        public SchemaReferenceInternal(ComponentName schemaName) {
            this.schemaName = schemaName;
        }

        public FragmentIdentifier getFragmentIdentifier() {
            return new FragmentIdentifier("foo", Scheme.EOA, "bar", "");
        }

        public Schema getReferencedComponent() {
            throw new UnsupportedOperationException();
        }

        public Class<Schema> getReferencedComponentClass() {
            return Schema.class;
        }

        public ComponentName getReferencedComponentName() {
            return schemaName;
        }

        public ComponentType getReferencedComponentType() {
            return ComponentType.SCHEMA;
        }

    }

}
