/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.message.axiom.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.util.HashSet;
import java.util.Set;
import java.util.UUID;

import javax.xml.namespace.QName;

import org.bluestemsoftware.specification.eoa.component.intrface.rt.SystemFault;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;

/**
 * Contains a message's underlying definition. Because definition is streamed using a pull
 * parser, and because we cannot know when user is finshed reading message, the inputstream
 * cannot be closed until jvm indicates message is no longer relevant.
 */
public class MessageDefinition extends File {

    private static final long serialVersionUID = 1L;

    private static Log log = SystemContext.getContext().getSystem().getLog(MessageDefinition.class);

    private QName messageName;
    private Set<InputStream> inputStreams;

    public static MessageDefinition newFile(QName messageName) {
        String fileName = UUID.randomUUID().toString();
        File temporaryDirectory = SystemContext.getContext().getSystem().getSystemTmpDir();
        return new MessageDefinition(temporaryDirectory, fileName + ".eoa-message", messageName);
    }

    public InputStream newInputStream() {
        InputStream in = null;
        try {
            in = new FileInputStream(this);
        } catch (FileNotFoundException fe) {
            throw new IllegalStateException("Temp file containing provider definition for message "
                    + messageName
                    + " was prematurely deleted from file system.");
        }
        inputStreams.add(in);
        return in;
    }

    private MessageDefinition(File parent, String child, QName messageName) {
        super(parent, child);
        this.messageName = messageName;
        inputStreams = new HashSet<InputStream>();
        addShutdownHook();
    }

    /*
     * (non-Javadoc)
     * @see java.lang.Object#finalize()
     */
    @Override
    protected void finalize() throws Throwable {
        cleanUp();
    }

    /*
     * because finalizers are not guaranteed to run, we must register a shutdown hook and
     * clean-up, if we haven't already been deleted.
     */
    private void addShutdownHook() {
        Runtime.getRuntime().addShutdownHook(new Thread() {
            @Override
            public void run() {
                cleanUp();
            }
        });
    }

    /*
     * closes all registered inputstreams and attempts to delete self from file system. note
     * that we do not delete system fault message, because it may be required to generate fault
     * responses during shutdown
     */
    private void cleanUp() {
        for (InputStream in : inputStreams) {
            try {
                in.close();
            } catch (Exception ignore) {
            }
        }
        if (!messageName.equals(SystemFault.MESSAGE_NAME)) {
            if (exists() && !delete()) {
                log.debug("Failed to delete temp file "
                        + getName()
                        + " containing provider definition for message "
                        + messageName
                        + ".");
            }
        }
    }

}
