/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.test.feature.axis.server;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import javax.servlet.Servlet;
import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamSource;

import org.apache.axis2.AxisFault;
import org.apache.axis2.context.ConfigurationContext;
import org.apache.axis2.context.ConfigurationContextFactory;
import org.apache.axis2.engine.AxisConfigurator;
import org.bluestemsoftware.open.eoa.commons.util.DOMUtils;
import org.bluestemsoftware.open.eoa.commons.util.DOMValidator;
import org.bluestemsoftware.open.eoa.test.feature.axis.server.util.AxisConfiguratorImpl;
import org.bluestemsoftware.open.eoa.test.feature.axis.server.util.AxisServletImpl;
import org.bluestemsoftware.open.eoa.test.feature.axis.server.util.Constants;
import org.bluestemsoftware.open.eoa.test.feature.axis.server.util.DOMSerializer;
import org.bluestemsoftware.open.eoa.test.feature.axis.server.util.ServerAuthInfoImpl;
import org.bluestemsoftware.specification.eoa.DeploymentContext;
import org.bluestemsoftware.specification.eoa.DeploymentException;
import org.bluestemsoftware.specification.eoa.Resource;
import org.bluestemsoftware.specification.eoa.ext.Extension;
import org.bluestemsoftware.specification.eoa.ext.feature.FeatureException;
import org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerFeature;
import org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerFeature.ServerAuthInfo;
import org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerFeatureException.UniquePathException;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.transport.http.HTTPAuthenticationScheme;
import org.bluestemsoftware.specification.eoa.system.ManagementContext;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.server.Feature;
import org.bluestemsoftware.specification.eoa.system.server.Server;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.InputSource;

public class AxisServerFeatureProvider implements Feature.Provider {
    
    public static final String NAME = "axis-server-feature";
    public static final String IMPL = AxisServerFeatureProvider.class.getName();

    private File resourcesDir;
    private static ConfigurationContext configurationContext;
    private Extension consumer;
    private DeploymentContext deploymentContext;
    private Element configuration;

    public String spi_getFeatureImpl() {
        return IMPL;
    }

    public void spi_init(Set<ManagementContext> managementContexts) throws FeatureException {
        
        File extensionEtcDir = ((Feature)consumer).getExtensionEtcDir();
        File configurationFile = new File(extensionEtcDir, "config.xml");
        if (configurationFile.exists()) {
            try {
                DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
                factory.setNamespaceAware(true);
                DocumentBuilder documentBuilder = factory.newDocumentBuilder();
                Document configurationDoc = documentBuilder.parse(configurationFile);
                configuration = configurationDoc.getDocumentElement();
            } catch (Exception ex) {
                throw new FeatureException("Error loading configuration. " + ex);
            }
        }

        // if configuration is null, retrieve default configuration contained within
        // deployment and persist it

        if (configuration == null) {
            String loc = "classpath:///schema/http.bluestemsoftware.org.open.eoa.ext.feature.axis.server.config.1.0.xml";
            Writer writer = null;
            try {
                Resource resource = deploymentContext.getResource(loc);
                InputSource inputSource = new InputSource(resource.getInputStream());
                inputSource.setSystemId(loc);
                DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
                factory.setNamespaceAware(true);
                DocumentBuilder documentBuilder = factory.newDocumentBuilder();
                documentBuilder.setEntityResolver(deploymentContext);
                configuration = documentBuilder.parse(inputSource).getDocumentElement();
                writer = new FileWriter(configurationFile);
                DOMSerializer.serializeNode(configuration, writer, "UTF-8", true);
            } catch (Exception ex) {
                throw new FeatureException("Error retrieving default configuration. " + ex);
            } finally {
                if (writer != null) {
                    try {
                        writer.close();
                    } catch (IOException ignore) {
                    }
                }
            }
        }

        validateConfiguration(new DOMSource(configuration));

        QName childName = new QName(Constants.FEATURE_SCHEMA_NS, "auth");
        Element authElement = DOMUtils.getChildElement(configuration, childName);
        String authType = authElement.getAttribute("type");
        ServerAuthInfo authInfo = null;
        if (authType.equals("BASIC")) {
            authInfo = new ServerAuthInfoImpl("axis", HTTPAuthenticationScheme.BASIC);
        } else if (authType.equals("DIGEST")) {
            authInfo = new ServerAuthInfoImpl("axis", HTTPAuthenticationScheme.DIGEST);
        } 
        
        resourcesDir = new File(java.lang.System.getProperty("testClassesDirectory"));
        try {
            URL conf = new File(resourcesDir, "axis/conf/axis2.xml").toURI().toURL();
            URL repo = new File(resourcesDir, "axis/").toURI().toURL();
            AxisConfigurator ac = new AxisConfiguratorImpl(conf, repo);
            configurationContext = ConfigurationContextFactory.createConfigurationContext(ac);
            //new ListenerManager().init(configurationContext);
        } catch (Exception ex) {
            throw new FeatureException(ex);
        }
        
        //configurationContext.getListenerManager().start();
        try {
            Server eoaServer = SystemContext.getContext().getSystem().getServer();
            HTTPServerFeature httpServer = eoaServer.getFeature(HTTPServerFeature.class);
            Map<String, String> params = new HashMap<String,String>();
            Map<String, Object> attributes = new HashMap<String, Object>();
            attributes.put("CONFIGURATION_CONTEXT", configurationContext);
            ClassLoader featureLoader = deploymentContext.getClassLoader();
            Servlet servlet = new AxisServletImpl(featureLoader);
            httpServer.addServlet(servlet, "/axis2", authInfo, params, attributes);
        } catch (UniquePathException ue) {
            throw new AxisServerFeatureException(ue.getMessage());
        } catch (Exception ex) {
            throw new AxisServerFeatureException("Failed to configure servlet. " + ex);
        }
    }

    public void spi_destroy() {
        try {
            if (configurationContext != null) {
                configurationContext.cleanupContexts();
                configurationContext.terminate();
            }
        } catch (AxisFault af) {
            System.err.println(af.getMessage());
        }
    }

    public void spi_setConsumer(Extension consumer) {
        this.consumer = consumer;
        deploymentContext = consumer.getExtensionFactory().getFactoryContext();
    }

    public static ConfigurationContext getConfigurationContext() {
        return configurationContext;
    }
    
    private void validateConfiguration(DOMSource domSource) throws FeatureException {

        Resource resource = null;
        try {
            resource = deploymentContext.getResource(Constants.FEATURE_SCHEMA_LOC);
        } catch (DeploymentException de) {
            throw new FeatureException("Error validating configuration. " + de);
        }

        // note that we instantiate schema factory impl directly. a bug in jdk 1.5
        // prevents discovery using jaxp newInstance method. and ... the default
        // jdk 1.5 impl has a bug re: parsing qnames, i.e. a bogus UndeclaredPrefix
        // error

        javax.xml.validation.SchemaFactory schemaFactory = null;
        schemaFactory = new org.apache.xerces.jaxp.validation.XMLSchemaFactory();
        javax.xml.validation.Schema schema = null;
        try {
            schema = schemaFactory.newSchema(new StreamSource(resource.getInputStream()));
        } catch (Exception ex) {
            throw new FeatureException("Error parsing configuration schema. " + ex);
        }

        DOMValidator domValidator = new DOMValidator(schema);
        String validationError;
        try {
            validationError = domValidator.validate(domSource);
        } catch (Exception ex) {
            throw new FeatureException("Error validating configuration. " + ex);
        }
        if (validationError != null) {
            throw new FeatureException("Configuration invalid. " + validationError);
        }

    }

}
