/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.feature.http.client.commons.mgmt;

import java.io.ByteArrayInputStream;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.Map;

import javax.management.ObjectName;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.xbean.spring.context.impl.XBeanXmlBeanFactory;
import org.bluestemsoftware.open.eoa.ext.feature.http.client.commons.mgmt.jmx.HTTPClientMXBean;
import org.bluestemsoftware.open.eoa.ext.feature.http.client.commons.util.Constants;
import org.bluestemsoftware.open.eoa.ext.feature.http.client.commons.util.DOMSerializer;
import org.bluestemsoftware.open.eoa.ext.feature.http.client.commons.xbean.Client;
import org.bluestemsoftware.open.eoa.ext.feature.http.client.commons.xbean.HTTPClientConfig;
import org.bluestemsoftware.open.eoa.ext.feature.http.client.commons.xbean.Host;
import org.bluestemsoftware.open.eoa.ext.feature.http.client.commons.xbean.Property;
import org.bluestemsoftware.specification.eoa.DeploymentContext;
import org.bluestemsoftware.specification.eoa.Resource;
import org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientFeature;
import org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientFeatureException;
import org.bluestemsoftware.specification.eoa.ext.feature.jmx.server.JMXServerFeature;
import org.bluestemsoftware.specification.eoa.ext.feature.jmx.server.JMXServerFeatureException;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.server.Server;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.PropertyPlaceholderConfigurer;
import org.springframework.core.io.InputStreamResource;
import org.w3c.dom.Element;
import org.xml.sax.InputSource;

/**
 * Encapsulates feature management which includes configuration management and implementation
 * of administrative operations which MAY be invoked at runtime.
 * <p>
 * Note that xbean objects represent a read-only model of our configuration, i.e all changes
 * are manifested within configuration element which is persisted on {@link Server}. Changes
 * to configuration will not take effect until feature is re-loaded, at which point, xbean
 * object model will reflect the new configuration.
 */
public class HTTPClient implements HTTPClientMXBean {

    private Element configuration;
    private ObjectName objectName;
    private JMXServerFeature jmxServerFeature;

    // read-only xbean model. not jmx open type compliant
    private Client client;
    private Map<String, Host> hosts = new HashMap<String, Host>();

    // read-only jmx attribute model. open type compliant
    private Map<String, String> clientConfiguration = new HashMap<String,String>();
    private Map<String, Map<String,String>> hostConfiguration = new HashMap<String, Map<String,String>>();
    
    public HTTPClient() {
    }

    public Client getClient() {
        return client;
    }

    public Map<String, Host> getHosts() {
        return hosts;
    }

    public void setConfiguration(Element configuration) {
        this.configuration = configuration;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.open.eoa.ext.feature.http.client.commons.mgmt.jmx.HTTPClientMXBean#getClientConfiguration()
     */
    public Map<String,String> getClientConfiguration() {
        return clientConfiguration;
    }
    
    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.open.eoa.ext.feature.http.client.commons.mgmt.jmx.HTTPClientMXBean#getHostConfiguration()
     */
    public Map<String, Map<String,String>> getHostConfiguration() {
        return hostConfiguration;
    }

    // TODO: expose configuration setters via a HTTPClient portlet. when a managed
    // attribute is changed, we will update the configuration element and issue
    // an update on server

    public void configure(DeploymentContext deploymentContext, HTTPClientFeature httpClientFeature) throws HTTPClientFeatureException {

        if (configuration == null) {
            try {
                Resource resource = deploymentContext.getResource(Constants.DEFAULT_CONFIGURATION_LOC);
                InputSource inputSource = new InputSource(resource.getInputStream());
                inputSource.setSystemId(Constants.DEFAULT_CONFIGURATION_LOC);
                DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
                factory.setNamespaceAware(true);
                DocumentBuilder documentBuilder = factory.newDocumentBuilder();
                documentBuilder.setEntityResolver(deploymentContext);
                configuration = documentBuilder.parse(inputSource).getDocumentElement();
            } catch (Exception ex) {
                throw new HTTPClientFeatureException("Error retrieving default configuration. " + ex);
            }
        }

        ClassLoader currentClassLoader = Thread.currentThread().getContextClassLoader();

        try {

            // unfortunately spring has no dom input resource, so we must convert
            // to string and then into a byte inputstream

            ByteArrayInputStream bais = null;
            try {
                StringWriter stringWriter = new StringWriter();
                DOMSerializer.serializeNode(configuration, stringWriter, "UTF-8", true);
                bais = new ByteArrayInputStream(stringWriter.toString().getBytes("UTF-8"));
            } catch (Exception ex) {
                throw new HTTPClientFeatureException("Error parsing bean definitions. ", ex);
            }

            XBeanXmlBeanFactory beanFactory = null;
            Thread.currentThread().setContextClassLoader(deploymentContext.getClassLoader());
            try {
                beanFactory = new XBeanXmlBeanFactory(new InputStreamResource(bais));
            } catch (Exception ex) {
                throw new HTTPClientFeatureException(ex.toString(), ex);
            }

            PropertyPlaceholderConfigurer pphc = new PropertyPlaceholderConfigurer();
            pphc.postProcessBeanFactory(beanFactory);

            try {
                HTTPClientConfig config = null;
                String[] names = beanFactory.getBeanNamesForType(HTTPClientConfig.class);
                for (int i = 0; i < names.length; i++) {
                    String name = names[i];
                    config = (HTTPClientConfig)beanFactory.getBean(name);
                    if (config != null) {
                        break;
                    }
                }
                if (config == null) {
                    throw new HTTPClientFeatureException(
                            "No 'httpclient' xbean found within feature configuration.");
                } else {
                    client = config.getClient();
                    if (config.getHosts() != null) {
                        for (Host host : config.getHosts()) {
                            hosts.put(host.getName(), host);
                        }
                    }
                }
            } catch (BeansException be) {
                throw new HTTPClientFeatureException("Error parsing bean definitions. " + be);
            }

            // convert xbean model into a jmx open type compliant model,
            // i.e. for jmx management

            createOpenTypeCompliantConfiguration();

        } finally {
            Thread.currentThread().setContextClassLoader(currentClassLoader);
        }

        // if jmx server feature is enabled, expose management methods defined
        // on our mbean interface

        Server server = SystemContext.getContext().getSystem().getServer();
        jmxServerFeature = server.getFeature(JMXServerFeature.class);
        if (jmxServerFeature != null) {
            String name = this.getClass().getSimpleName();
            try {
                objectName = jmxServerFeature.registerMBean(httpClientFeature, this, null, name);
            } catch (JMXServerFeatureException je) {
                throw new HTTPClientFeatureException(je);
            }
        }

    }

    public void destroy() {
        client = null;
        hosts = null;
        if (objectName != null) {
            try {
                jmxServerFeature.unRegisterMBean(objectName);
            } catch (JMXServerFeatureException ignore) {
            }
        }
    }

    /**
     * Converts configuration to open type convertable objects per MXMBean spec.
     * 
     * @throws HTTPClientFeatureException
     */
    private void createOpenTypeCompliantConfiguration() throws HTTPClientFeatureException {
        if (client != null) {
            for (Property property : client.getProperties()) {
                String name = property.getName();
                String value = "" + property.getValue();
                clientConfiguration.put(name, value);
            }
        }
        for (Host host : hosts.values()) {
            Map<String, String> hostProperties = new HashMap<String, String>();
            hostConfiguration.put(host.getName(), hostProperties);
            for (Property property : host.getProperties()) {
                String name = property.getName();
                String value = "" + property.getValue();
                hostProperties.put(name, value);
            }
        }
    }

}
