/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.feature.http.client.commons;

import java.util.Set;

import org.apache.commons.httpclient.HostConfiguration;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.apache.commons.httpclient.UsernamePasswordCredentials;
import org.apache.commons.httpclient.auth.AuthScope;
import org.bluestemsoftware.open.eoa.ext.feature.http.client.commons.xbean.Host;
import org.bluestemsoftware.open.eoa.ext.feature.http.client.commons.xbean.Property;
import org.bluestemsoftware.specification.eoa.DeploymentContext;
import org.bluestemsoftware.specification.eoa.ext.Extension;
import org.bluestemsoftware.specification.eoa.ext.feature.FeatureException;
import org.bluestemsoftware.specification.eoa.ext.feature.auth.HostInfo;
import org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClient;
import org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientFeature;
import org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientFeatureException;
import org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientFeature.ClientAuthInfo;
import org.bluestemsoftware.specification.eoa.ext.management.jmx.JMXServerContext;
import org.bluestemsoftware.specification.eoa.system.ManagementContext;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;

public final class HTTPClientFeatureImpl implements HTTPClientFeature.Provider {

    private static final Log log = SystemContext.getContext().getSystem().getLog(HTTPClientFeature.class);
    private HTTPClientFeature consumer;
    private JMXServerContext jmxServerContext;
    private DeploymentContext deploymentContext;
    private org.bluestemsoftware.open.eoa.ext.feature.http.client.commons.mgmt.HTTPClient httpClient;
    private MultiThreadedHttpConnectionManager connectionManager;

    public static final String IMPL = HTTPClientFeatureImpl.class.getName();

    public HTTPClientFeatureImpl() {
        this.httpClient = new org.bluestemsoftware.open.eoa.ext.feature.http.client.commons.mgmt.HTTPClient();
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.Extension$Provider#spi_setConsumer(org.bluestemsoftware.specification.eoa.ext.Extension.Consumer)
     */
    public void spi_setConsumer(Extension consumer) {
        this.consumer = (HTTPClientFeature)consumer;
        this.deploymentContext = consumer.getExtensionFactory().getFactoryContext();
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature.Provider#spi_init(java.util.Set)
     */
    public void spi_init(Set<ManagementContext> managementContexts) throws FeatureException {
        log.debug("initializing commons httpclient framework");
        connectionManager = new MultiThreadedHttpConnectionManager();
        for (ManagementContext managementContext : managementContexts) {
            if (managementContext instanceof JMXServerContext) {
                jmxServerContext = (JMXServerContext)jmxServerContext;
                break;
            }
        }
        httpClient.configure(deploymentContext, consumer, jmxServerContext);
        log.debug("framework initialized");
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientFeature$Provider#spi_createHTTPClient()
     */
    public HTTPClient spi_createHTTPClient() throws HTTPClientFeatureException {

        HttpClient underlying = new HttpClient(connectionManager);

        if (httpClient.getClient() != null) {
            for (Property property : httpClient.getClient().getProperties()) {
                underlying.getParams().setParameter(property.getName(), property.getValue());
            }
        }

        return new HTTPClientImpl(consumer.getExtensionFactory(), underlying);

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientFeature$Provider#spi_createHTTPClient(java.lang.String,
     *      org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientFeature.AuthInfo)
     */
    public HTTPClient spi_createHTTPClient(HostInfo hostInfo, ClientAuthInfo authInfo) throws HTTPClientFeatureException {

        HttpClient underlying = new HttpClient(connectionManager);
        
        if (httpClient.getClient() != null) {
            for (Property p : httpClient.getClient().getProperties()) {
                underlying.getParams().setParameter(p.getName(), p.getValue());
            }
        }

        // configure client with optional host configuration which supplements/overrides
        // client configuration, if any

        Host host = httpClient.getHosts().get(hostInfo.getHost());
        if (host != null) {
            HostConfiguration hostConfiguration = new HostConfiguration();
            underlying.setHostConfiguration(hostConfiguration);
            hostConfiguration.setHost(hostInfo.getHost());
            if (host.getProperties() != null) {
                for (Property p : host.getProperties()) {
                    hostConfiguration.getParams().setParameter(p.getName(), p.getValue());
                }
            }
        }

        // if host requires http authentication, configure credentials which will
        // be submitted for indicated realm and indicated scheme challenge rcvd
        // by indicated host on indicated port (or all ports if port is '-1')

        if (authInfo != null) {
            String user = authInfo.getUserName();
            String pass = authInfo.getPassword();
            UsernamePasswordCredentials upc = new UsernamePasswordCredentials(user, pass);
            String scheme = authInfo.getScheme().toString().toLowerCase();
            AuthScope authScope = null;
            String hostName = hostInfo.getHost();
            int port = hostInfo.getPort();
            authScope = new AuthScope(hostName, port, authInfo.getRealm(), scheme);
            underlying.getState().setCredentials(authScope, upc);
        }
        
        // note that if user enabled premptive authentication and
        // scheme is basic, authscope supplied above will match.
        // if scheme is digest, credentials will not be sent
        // premptively, i.e. nonce, algorithm, etc ... are only
        // known following the 401 challenge

        return new HTTPClientImpl(consumer.getExtensionFactory(), underlying);

    }

    public void spi_destroy() {
        log.debug("destroy begin");
        if (connectionManager != null) {
            connectionManager.shutdown();
        }
        if (httpClient != null) {
            httpClient.destroy();
        }
        log.debug("destroy end");
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_getFeatureImpl()
     */
    public String spi_getFeatureImpl() {
        return IMPL;
    }

}
