/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.feature.http.client.commons;

import java.io.IOException;
import java.net.ProtocolException;
import java.util.Map;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.HttpMethodBase;
import org.apache.commons.httpclient.HttpVersion;
import org.apache.commons.httpclient.cookie.CookiePolicy;
import org.apache.commons.httpclient.methods.DeleteMethod;
import org.apache.commons.httpclient.methods.EntityEnclosingMethod;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.methods.PutMethod;
import org.bluestemsoftware.open.eoa.ext.feature.http.client.commons.util.Constants;
import org.bluestemsoftware.open.eoa.ext.feature.http.client.commons.util.RequestEntityImpl;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactory;
import org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClient;
import org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientFeatureFactory;
import org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientRequest;
import org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientResponse;
import org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientRequest.Method;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;

public class HTTPClientImpl implements HTTPClient {

    private static final Log log = SystemContext.getContext().getSystem().getLog(HTTPClient.class);

    private HTTPClientFeatureFactory factory;
    private HttpClient underlying;

    public HTTPClientImpl(ExtensionFactory factory, HttpClient underlying) {
        this.factory = (HTTPClientFeatureFactory)factory;
        this.underlying = underlying;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClient#doSend(org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientRequest)
     */
    public HTTPClientResponse doSend(HTTPClientRequest request) throws IOException {
        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();
        try {

            thread.setContextClassLoader(factory.getFactoryContext().getClassLoader());

            HttpMethodBase httpMethod = null;
            if (request.getMethod() == Method.GET) {
                httpMethod = new GetMethod(request.getAddress());
                if (log.isDebugEnabled()) {
                    log.debug("sending GET request to " + request.getAddress());
                }
            } else if (request.getMethod() == Method.DELETE) {
                httpMethod = new DeleteMethod(request.getAddress());
                if (log.isDebugEnabled()) {
                    log.debug("sending DELETE request to " + request.getAddress());
                }
            } else {

                boolean isEmptyRequest = request.getRequestEntity().isEmpty();

                // use chunked encoding unless client uses a version of http older than 1.1
                // or caller specified something other than chunked encoding via headers.
                // note that if request is empty, we do not use chunked encoding. some
                // platforms, e.g. jetty, ignores a chunked empty request

                boolean isChunked = false;

                if (isEmptyRequest) {
                    request.getRequestHeaders().put(Constants.CONTENT_LENGTH_HEADER, "0");
                } else {
                    if (underlying.getParams().getVersion().greaterEquals(HttpVersion.HTTP_1_1)) {
                        String tc = request.getRequestHeaders().get(Constants.TRANSFER_ENCODING_HEADER);
                        if (tc != null) {
                            if (tc.equals(Constants.TRANSFER_ENCODING_CHUNKED)) {
                                isChunked = true;
                            }
                        } else {
                            isChunked = true;
                        }
                    }
                }

                // verify that content type header was set by caller, which our api
                // defines as a requirement

                if (request.getRequestHeaders().get(Constants.CONTENT_TYPE_HEADER) == null) {
                    throw new IOException("Content-Type header missing");
                }
                
                if (request.getMethod() == Method.POST) {
                    httpMethod = new PostMethod(request.getAddress());
                    if (log.isDebugEnabled()) {
                        log.debug("sending POST request to " + request.getAddress());
                    }
                } else {
                    httpMethod = new PutMethod(request.getAddress());
                    if (log.isDebugEnabled()) {
                        log.debug("sending PUT request to " + request.getAddress());
                    }
                }
                ((EntityEnclosingMethod)httpMethod).setContentChunked(isChunked);
                ((EntityEnclosingMethod)httpMethod).setRequestEntity(new RequestEntityImpl(request.getRequestEntity()));
           
            }

            for (Map.Entry<String, String> entry : request.getRequestHeaders().entrySet()) {
                httpMethod.addRequestHeader(entry.getKey(), entry.getValue());
            }

            // note that we set 'automatic' cookie management to none. if the
            // underlying protocol on binding has cookies enabled, they must
            // set them them manually, i.e. by adding request header(s), e.g.
            // headers. put("Cookie", "special-cookie=value");

            httpMethod.getParams().setCookiePolicy(CookiePolicy.IGNORE_COOKIES);

            // TODO: if method is POST, http client will not handle redirects
            // automagically. we need to handle them ourselves.
            // see: http://jakarta.apache.org/commons/httpclient/redirects.html

            try {

                underlying.executeMethod(httpMethod);

            } catch (IOException ex) {

                // IOException defines a subclass used to communicate protocol
                // related errors, i.e. non-recoverable errors, which is not
                // subclassed by HttpException. so that caller can differentiate
                // between a potentially recoverable and non-recoverable error,
                // we rethrow HttpException as a protocol exception with
                // message set to string version of exception which will include
                // stack trace and causes (the fault is handled internally, i.e.
                // we're not sending this over the wire, so this should be ok).

                if (ex instanceof HttpException) {
                    throw new ProtocolException(ex.toString());
                } else {
                    throw ex;
                }

            }

            if (log.isDebugEnabled()) {
                log.debug("response status: " + httpMethod.getStatusLine());
            }

            return new HTTPClientResponseImpl(factory, httpMethod);

        } finally {
            thread.setContextClassLoader(cl);
        }
    }

}
