/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.feature.http.client.commons;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.httpclient.Header;
import org.apache.commons.httpclient.HttpMethodBase;
import org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientFeatureFactory;
import org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientResponse;
import org.xml.sax.InputSource;

public final class HTTPClientResponseImpl implements HTTPClientResponse {

    private HTTPClientFeatureFactory factory;
    private HttpMethodBase httpMethod;
    private Map<String, String> headers;
    
    public HTTPClientResponseImpl(HTTPClientFeatureFactory factory, HttpMethodBase httpMethod) {
        this.httpMethod = httpMethod;
        this.factory = factory;
    }
    
    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientResponse#getStatusCode()
     */
    public int getStatusCode() {
        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();
        try {
            thread.setContextClassLoader(factory.getFactoryContext().getClassLoader());
            return httpMethod.getStatusCode();
        } finally {
            thread.setContextClassLoader(cl);
        }   
    }
    
    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientResponse#getStatusLine()
     */
    public String getStatusReasonPhrase() {
        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();
        try {
            thread.setContextClassLoader(factory.getFactoryContext().getClassLoader());
            return httpMethod.getStatusLine().getReasonPhrase();
        } finally {
            thread.setContextClassLoader(cl);
        }   
    }
    
    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientResponse#getContent()
     */
    public InputSource getContent() throws IOException {
        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();
        try {
            thread.setContextClassLoader(factory.getFactoryContext().getClassLoader());
            InputStream inputStream = httpMethod.getResponseBodyAsStream();
            if (inputStream != null) {
                InputSource content = new InputSource(inputStream);
                content.setEncoding(httpMethod.getResponseCharSet());
                return content;
            }
        } finally {
            thread.setContextClassLoader(cl);
        } 
        return null;
    }
    
   /*
    * (non-Javadoc)
    * @see org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientResponse#getContentLength()
    */
    public long getContentLength() {
        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();
        try {
            thread.setContextClassLoader(factory.getFactoryContext().getClassLoader());
            return httpMethod.getResponseContentLength();
        } finally {
            thread.setContextClassLoader(cl);
        }   
    }
    
    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientResponse#getHeader(java.lang.String)
     */
    public String getHeader(String name) {
        if (headers == null) {
            headers = new HashMap<String, String>();
            Thread thread = Thread.currentThread();
            ClassLoader cl = thread.getContextClassLoader();
            try {
                thread.setContextClassLoader(factory.getFactoryContext().getClassLoader());
                for (Header header : httpMethod.getResponseHeaders()) {
                    headers.put(header.getName(), header.getValue());
                }
            } finally {
                thread.setContextClassLoader(cl);
            } 
        }
        return headers.get(name);
    }
    
    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientResponse#getHeaders()
     */
    public Map<String, String> getHeaders() {
        if (headers == null) {
            headers = new HashMap<String, String>();
            Thread thread = Thread.currentThread();
            ClassLoader cl = thread.getContextClassLoader();
            try {
                thread.setContextClassLoader(factory.getFactoryContext().getClassLoader());
                for (Header header : httpMethod.getResponseHeaders()) {
                    headers.put(header.getName(), header.getValue());
                }
            } finally {
                thread.setContextClassLoader(cl);
            }            
        }
        return headers;
    }

    
    /**
     * Convenience method introduced for testing, i.e. NOT for production use.
     * 
     * @return
     */
    public String getContentAsString() throws IOException {
        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();
        try {
            thread.setContextClassLoader(factory.getFactoryContext().getClassLoader());
            InputSource content = getContent();
            if (content == null) {
                return "";
            }
            InputStream is = content.getByteStream();
            ByteArrayOutputStream os = new ByteArrayOutputStream();
            byte[] buffer = new byte[4096];
            int len;
            while ((len = is.read(buffer)) > 0) {
                os.write(buffer, 0, len);
            }
            return new String(os.toByteArray(),content.getEncoding());
        } finally {
            thread.setContextClassLoader(cl);
        }        
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientResponse#releaseConnection()
     */
    public void releaseConnection() {
        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();
        try {
            thread.setContextClassLoader(factory.getFactoryContext().getClassLoader());
            httpMethod.releaseConnection();
        } finally {
            thread.setContextClassLoader(cl);
        }
    }

  
}
